#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#define VERSION_1	0
#define VERSION_2	1

int main() 
{
	int serial_port = open( "/dev/ttyUSB0", O_RDONLY );
	if ( serial_port == -1 ) {
		printf( "Error %i from open : %s\n", errno, strerror(errno) );
	}

	struct termios tty_options;

	if ( tcgetattr( serial_port, &tty_options ) != 0 ) {
		printf( "Error %i from tcgetattr : %s\n", errno, strerror(errno) );
		exit(1);
	}

	tty_options.c_cflag &= ~PARENB;			// Clear parity bit, disabling parity (most common)
	tty_options.c_cflag &= ~CSTOPB; 		// Clear stop field, only one stop bit used in communication (most common)
	tty_options.c_cflag &= ~CSIZE; 			// Clear all bits that set the data size 
	tty_options.c_cflag |= CS8; 			// 8 bits per byte (most common)
	tty_options.c_cflag &= ~CRTSCTS; 		// Disable RTS/CTS hardware flow control (most common)
	tty_options.c_cflag |= CREAD | CLOCAL; 	// Turn on READ & ignore ctrl lines (CLOCAL = 1)

	tty_options.c_lflag &= ~ICANON;
	tty_options.c_lflag &= ~ECHO;			// Disable echo
	tty_options.c_lflag &= ~ECHOE; 			// Disable erasure
	tty_options.c_lflag &= ~ECHONL;			// Disable new-line echo
	tty_options.c_lflag &= ~ISIG;			// Disable interpretation of INTR, QUIT and SUSP
	tty_options.c_iflag &= ~(IXON | IXOFF | IXANY);		// Turn off s/w flow ctrl
	tty_options.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL);	// Disable any special handling of received bytes

	tty_options.c_oflag &= ~OPOST; // Prevent special interpretation of output bytes (e.g. newline chars)
	tty_options.c_oflag &= ~ONLCR; // Prevent conversion of newline to carriage return/line feed
	// tty_options.c_oflag &= ~OXTABS; // Prevent conversion of tabs to spaces (NOT PRESENT ON LINUX)
	// tty_options.c_oflag &= ~ONOEOT; // Prevent removal of C-d chars (0x004) in output (NOT PRESENT ON LINUX)

	tty_options.c_cc[VTIME] = 1;
#if VERSION_1
	tty_options.c_cc[VMIN] = 3;
#endif
#if VERSION_2
	tty_options.c_cc[VMIN] = 2;
#endif

	cfsetispeed( &tty_options, B9600 );
	cfsetospeed( &tty_options, B9600 );

	if ( tcsetattr( serial_port, TCSANOW, &tty_options ) != 0 ) {
		printf( "Error %i from tcsetattr: %s\n", errno, strerror(errno) );
		exit(1);
	}

#if VERSION_1
	char payload[3];
#endif

#if VERSION_2
	u_int16_t payload;
	u_int8_t measurement_id;
	float physical_value;
	char buf[50];
#endif

	int num_bytes = 0;

	while (1) {
		num_bytes = read( serial_port, &payload, sizeof(payload) );

		if ( num_bytes < 0 ) {
			printf( "Error reading: %s", strerror(errno) );
			exit(1);
		}

#if VERSION_1
		printf( "%c%c%c\n", payload[0], payload[1], payload[2] );
#endif

#if VERSION_2
		printf( "0x%x\n", payload );

		measurement_id = payload & 0x02;

		if (measurement_id == 0x02) {
			payload &= 0xfffc;
			physical_value = ( ( payload * 125.0 ) / 65536 ) - 6;;
			sprintf(buf,
			        "Relative humidity : %d.%02u %%\n",
					(u_int8_t) physical_value,
					(u_int8_t) ((physical_value - (u_int8_t) physical_value ) * 100));
			printf("%s", buf);
		}
		if (measurement_id == 0x00) {
			physical_value = ( (payload * 175.72 ) /65536 ) - 46.85;
			sprintf(buf,
			        "Temperature : %d.%02u °C\n",
					(u_int8_t) physical_value,
					(u_int8_t) ((physical_value - (u_int8_t) physical_value ) * 100));
			printf("%s", buf);
		}
#endif
	}

	return EXIT_SUCCESS;
}
