/**
 * ENSICAEN
 * 6 Boulevard Maréchal Juin
 * F-14050 Caen Cedex
 *
 * This file is owned by ENSICAEN students. No portion of this
 * document may be reproduced, copied or revised without written
 * permission of the authors.
 */

/**
 * @author Dimitri Boudier <dimitri.boudier@ensicaen.fr>
 * @version 1.2.1 - 2024-08-06
 *
 * @todo    Nothing.
 * @bug     None.
 * @note    This driver can be used with only ONE Weather click at once.
 */

/**
 * @file weather_click.h
 * @brief Driver for the MIKROE "Weather click" board
 *
 * Ref. https://www.mikroe.com/weather-click
 */


#ifndef INC_WEATHER_CLICK_H_
#define INC_WEATHER_CLICK_H_



/*****************************************************************************
 * MACRO CONSTANT DEFINITIONS
 *****************************************************************************/

#define WEATHER_I2C_ADDR			//!< @TODO  //!< Weather click board I2C address
#define WEATHER_CHIP_ID				0x60		//!< Weather click ID number

/**
 * @defgroup WEATHER_registers_address
 * @{
 */
#define WEATHER_REG_ID				0xD0
#define WEATHER_REG_RESET			0xE0
#define WEATHER_REG_CTRL_HUM		0xF2
#define WEATHER_REG_STATUS			0xF3
#define WEATHER_REG_CTRL_MEAS		0xF4
#define WEATHER_REG_CONFIG			0xF5
#define WEATHER_REG_PRESS			0xF7
#define WEATHER_REG_PRESS_msb		0xF7
#define WEATHER_REG_PRESS_lsb		0xF8
#define WEATHER_REG_PRESS_xlsb		0xF9
#define WEATHER_REG_TEMP			0xFA
#define WEATHER_REG_TEMP_msb		0xFA
#define WEATHER_REG_TEMP_lsb		0xFB
#define WEATHER_REG_TEMP_xlsb		0xFC
#define WEATHER_REG_HUM				0xFD
#define WEATHER_REG_HUM_msb			0xFD
#define WEATHER_REG_HUM_lsb			0xFE
#define WEATHER_REG_CAL_T			0x88
#define WEATHER_REG_CAL_P			0x8E
#define WEATHER_REG_CAL_H			0xE1
/** @} */

/**
 * @defgroup WEATHER_RESET_register_bits
 * @{
 */
#define WEATHER_CMD_RESET			0xB6
/** @} */

/**
 * @defgroup WEATHER_CTRL_HUM_register_bits
 * @{
 */
#define WEATHER_HUM_MEAS_DISABLE	0x00
#define WEATHER_HUM_OVERSAMPLEx1	0x01
#define WEATHER_HUM_OVERSAMPLEx2	0x02
#define WEATHER_HUM_OVERSAMPLEx4	0x03
#define WEATHER_HUM_OVERSAMPLEx8	0x04
#define WEATHER_HUM_OVERSAMPLEx16	0x05
/** @} */

/**
 * @defgroup WEATHER_CTRL_MEAS_register_bits
 * @{
 */
#define WEATHER_TEMP_MEAS_DISBLE	(0x00<<5)
#define WEATHER_TEMP_OVERSAMPLEx1	(0x01<<5)
#define WEATHER_TEMP_OVERSAMPLEx2	(0x02<<5)
#define WEATHER_TEMP_OVERSAMPLEx4	(0x03<<5)
#define WEATHER_TEMP_OVERSAMPLEx8	(0x04<<5)
#define WEATHER_TEMP_OVERSAMPLEx16	(0x05<<5)
#define WEATHER_PRESS_MEAS_DISABLE	(0x00<<2)
#define WEATHER_PRESS_OVERSAMPLEx1	(0x01<<2)
#define WEATHER_PRESS_OVERSAMPLEx2	(0x02<<2)
#define WEATHER_PRESS_OVERSAMPLEx4	(0x03<<2)
#define WEATHER_PRESS_OVERSAMPLEx8	(0x04<<2)
#define WEATHER_PRESS_OVERSAMPLEx16	(0x05<<2)
#define WEATHER_MODE_SLEEP			0x00
#define WEATHER_MODE_FORCED			0x01
#define WEATHER_MODE_NORMAL			0x03
/** @} */

/**
 * @defgroup WEATHER_CONFIG_register_bits
 * @{
 */
#define WEATHER_STANDBY_05ms		(0x00<<5)
#define WEATHER_STANDBY_62ms		(0x01<<5)
#define WEATHER_STANDBY_125ms		(0x02<<5)
#define WEATHER_STANDBY_250s		(0x03<<5)
#define WEATHER_STANDBY_500ms		(0x04<<5)
#define WEATHER_STANDBY_1000ms		(0x05<<5)
#define WEATHER_STANDBY_10ms		(0x06<<5)
#define WEATHER_STANDBY_20ms		(0x07<<5)
#define WEATHER_IIR_OFF				(0x00<<2)
#define WEATHER_IIR_2_COEF			(0x01<<2)
#define WEATHER_IIR_4_COEF			(0x02<<2)
#define WEATHER_IIR_8_COEF			(0x03<<2)
#define WEATHER_IIR_16_COEF			(0x04<<2)
#define WEATHER_SPI_3WIRE_OFF		0x00
#define WEATHER_SPI_3WIRE_ON		0x01
/** @} */

/**
 * @defgroup WEATHER_skipped_measurement_values
 * @{
 */
#define WEATHER_HUM_MEAS_SKIPPED	0x8000	// A 0x8000 value means measurement has been skipped
#define WEATHER_TEMP_MEAS_SKIPPED	0x80000	// A 0x80000 value means measurement has been skipped
#define WEATHER_PRESS_MEAS_SKIPPED	0x80000	// A 0x80000 value means measurement has been skipped
/** @} */


/*****************************************************************************
 * STRUCTURE DECLARATIONS
 *****************************************************************************/

/*!
 * Handle to an initialized Light click.
 *
 * This handle contains the handle to the SPI peripheral to use,
 * and the Slave Select GPIO.
 */
typedef struct {
	I2C_HandleTypeDef *hi2c;	//!< I2C peripheral handle
}WEATHER_handle_t;


/**
 * Structure that contains all the calibration parameters
 *
 * The calibration parameters must be read from the Weather click and
 * stored into this structure.
 */
typedef struct {
	uint16_t 	dig_T1;
	int16_t 	dig_T2;
	int16_t 	dig_T3;
	uint16_t 	dig_P1;
	int16_t 	dig_P2;
	int16_t 	dig_P3;
	int16_t 	dig_P4;
	int16_t 	dig_P5;
	int16_t 	dig_P6;
	int16_t 	dig_P7;
	int16_t 	dig_P8;
	int16_t 	dig_P9;
	uint8_t 	dig_H1;
	int8_t 		dig_H2;
	uint8_t 	dig_H3;
	int16_t 	dig_H4;
	int16_t 	dig_H5;
	int8_t 		dig_H6;
	int32_t 	t_fine;
}CalibParams_t;



/*****************************************************************************
 * FUNCTION DECLARATIONS
 *****************************************************************************/

/**
 * Initializes the Weather click board
 *
 * @param[in]	hi2c	pointer to the I2C peripheral handle of the STM32
 */
void WEATHER_init(I2C_HandleTypeDef *hi2c);


/**
 * Reads n_bytes of data from a specified register
 *
 * @param[in]	reg_addr	the address of the register to be read
 * @param[out]	reg_value	the buffer in which read data will be stored
 * @param[in]	n_bytes		the number of bytes to read
 */
void WEATHER_readReg(uint8_t reg_addr, uint8_t* reg_value, uint8_t n_bytes);


/**
 * Writes data into a specified register
 *
 * @param[in]	reg_addr	the address of the register to be modified
 * @param[in]	reg_value	the value of the byte to write
 */
void WEATHER_writeReg(uint8_t reg_addr, uint8_t reg_value);


/**
 * Gives a temperature value, expressed in degC
 *
 * @param[out]	temp	pointer to the temperature value
 */
void WEATHER_getTemperature(float* temp);


/**
 * Gives a pressure value, expressed in hPa
 *
 * @param[out]	press	pointer to the pressure value
 *
 * @note		Pressure compensation needs compensated temperature first.
 */
void WEATHER_getPressure(float* press);


/**
 * Gives a relative humidity value, expressed in %RH
 *
 * @param[out]	hum		pointer to the relative humidity value
 *
 * @note		Humidity compensation needs compensated temperature first.
 */
void WEATHER_getHumidity(float* hum);


/**
 * Gives temperature (degC), pressure (hPa) and humidity (%RH) values
 *
 * @param[out]	press	pointer to the pressure value
 * @param[out]	temp	pointer to the temperature value
 * @param[out]	hum		pointer to the relative humidity value
 */
void WEATHER_getAll(float* press, float* temp, float* hum);


#endif /* INC_WEATHER_CLICK_H_ */
