/**
 * ENSICAEN
 * 6 Boulevard Maréchal Juin
 * F-14050 Caen Cedex
 *
 * This file is owned by ENSICAEN students. No portion of this
 * document may be reproduced, copied or revised without written
 * permission of the authors.
 */

/**
 * @author  Dimitri Boudier <dimitri.boudier@ensicaen.fr>
 * @version 1.0.0 - 2023-10-14
 *
 * @todo    Nothing.
 * @bug     None.
 * @note    This driver can be used with only ONE Accel 2 click at once.
 */

/**
 * @file accel_2_click.h
 * @brief Driver for the MIKROE "Accel 2 click" board
 *
 * Ref. https://www.mikroe.com/accel-2-click
 */
 

#ifndef INC_ACCEL_2_CLICK_H_
#define INC_ACCEL_2_CLICK_H_



/*****************************************************************************
 * MACRO CONSTANT DEFINITIONS
 *****************************************************************************/

#define ACCEL_2_READ_REG            0x80 //!< MSbit must be '1' for read operations

/**
 * @defgroup ACCEL_2_registers_address
 * @{
 */
#define ACCEL_2_REG_OUT_T           0x0C
#define ACCEL_2_REG_INFO1           0x0D
#define ACCEL_2_REG_INFO2           0x0E
#define ACCEL_2_REG_WHO_AM_I        0x0F
#define ACCEL_2_REG_OFF_X           0x10
#define ACCEL_2_REG_OFF_Y           0x11
#define ACCEL_2_REG_OFF_Z           0x12
#define ACCEL_2_REG_CS_X            0x13
#define ACCEL_2_REG_CS_Y            0x14
#define ACCEL_2_REG_CS_Z            0x15
#define ACCEL_2_REG_LC_L            0x16
#define ACCEL_2_REG_LC_H            0x17
#define ACCEL_2_REG_STAT            0x18
#define ACCEL_2_REG_PEAK1           0x19
#define ACCEL_2_REG_PEAK2           0x1A
#define ACCEL_2_REG_VFC_1           0x1B
#define ACCEL_2_REG_VFC_2           0x1C
#define ACCEL_2_REG_VFC_3           0x1D
#define ACCEL_2_REG_VFC_4           0x1E
#define ACCEL_2_REG_THRS3           0x1F
#define ACCEL_2_REG_CTRL_REG4       0x20
#define ACCEL_2_REG_CTRL_REG1       0x21
#define ACCEL_2_REG_CTRL_REG2       0x22
#define ACCEL_2_REG_CTRL_REG3       0x23
#define ACCEL_2_REG_CTRL_REG5       0x24
#define ACCEL_2_REG_CTRL_REG6       0x25
#define ACCEL_2_REG_STATUS          0x27
#define ACCEL_2_REG_OUT_X_L         0x28
#define ACCEL_2_REG_OUT_X_H         0x29
#define ACCEL_2_REG_OUT_Y_L         0x2A
#define ACCEL_2_REG_OUT_Y_H         0x2B
#define ACCEL_2_REG_OUT_Z_L         0x2C
#define ACCEL_2_REG_OUT_Z_H         0x2D
#define ACCEL_2_REG_FIFO_CTRL       0x2E
#define ACCEL_2_REG_FIFO_SRC        0x2F
#define ACCEL_2_REG_ST1_X           0x40 // 0x40-4F
#define ACCEL_2_REG_TIM4_1          0x50
#define ACCEL_2_REG_TIM3_1          0x51
#define ACCEL_2_REG_TIM2_1          0x52 // 0x52-53
#define ACCEL_2_REG_TIM1_1          0x54 // 0x54-55
#define ACCEL_2_REG_THRS2_1         0x56
#define ACCEL_2_REG_THRS1_1         0x57
#define ACCEL_2_REG_MASK1_B         0x59
#define ACCEL_2_REG_MASK1_A         0x5A
#define ACCEL_2_REG_SETT1           0x5B
#define ACCEL_2_REG_PR1             0x5C
#define ACCEL_2_REG_TC1             0x5D // 0x5D-5E
#define ACCEL_2_REG_OUTS1           0x5F
#define ACCEL_2_REG_ST2_X           0x60 // 0x60-6F
#define ACCEL_2_REG_TIM4_2          0x70
#define ACCEL_2_REG_TIM3_2          0x71
#define ACCEL_2_REG_TIM2_2          0x72 // 0x72-73
#define ACCEL_2_REG_TIM1_2          0x74 // 0x74-75
#define ACCEL_2_REG_THRS2_2         0x76
#define ACCEL_2_REG_THRS1_2         0x77
#define ACCEL_2_REG_DES2            0x78
#define ACCEL_2_REG_MASK2_B         0x79
#define ACCEL_2_REG_MASK2_A         0x7A
#define ACCEL_2_REG_SETT2           0x7B
#define ACCEL_2_REG_PR2             0x7C
#define ACCEL_2_REG_TC2             0x7D // 0x7D-7E
#define ACCEL_2_REG_OUTS2           0x7F
/** @} */

/**
 * @defgroup ACCEL_2_registers_default_value
 * @{
 */
#define ACCEL_2_INFO1_VALUE             0x21
#define ACCEL_2_INFO2_VALUE             0x00
#define ACCEL_2_WHO_AM_I_VALUE          0x3F
/** @} */

/**
 * @defgroup ACCEL_2_CTRL_REG4_register_bits
 * @{
 */
#define ACCEL_2_X_ENABLE                0x01
#define ACCEL_2_Y_ENABLE                0x02
#define ACCEL_2_Z_ENABLE                0x04
#define ACCEL_2_XYZ_ENABLE              0x07
#define ACCEL_2_BDU_CONTINUOUS_UPDATE   0x00
#define ACCEL_2_BDU_WAIT_FOR_READ       0x08
#define ACCEL_2_POWER_DOWN              0x00
#define ACCEL_2_OUTPUTDATARATE_3HZ125   0x10
#define ACCEL_2_OUTPUTDATARATE_6HZ25    0x20
#define ACCEL_2_OUTPUTDATARATE_12HZ5    0x30
#define ACCEL_2_OUTPUTDATARATE_25HZ     0x40
#define ACCEL_2_OUTPUTDATARATE_50HZ     0x50
#define ACCEL_2_OUTPUTDATARATE_100HZ    0x60
#define ACCEL_2_OUTPUTDATARATE_400HZ    0x70
#define ACCEL_2_OUTPUTDATARATE_800HZ    0x80
#define ACCEL_2_OUTPUTDATARATE_1600HZ   0x90
/** @} */

/**
 * @defgroup ACCEL_2_CTRL_REG5_register_bits
 * @{
 */
#define ACCEL_2_AAFILTER_BW_800HZ       0x00
#define ACCEL_2_AAFILTER_BW_200HZ       0x40
#define ACCEL_2_AAFILTER_BW_400HZ       0x80
#define ACCEL_2_AAFILTER_BW_050HZ       0xC0
#define ACCEL_2_FULL_SCALE_2G           0x00
#define ACCEL_2_FULL_SCALE_4G           0x08
#define ACCEL_2_FULL_SCALE_6G           0x10
#define ACCEL_2_FULL_SCALE_8G           0x18
#define ACCEL_2_FULL_SCALE_16G          0x20
#define ACCEL_2_FULL_SCALE_BITS         0x38
#define ACCEL_2_SELF_TEST_DISABLE       0x00
#define ACCEL_2_SELF_TEST_POS_SIGN      0x02
#define ACCEL_2_SELF_TEST_NEG_SIGN      0x04
#define ACCEL_2_SPI_4WIRE               0x00
#define ACCEL_2_SPI_3WIRE               0x01
/** @} */

/**
 * @defgroup ACCEL_2_CTRL_REG6_register_bits
 * @{
 */
#define ACCEL_2_FORCE_REBOOT            0x80
#define ACCEL_2_FIFO_ENABLE             0x40
#define ACCEL_2_FIFO_DISABLE            0x00
#define ACCEL_2_WATERMARK_ENABLE        0x20
#define ACCEL_2_WATERMARK_DISABLE       0x00
#define ACCEL_2_ADDRESS_INC_ENABLE      0x10
#define ACCEL_2_ADDRESS_INC_DISABLE     0x00
/** @} */

/**
 * @defgroup ACCEL_2_STATUS_register_bits
 * @{
 */
#define ACCEL_2_XYZ_DATA_OVERRUN        0x80
#define ACCEL_2_Z_DATA_OVERRUN          0x40
#define ACCEL_2_Y_DATA_OVERRUN          0x20
#define ACCEL_2_X_DATA_OVERRUN          0x10
#define ACCEL_2_XYZ_DATA_AVAILABLE      0x08
#define ACCEL_2_Z_DATA_AVAILABLE        0x04
#define ACCEL_2_Y_DATA_AVAILABLE        0x02
#define ACCEL_2_X_DATA_AVAILABLE        0x01
/** @} */

/**
 * @defgroup ACCEL_2_FIFO_CTRL_register_bits
 * @{
 */
#define ACCEL_2_BYPASS_MODE             0x00
#define ACCEL_2_FIFO_MODE               0x20
#define ACCEL_2_STREAM_MODE             0x40
/** @} */

/**
 * @defgroup ACCEL_2_sensitivity_values
 * @{
 */
#define ACCEL_2_SENSITIVITY_AT_2G       ((float)0.00006)
#define ACCEL_2_SENSITIVITY_AT_4G       ((float)0.00012)
#define ACCEL_2_SENSITIVITY_AT_6G       ((float)0.00018)
#define ACCEL_2_SENSITIVITY_AT_8G       ((float)0.00024)
#define ACCEL_2_SENSITIVITY_AT_16G      ((float)0.00073)
/** @} */



/*****************************************************************************
 * STRUCTURE DECLARATIONS
 *****************************************************************************/

/**
 * Handle to an initialized Accel 2 click.
 *
 * This handle contains the handle to the SPI peripheral to use,
 * and the Slave Select GPIO.
 */
typedef struct {
    SPI_HandleTypeDef *hspi;    //!< SPI peripheral handle
    GPIO_TypeDef* SS_Port;      //!< Slave Select pin port
    uint16_t SS_Pin;            //!< Slave Select pin number
}ACCEL_2_handle_t;



/*****************************************************************************
 * FUNCTION DECLARATIONS
 *****************************************************************************/

/**
 * Initializes the click board by checking the device ID.
 * If the device ID does not match or is not returned, the initilization fails.
 *
 * @param[in]   hspi        the handle to the STM32 SPI peripheral
 * @param[in]   SS_Port     the port of the Slave Select (or Chip Select) pin
 * @param[in]   SS_Pin      the pin number of the Slave Select (or Chip Select) pin
 *
 * Initializes the click board by attaching it to a SPI handle and a CS/SS GPIO.
 * If using a MIKROE Arduino UNO Click Shield on a NUCLEO-WL55JC1 board:
 * - Slot 1, SS is on D10/PA4
 * - Slot 2, SS is on D9 /PA9
 * If using a MIKROE Arduino UNO Click Shield on a NUCLEO-L073RZ or NUCLEO-L476RG board:
 * - Slot 1, SS is on D10/PB6
 * - Slot 2, SS is on D9 /PC7
 */
void ACCEL_2_init(SPI_HandleTypeDef *hspi, GPIO_TypeDef *SS_Port, uint16_t SS_Pin);


/**
 * Writes one byte of data into the specified register.
 *
 * @param[in]   reg_addr    the register address
 * @param[in]   reg_value   the register value
 */
void ACCEL_2_writeReg(uint8_t reg_addr, uint8_t reg_value);


/**
 * Reads "n_bytes" bytes of data from the specified register.
 *
 * @param[in]   reg_addr    the register address
 * @param[in]   reg_value   pointer to an array, to store the received data 
 * @param[in]   n_bytes     the number of bytes to read
 */
void ACCEL_2_readReg(uint8_t reg_addr, uint8_t* reg_value, uint8_t n_bytes);


/**
 * Get the sensitivity value from the full-scale parmeter.
 *
 * @param[out]  sensitivity	pointer to the sensitivity value
 */
void ACCEL_2_getSensitivity(float* sensitivity);


/**
 * Get the x-axis acceleration value.
 *
 * @param[out]  x_accel     pointer to the output value
 */
void ACCEL_2_getAccelX(float* x_accel);


/**
 * Get the y-axis acceleration value.
 *
 * @param[out]  y_accel     pointer to the output value
 */
void ACCEL_2_getAccelY(float* y_value);


/**
 * Get the z-axis acceleration value.
 *
 * @param[out]  z_accel     pointer to the output value
 */
void ACCEL_2_getAccelZ(float* z_value);


/**
 * Get the acceleration value from all axes.
 *
 * @param[out]  x_accel     pointer to the output value
 * @param[out]  y_accel     pointer to the output value
 * @param[out]  z_accel     pointer to the output value
 */
void ACCEL_2_getAccelXYZ(float* x_accel, float* y_accel, float* z_accel);


#endif /* INC_ACCEL_2_CLICK_H_ */
