#ifndef __EVAL_LIB_H
#define __EVAL_LIB_H

/*
* Eval file for DSP exam
* 2023, Dimitri Boudier
*/

/* texas instruments dependencies - container types */
#include <c6x.h>

/**
 * @brief Eval function in canonical C (single precision IEEE754)
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp.c
 */
void eval_sp(	const float * restrict src1, 	\
				const float * restrict src2,	\
				float * restrict dest, 			\
				int n);

/**
 * @brief Eval function in radix 4 canonical C (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_r4.c
 */
void eval_sp_r4(	const float * restrict src1, 	\
					const float * restrict src2,	\
					float * restrict dest, 			\
					int n);

/**
 * @brief Eval function in radix 4 vectorized C (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_opt_r4.c
 */
void eval_sp_opt_r4(	const float * restrict src1, 	\
						const float * restrict src2,	\
						float * restrict dest, 			\
						int n);

/**
 * @brief Eval function in C6600 assembler without optimization
 *  (single precision IEEE754)
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_asm.s
 */
void eval_sp_asm (	const float * restrict src1, 	\
					const float * restrict src2,	\
					float * restrict dest, 			\
					int n);

						
/**
 * @brief Eval function in C6600 assembler with typical VLIW
 * optimization (single precision IEEE754)
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_asm_manual.s
 */
void eval_sp_asm_vliw (	const float * restrict src1, 	\
						const float * restrict src2,	\
						float * restrict dest, 			\
						int n);

/**
 * @brief Eval function in C6600 assembler with pipeline
 * software optimization (single precision IEEE754)
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_asm_softPipeline.s
 */
void eval_sp_asm_softPipeline (	const float * restrict src1, 	\
								const float * restrict src2,	\
								float * restrict dest, 			\
								int n);

/**
 * @brief Eval function in radix 4 vectorized assembler
 *  (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param src1,  	input array pointer (n elements)
 * @param src2, 	input array pointer (n elements)
 * @param dest, 	output array pointer (n elements)
 * @param n,		arrays size
 * @include eval_sp_asm_r4.s
 */
void eval_sp_asm_r4 (	const float * restrict src1, 	\
						const float * restrict src2,	\
						float * restrict dest, 			\
						int n);

#endif /* __FIR_LIB_H */
