/*
 * uart/src/uart.c
 * uart2 driver library
 * author 
 */

/* Microchip Harmony dependencies */
#include <sys/attribs.h>
#include <peripheral/peripheral.h>

#include <kuart/include/kuart.h>

#define UART_XON   0x11
#define UART_XOFF  0x13

void __ISR(_UART2_RX_VECTOR, ipl1AUTO) _IntHandlerDrvUsartReceiveInstance0(void)
{
    char rxdata, lost;
	signed portBASE_TYPE xHigherPriorityTaskWoken;

    if(PLIB_INT_SourceFlagGet( INT_ID_0,INT_SOURCE_USART_2_RECEIVE ))
    {
		/* Flush UART2 FIFO */
		while( U2STAbits.URXDA )
			rxdata = U2RXREG;

		 xQueueSendFromISR( queue_uart2Rx_comIsrToTask, &rxdata, &xHigherPriorityTaskWoken );

		/* Clear up the interrupt flag */
		PLIB_INT_SourceFlagClear( INT_ID_0,INT_SOURCE_USART_2_RECEIVE );	
	}
	
	portEND_SWITCHING_ISR( xHigherPriorityTaskWoken );
}


void uart_init(uInt32_t bdrate)
{
    // Rx - Tx RTOS protection
    mutex_uart2Tx_Protect = xSemaphoreCreateMutex();
    queue_uart2Rx_comIsrToTask = xQueueCreate( UART_BUFF_SIZE, sizeof( char ) );
 
    /* PORT B Initialization */
    PLIB_PORTS_OpenDrainEnable(PORTS_ID_0, PORT_CHANNEL_B, 0x0);
    PLIB_PORTS_Write( PORTS_ID_0, PORT_CHANNEL_B,  0x0);
    PLIB_PORTS_DirectionOutputSet( PORTS_ID_0, PORT_CHANNEL_B,  0xffff ^ 0xFFFF);
    PLIB_PORTS_ChangeNoticePerPortTurnOn(PORTS_ID_0, PORT_CHANNEL_B);
    PLIB_PORTS_ChannelModeSelect(PORTS_ID_0, PORT_CHANNEL_B, 0xbfff ^ 0xFFFF, PORTS_PIN_MODE_DIGITAL);
    PLIB_PORTS_ChannelChangeNoticeEnable(PORTS_ID_0, PORT_CHANNEL_B, 0x0);
    PLIB_PORTS_ChannelChangeNoticePullUpEnable(PORTS_ID_0, PORT_CHANNEL_B, 0x0);
    PLIB_PORTS_ChannelChangeNoticePullDownEnable(PORTS_ID_0, PORT_CHANNEL_B, 0x0);     
    
    /* PORT G Initialization */
    PLIB_PORTS_OpenDrainEnable(PORTS_ID_0, PORT_CHANNEL_G, 0x0);
    PLIB_PORTS_Write( PORTS_ID_0, PORT_CHANNEL_G,  0x0);
    PLIB_PORTS_DirectionOutputSet( PORTS_ID_0, PORT_CHANNEL_G,  0xf3c3 ^ 0xFFFF);
    PLIB_PORTS_ChangeNoticePerPortTurnOn(PORTS_ID_0, PORT_CHANNEL_G);
    PLIB_PORTS_ChannelModeSelect(PORTS_ID_0, PORT_CHANNEL_G, 0x8380 ^ 0xFFFF, PORTS_PIN_MODE_DIGITAL);
    PLIB_PORTS_ChannelChangeNoticeEnable(PORTS_ID_0, PORT_CHANNEL_G, 0x0);
    PLIB_PORTS_ChannelChangeNoticePullUpEnable(PORTS_ID_0, PORT_CHANNEL_G, 0x0);
    PLIB_PORTS_ChannelChangeNoticePullDownEnable(PORTS_ID_0, PORT_CHANNEL_G, 0x0); 
    
    /* PPS Remapping */
    PLIB_PORTS_RemapInput(PORTS_ID_0, INPUT_FUNC_U2RX, INPUT_PIN_RPG6 );
    PLIB_PORTS_RemapOutput(PORTS_ID_0, OUTPUT_FUNC_U2TX, OUTPUT_PIN_RPB14 );
 
    /* UART2 Config */
    U2MODE = (uInt32_t) 0x00468800;
    U2STA = (uInt32_t) 0x00001400;    
    U2BRG = (uInt32_t) ((uInt32_t) PB2_FREQ_HZ / ((uInt32_t) 16 * bdrate )) - 1;       

    /* Enable and clear the interrupt source in case of interrupt mode */
    PLIB_INT_SourceFlagClear( INT_ID_0,INT_SOURCE_USART_2_RECEIVE );
    PLIB_INT_SourceEnable( INT_ID_0,INT_SOURCE_USART_2_RECEIVE );

    /* Interrupts config */
    PLIB_INT_VectorPrioritySet( INT_ID_0,INT_VECTOR_UART2_RX, INT_PRIORITY_LEVEL1);
    PLIB_INT_VectorSubPrioritySet(INT_ID_0,INT_VECTOR_UART2_RX,INT_SUBPRIORITY_LEVEL0);
    PLIB_INT_MultiVectorSelect( INT_ID_0 );
 
    /* UART2 ENABLE */
    PLIB_USART_Enable(USART_ID_2);
       
    /* Enable Global Interrupts */
    PLIB_INT_Enable( INT_ID_0 );
}


uInt8_t uart_getc(uInt8_t *payload, uInt8_t echo)
{
    uInt8_t     tmp;

    xQueueReceive( queue_uart2Rx_comIsrToTask, &tmp, portMAX_DELAY );
        
	if (echo)
		uart_putc(tmp); 

	*payload = tmp;
    return TRUE; 
}


uInt8_t uart_gets(uInt8_t *rx_buf, uInt8_t buf_limit, uInt8_t echo)
{
    static uInt8_t i = 0;
    uInt8_t tmp;

    do {
        if (!uart_getc(&tmp, echo))
            goto fail_nopayload; 

        if (i >= buf_limit) {
            rx_buf[i-1] = '\0';
            i = 0;
            goto fail_limitbuffer;
        } 
        
        // TODO : bug with destination buffer writing
        rx_buf[i++] = tmp; 
    } while (tmp != '\r');
    
    rx_buf[--i] = '\0';
    i = 0;
    return TRUE;
fail_nopayload:    
fail_limitbuffer:
    return FALSE;    
}


void uart_putc(uInt8_t payload)
{
	while(uart_tx_busy());
  	U2TXREG = payload;   
}


void uart_puts(uInt8_t *str)
{
    xSemaphoreTake( mutex_uart2Tx_Protect, portMAX_DELAY );
    {
    while (*str != '\0')
        uart_putc(*str++);
    }
    xSemaphoreGive( mutex_uart2Tx_Protect );
}