/**
 * @file kuart/include/kuart.h
 * @brief header file for UART library with FreeRTOS system calls 
 * @author hugo descoubes - copyleft
 */

/** @ingroup UART */
/* @{ */
/** @defgroup  UART */
/** @ingroup UART */
/* @{ */
#ifndef _KUART_H
#define _KUART_H

#include <common/include/common.h>
#include <FreeRTOS.h>
#include <task.h>
#include <queue.h>
#include <semphr.h>

/* API parameters */
#define UART_BAUD_RATE         9600
#define UART_BUFF_SIZE         100   // warning - 6 elements minimum
#define UART_ECHO              1
#define UART_ECHO_OFF          0

SemaphoreHandle_t mutex_uart2Tx_Protect;
QueueHandle_t queue_uart2Rx_comIsrToTask;

/* API reference */

 /**
  * @brief uart configuration
  * @li mode : asynchronous
  * @li pins : RX1 input connected to RC7, TX1 output connected to RC6
  * @li protocol : 8 bits payload, 1 stop bit, no error detection (no parity)
  * @li baud rate : 16 bits mode, hight speed mode   
  * @li interrupt : enable only for receiving with low priority ISR 
  * @li enable : enable receiver, transmitter and module
  * @warning connect USB UART click board on J28 curiosity HPC connector 
  * @param bdrate communication baud rate in decimal value
  */
void uart_init(uInt32_t bdrate);

 /**
  * @brief one byte payload sending
  * @warning sending by polling 
  * @param payload 8 bits payload to send
  */
void uart_putc(uInt8_t payload);

 /**
  * @brief send caracters string 
  * @warning sending by polling 
  * @param str pointer to a caracters string 
  */
void uart_puts(uInt8_t *str);

 /**
  * @brief one byte payload receiving
  * @warning receiving by low level priority interrupt 
  * with circular buffer and software flow control XON/XOFF 
  * @param payload pointer to receiving data buffer
  * @param echo enable (1) or disable (0) application data echo
  * @return TRUE (1) if payload has been received, else return FALSE (0)
  */
uInt8_t uart_getc(uInt8_t *payload, uInt8_t echo);

 /**
  * @brief caracters string receiving
  * @warning receiving by low level priority interrupt 
  * with circular buffer and software flow control XON/XOFF 
  * @param rx_buf pointer to receiving buffer
  * @param buf_limit application buffer size to avoid overflowing
  * @param echo enable (1) or disable (0) application data echo
  * @return TRUE (1) if caracter string was successfully built, 
  * else return FALSE (0)
  */
uInt8_t uart_gets(uInt8_t *rx_buf, uInt8_t buf_limit, uInt8_t echo);

 /**
  * @brief waiting the end of current payload sending
  * @return return TRUE (1) if module is busy, else return FALSE (0) 
  */
#define uart_tx_busy() !U2STAbits.TRMT

 /**
  * @brief wait for data receiving
  * @return return TRUE (!=0) if data has arrived, else return FALSE (0)
  */
#define uart_rx_ready() uart_rx.nb_elt

#endif /*_KUART_H*/


















