#ifndef __FIR_LIB_H
#define __FIR_LIB_H

/**
* @file firlib\h\firlib.h
* @brief FIR filtering library for TI DSP TMS320C6678 architecture
* @mainpage Documentation of FIR filtering library (Developped at ENSICAEN)
* @brief Texas Instruments ressources and dependencies to benchmark ENSICAEN FIR library :\n
*
* @li documentations :\n
* - FIR library documentation ..\\doc\\firlib.chm\n
* - TMS320C6678 TI references <a href="http://www.ti.com/product/tms320c6678">www.ti.com/product/tms320c6678</a>\n\n
*
* @li include dependencies :\n
* - FIR library header ..\\h\\firlib.h\n
*
* @li library dependencies :\n
* - FIR Library ..\\lib\\firlib.lib\n
*
* @author
* @date
*/

/* texas instruments dependencies - container types */
#include <c6x.h>

/**
 * @brief FIR filtering function in canonical C (single precision IEEE754)
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp.c
 */
void fir_sp(	const float * restrict xk, 	\
				const float * restrict a,	\
				float * restrict yk, 		\
				int na,						\
				int nyk);

/**
 * @brief FIR filtering function in radix 4 canonical C (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_r14.c
 */

void fir_sp_r14 (	const float * restrict xk, 	\
					const float * restrict a,	\
					float * restrict yk, 		\
					int na,						\
					int nyk);


/**
 * @brief FIR filtering function in radix 4 canonical C (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_r4.c
 */
void fir_sp_r4(	const float * restrict xk, 	\
				const float * restrict a,	\
				float * restrict yk, 		\
				int na,						\
				int nyk);

/**
 * @brief FIR filtering function in radix 4 vectorized C (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_opt_r4.c
 */
void fir_sp_opt_r4(	const float * restrict xk, 	\
					const float * restrict a,	\
					float * restrict yk, 		\
					int na,						\
					int nyk);

/**
 * @brief FIR filtering function in C6600 assembler without optimization
 *  (single precision IEEE754)
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_asm.s
 */
void fir_sp_asm (	const float * restrict xk, 	\
					const float * restrict a,	\
					float * restrict yk, 		\
					int na,						\
					int nyk);

						
/**
 * @brief FIR filtering function in C6600 assembler with typical VLIW 
 * optimization (single precision IEEE754)
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_asm_manual.s
 */
void fir_sp_asm_vliw (	const float * restrict xk, 	\
						const float * restrict a,	\
						float * restrict yk, 		\
						int na,						\
						int nyk);

/**
 * @brief FIR filtering function in C6600 assembler with pipeline
 * software optimization (single precision IEEE754)
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_asm_softPipeline.s
 */
void fir_sp_asm_softPipeline (	const float * restrict xk, 	\
								const float * restrict a,	\
								float * restrict yk, 		\
								int na,						\
								int nyk);

/**
 * @brief FIR filtering function in radix 4 vectorized assembler
 *  (single precision IEEE754)
 * @warning radix 4 input and coefficients arrays sizes
 * @param xk input array pointer (nyk+na-1 elements)
 * @param a coefficients array pointer (na elements)
 * @param yk output array pointer (nyk elements)
 * @param na number of coefficients
 * @param nyk output array size
 * @include fir_sp_asm_r4.s
 */
void fir_sp_asm_r14 (	const float * restrict xk, 	\
						const float * restrict a,	\
						float * restrict yk, 		\
						int na,						\
						int nyk);

#endif /* __FIR_LIB_H */
