%**********
% File :    main_mem.m
% Author :  Hugo Descoubes
% brief :   - test program for FIR filtering algorithm application
%           - input vector generation   
%           - input vector truncation, memory transfers and processing
%           - temporal and frequency analysis
% date :    june 2014
%**********
clear all
close all

%*** PARAMETERS
load coeff.mat                      % inport FIR coefficients  
a_sp = single(a);
Fs = 10000;                         % sample frequency 
F2 = 1000;                          % frequency harmonic n2 (input vector)
F1 = 100;                           % frequency harmonic n1 (input vector)
Ts = 1/Fs;                          % period frequency
                                    % TMS32C6678 memory model :
                                    % L1D/L1P 32Kb per core 
                                    % L2 512Kb per core
                                    % MSM SRAM 4Mb shared by cores
                                    % DDR3 512Mb shared by cores (board TMDSEVM6678L)
%DDR_ARRAY_LENGTH = 1048576;         % 1M/4Mb : array length in DDR
DDR_ARRAY_LENGTH = 1048576;         % 1M/4Mb : array length in DDR
L2_ARRAY_LENGTH = 32768;            % 32K/128Kb : array length in L2 SRAM
A_LENGTH = length(a_sp);            % 64/256b : length of coefficients array
YK_LENGTH = DDR_ARRAY_LENGTH - A_LENGTH + 1; % length of output temporal array in DDR
t=0 : Ts : (DDR_ARRAY_LENGTH-1)*Ts; % time vector
        
%*** INPUT VECTOR GENERATION AND ARRAYS PREALLOCATION
xk          = sin(2*pi*F1*t) + sin(2*pi*F2*t); 
xk_sp_DDR   = single(xk);
xk_sp_L2    = single(zeros(1,L2_ARRAY_LENGTH + A_LENGTH - 1));
yk_sp_L2    = single(zeros(1,L2_ARRAY_LENGTH));
yk_sp_DDR   = single(zeros(1,YK_LENGTH));

%*** MEMORY TRANSFERS
% arrays lengths :
% xk_sp_DDR     |------------------------- 4Mb --------------------------|
% xk_sp_L2      |------------- 128Kb + 256b - 4 -------------| overlap 
% a_sp          |- 256b -|
% yk_sp_L2      |------------- 128Kb -------------|
% yk_sp_DDR     |------------------- 4Mb - 256b + 4 -------------------|
%
% copy part of input array from DDR to L2 SRAM
for i=1 : L2_ARRAY_LENGTH : DDR_ARRAY_LENGTH
    
    % memcpy DDR to L2 SRAM
    % rq : DDR_ARRAY_LENGTH % L2_ARRAY_LENGTH = 0
    if i < DDR_ARRAY_LENGTH - L2_ARRAY_LENGTH
        for k=1 : L2_ARRAY_LENGTH + A_LENGTH - 1
            xk_sp_L2(k) = xk_sp_DDR(i+k-1); 
        end
    else
        for k=1 : L2_ARRAY_LENGTH
            xk_sp_L2(k) = xk_sp_DDR(i+k-1);
        end           
    end
    
    % FIR filtering - algorithm working with L1D cache
    yk_sp_L2 = fir_sp(xk_sp_L2, a_sp, A_LENGTH, L2_ARRAY_LENGTH);          
    
    % memcpy L2 SRAM to DDR - coherency of output DDR array  
    % rq : YK_LENGTH % L2_ARRAY_LENGTH = L2_ARRAY_LENGTH - A_LENGTH + 1
    if i < YK_LENGTH - L2_ARRAY_LENGTH
        for k=1 : L2_ARRAY_LENGTH
            yk_sp_DDR(i+k-1) = yk_sp_L2(k);
        end
    else
        for k=1 : L2_ARRAY_LENGTH - A_LENGTH + 1
            yk_sp_DDR(i+k-1) = yk_sp_L2(k);
        end           
    end   
    
end

%*** FIR FILTERING
yk_sp_DDR = [single(zeros(1,A_LENGTH - 1)) yk_sp_DDR];   
yk_matlab = filter(a,1,xk);                         % Matlab algorithm

%*** SIGNAL TEMPORAL ANALYSIS
figure
subplot(2,1,1);
plot(t,xk,'b');
title('digital signal filtering time scope - single precision');
grid on;
hold on;
plot(t,yk_matlab,'g');
hold on;
plot(t,yk_sp_DDR,'r');

%*** SIGNAL FREQUENCY ANALYSIS
window =  hanning(DDR_ARRAY_LENGTH);% window
xk_win = xk.*window';
yk_sp_DDR_win = yk_sp_DDR.*single(window)';

ZERO_PADDING = 0;                   % without zero padding 
xk_win_padd = [xk_win  zeros(1,ZERO_PADDING)];
yk_sp_DDR_win_padd = [yk_sp_DDR_win  single(zeros(1,ZERO_PADDING))];
XK_PADD_LENGTH = length(xk_win_padd);

Fr = Fs/XK_PADD_LENGTH;              % frequency spectrum resolution
f=0 : Fr : (XK_PADD_LENGTH-1)*Fr;    % frequency vector
Xn = abs(fft(xk_win_padd));
Yn = single(abs(fft(yk_sp_DDR_win_padd)));

subplot(2,1,2);
plot(f,Xn,'b');
title('digital signal filtering spectrum - single precision');
grid on;
hold on
plot(f,Yn,'r');
