#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>

#include "MQTTClient.h"

#define ADDRESS     "localhost:1883"
#define CLIENTID    "ClientPubLocalhost"
#define TOPIC       "Measurements"
#define QOS         1
#define TIMEOUT     6000L

#define DEBUG_CONSOLE	1

int main()
{
	int serial_port = open( "/dev/ttyUSB0", O_RDONLY );
	if ( serial_port == -1 ) {
		printf( "Error %i from open : %s\n", errno, strerror(errno) );
		exit(EXIT_FAILURE);
	}

	struct termios tty_options;

	if ( tcgetattr( serial_port, &tty_options ) != 0 ) {
		printf( "Error %i from tcgetattr : %s\n", errno, strerror(errno) );
		exit(EXIT_FAILURE);
	}

	tty_options.c_cflag &= ~PARENB;			// Clear parity bit, disabling parity (most common)
	tty_options.c_cflag &= ~CSTOPB; 		// Clear stop field, only one stop bit used in communication (most common)
	tty_options.c_cflag &= ~CSIZE; 			// Clear all bits that set the data size
	tty_options.c_cflag |= CS8; 			// 8 bits per byte (most common)
	tty_options.c_cflag &= ~CRTSCTS; 		// Disable RTS/CTS hardware flow control (most common)
	tty_options.c_cflag |= CREAD | CLOCAL; 	// Turn on READ & ignore ctrl lines (CLOCAL = 1)

	tty_options.c_lflag &= ~ICANON;
	tty_options.c_lflag &= ~ECHO;			// Disable echo
	tty_options.c_lflag &= ~ECHOE; 			// Disable erasure
	tty_options.c_lflag &= ~ECHONL;			// Disable new-line echo
	tty_options.c_lflag &= ~ISIG;			// Disable interpretation of INTR, QUIT and SUSP
	tty_options.c_iflag &= ~(IXON | IXOFF | IXANY);		// Turn off s/w flow ctrl
	tty_options.c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL);	// Disable any special handling of received bytes

	tty_options.c_oflag &= ~OPOST; // Prevent special interpretation of output bytes (e.g. newline chars)
	tty_options.c_oflag &= ~ONLCR; // Prevent conversion of newline to carriage return/line feed
	// tty_options.c_oflag &= ~OXTABS; // Prevent conversion of tabs to spaces (NOT PRESENT ON LINUX)
	// tty_options.c_oflag &= ~ONOEOT; // Prevent removal of C-d chars (0x004) in output (NOT PRESENT ON LINUX)

	tty_options.c_cc[VTIME] = 1;
	tty_options.c_cc[VMIN] = 2;

	cfsetispeed( &tty_options, B9600 );
	cfsetospeed( &tty_options, B9600 );

	if ( tcsetattr( serial_port, TCSANOW, &tty_options ) != 0 ) {
		printf( "Error %i from tcsetattr: %s\n", errno, strerror(errno) );
		exit(EXIT_FAILURE);
	}


	MQTTClient client;
   	MQTTClient_connectOptions conn_opts = MQTTClient_connectOptions_initializer;
   	MQTTClient_message pubmsg = MQTTClient_message_initializer;
	MQTTClient_deliveryToken token;
	int rc;

   	if ((rc = MQTTClient_create(&client, ADDRESS, CLIENTID,	MQTTCLIENT_PERSISTENCE_NONE, NULL)) != MQTTCLIENT_SUCCESS) {
		printf("Failed to create client, return code %d\n", rc);
		exit(EXIT_FAILURE);
	}

	conn_opts.keepAliveInterval = 20;
	conn_opts.cleansession = 1;
	if ((rc = MQTTClient_connect(client, &conn_opts)) != MQTTCLIENT_SUCCESS) {
		printf("Failed to connect, return code %d\n", rc);
		exit(EXIT_FAILURE);
	}


	u_int16_t payload;
	u_int8_t measurement_id;
	float humidity_physical_value;
	float temperature_physical_value;

	char tmp[50];
	int num_bytes = 0;

	while (1) {
		num_bytes = read( serial_port, &payload, sizeof(payload) );

		if ( num_bytes < 0 ) {
			printf( "Error reading: %s", strerror(errno) );
			exit(EXIT_FAILURE);
		}

#if DEBUG_CONSOLE
		printf( "\n~Raspberry Pi~\n");
		printf( "Sending to localhost ...\n" );
		printf( "0x%x", payload );
#endif

		measurement_id = payload & 0x02;

		if (measurement_id == 0x02) {
			payload &= 0xfffc;
			humidity_physical_value = ( ( payload * 125.0 ) / 65536 ) - 6;
			sprintf( tmp,
			         "Relative humidity : %d.%02u",
					 (u_int8_t) humidity_physical_value,
					 (u_int8_t) ((humidity_physical_value - (u_int8_t) humidity_physical_value) * 100 ) );
#if DEBUG_CONSOLE
			printf( " -> %s %%\n", tmp );
#endif
		}
		if (measurement_id == 0x00) {
			temperature_physical_value = ( (payload * 175.72 ) /65536 ) - 46.85;
			sprintf( tmp,
			         "Temperature : %d.%02u",
					 (u_int8_t) temperature_physical_value,
					 (u_int8_t) ((temperature_physical_value - (u_int8_t) temperature_physical_value) * 100) );
#if DEBUG_CONSOLE
			printf( "-> %s °C\n", tmp);
#endif
		}

		pubmsg.payload = tmp;
		pubmsg.payloadlen = (int)strlen(tmp);
		pubmsg.qos = QOS;
		pubmsg.retained = 0;
		if ((rc = MQTTClient_publishMessage(client, TOPIC, &pubmsg, &token)) != MQTTCLIENT_SUCCESS) {
			 printf("Failed to publish message, return code %d\n", rc);
			 exit(EXIT_FAILURE);
		}

		rc = MQTTClient_waitForCompletion(client, token, TIMEOUT);
	}

	return EXIT_SUCCESS;
}
