/* This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/kernel.h>
...

struct pn5xx_dev;

/**********************************************************
 * Interrupt control and handler
 **********************************************************/
static void pn5xx_disable_irq(struct pn5xx_dev *pn5xx_dev)
{
    unsigned long flags;
    spin_lock_irqsave(&pn5xx_dev->irq_enabled_lock, flags);
    if (pn5xx_dev->irq_enabled) {
        disable_irq_nosync(pn5xx_dev->client->irq);
        pn5xx_dev->irq_enabled = false;
    }
    spin_unlock_irqrestore(&pn5xx_dev->irq_enabled_lock, flags);
}

static irqreturn_t pn5xx_dev_irq_handler(int irq, void *dev_id)
{
    struct pn5xx_dev *pn5xx_dev = dev_id;
    pn5xx_disable_irq(pn5xx_dev);
    /* Wake up waiting readers */
    wake_up(&pn5xx_dev->read_wq);
    return IRQ_HANDLED;
}

/**********************************************************
 * private functions
 **********************************************************/
static void p61_update_access_state(struct pn5xx_dev *pn5xx_dev, p61_access_state_t current_state, bool set);
static void p61_access_lock(struct pn5xx_dev *pn5xx_dev) {mutex_lock(&pn5xx_dev->p61_state_mutex);}
static void p61_access_unlock(struct pn5xx_dev *pn5xx_dev) {mutex_unlock(&pn5xx_dev->p61_state_mutex);}

static void signal_handler(p61_access_state_t state, long nfc_pid)
{
    struct siginfo sinfo;
    pid_t pid;
    struct task_struct *task;

    sinfo.si_signo = SIG_NFC;sinfo.si_code = SI_QUEUE;sinfo.si_int = state;
    task = pid_task(find_vpid(nfc_pid), PIDTYPE_PID);
    send_sig_info(SIG_NFC, &sinfo, task);
}

static int pn5xx_enable(struct pn5xx_dev *dev, int mode) {...}
static void pn5xx_disable(struct pn5xx_dev *dev) {...}

/**********************************************************
 * driver functions
 **********************************************************/
static ssize_t pn5xx_dev_read(struct file *filp, char __user *buf,
        size_t count, loff_t *offset)
{
    struct pn5xx_dev *pn5xx_dev = filp->private_data;
    char tmp[MAX_BUFFER_SIZE];

    mutex_lock(&pn5xx_dev->read_mutex);
    if (!gpio_get_value(pn5xx_dev->irq_gpio)) {
        while (1) {
            enable_irq(pn5xx_dev->client->irq);
            pn5xx_dev->irq_enabled = true;
            wait_event_interruptible(
                    pn5xx_dev->read_wq,
                    !pn5xx_dev->irq_enabled);
            pn5xx_disable_irq(pn5xx_dev);
            if (gpio_get_value(pn5xx_dev->irq_gpio))
                break;
        }
    }

    /* Read data */
    ret = i2c_master_recv(pn5xx_dev->client, tmp, count);
    mutex_unlock(&pn5xx_dev->read_mutex);

    if (copy_to_user(buf, tmp, ret)) {return -EFAULT;}
    return 0;
}

static ssize_t pn5xx_dev_write(struct file *filp, const char __user *buf,
        size_t count, loff_t *offset)
{
    struct pn5xx_dev  *pn5xx_dev;
    char tmp[MAX_BUFFER_SIZE];
	int ret;
	
    pn5xx_dev = filp->private_data;
    if (copy_from_user(tmp, buf, count)) {return -EFAULT;}

    /* Write data */
    ret = i2c_master_send(pn5xx_dev->client, tmp, count);
    if (ret != count) {ret = -EIO;}
    return ret;
}

static int pn5xx_dev_open(struct inode *inode, struct file *filp)
{
    struct pn5xx_dev *pn5xx_dev = container_of(filp->private_data, struct pn5xx_dev, pn5xx_device);
    filp->private_data = pn5xx_dev;
    return 0;
}

static int pn5xx_dev_release(struct inode *inode, struct file *filp) {return 0;}

static long  pn5xx_dev_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    struct pn5xx_dev *pn5xx_dev = filp->private_data;

    p61_access_lock(pn5xx_dev);
    switch (cmd) {
    case PN5XX_SET_PWR: pn5xx_enable(pn5xx_dev, arg); break;
    case PN5XX_CLK_REQ: gpio_set_value(pn5xx_dev->clkreq_gpio, arg); break;
    case PN5XX_SET_NFC_SERVICE_PID: pn5xx_dev->nfc_service_pid = arg; break;
    default:
        return -EINVAL;
    }
    p61_access_unlock(pn5xx_dev);
    return 0;
}

static const struct file_operations pn5xx_dev_fops = {
    .owner    = THIS_MODULE,
    .llseek    = no_llseek,
    .read    = pn5xx_dev_read,
    .write    = pn5xx_dev_write,
    .open    = pn5xx_dev_open,
    .release  = pn5xx_dev_release,
    .unlocked_ioctl  = pn5xx_dev_ioctl,
};

static int pn5xx_get_pdata(struct device *dev,struct pn5xx_i2c_platform_data *pdata) {pdata = dev->platform_data; return 0;}

/*
 *  pn5xx_probe
 */
static int pn5xx_probe(struct i2c_client *client,
        const struct i2c_device_id *id)
{
    int ret;
    struct pn5xx_i2c_platform_data *pdata;  // gpio values, from board file or DT
    struct pn5xx_i2c_platform_data tmp_pdata;
    struct pn5xx_dev *pn5xx_dev;            // internal device specific data

    pdata = client->dev.platform_data;
    
    /* reserve the GPIO pins */
    gpio_request(pdata->irq_gpio, "nfc_int");
    client->irq = gpio_to_irq(pdata->irq_gpio);
    gpio_request(pdata->ven_gpio, "nfc_ven");
    gpio_request(pdata->clkreq_gpio, "nfc_clkreq");

    /* allocate the pn5xx driver information structure */
    pn5xx_dev = kzalloc(sizeof(*pn5xx_dev), GFP_KERNEL);
    if (pn5xx_dev == NULL) {
        dev_err(&client->dev, "failed to allocate memory for module data\n");
        ret = -ENOMEM;
        goto err_exit;
    }

    /* store the platform data in the driver info struct */
    pn5xx_dev->irq_gpio = pdata->irq_gpio;
    pn5xx_dev->ven_gpio  = pdata->ven_gpio;
    pn5xx_dev->firm_gpio  = pdata->firm_gpio;
    pn5xx_dev->nfc_ven_enabled = false;
    pn5xx_dev->clkreq_gpio = pdata->clkreq_gpio;
    
    pn5xx_dev->client   = client;

    /* finish configuring the I/O */
    gpio_direction_input(pn5xx_dev->irq_gpio);
    gpio_direction_output(pn5xx_dev->ven_gpio, 0);
    gpio_direction_output(pn5xx_dev->firm_gpio, 0);
    gpio_direction_output(pn5xx_dev->clkreq_gpio, 0);
        
    /* init mutex and queues */
    init_waitqueue_head(&pn5xx_dev->read_wq);
    mutex_init(&pn5xx_dev->read_mutex);
    spin_lock_init(&pn5xx_dev->irq_enabled_lock);

    /* register as a misc device - character based with one entry point */
    pn5xx_dev->pn5xx_device.minor = MISC_DYNAMIC_MINOR;
    pn5xx_dev->pn5xx_device.name = CHIP;
    pn5xx_dev->pn5xx_device.fops = &pn5xx_dev_fops;
    ret = misc_register(&pn5xx_dev->pn5xx_device);
    if (ret) {
        goto err_misc_register;
    }

    pn5xx_dev->irq_enabled = true;
    ret = request_irq(client->irq, pn5xx_dev_irq_handler,
              IRQF_TRIGGER_HIGH, client->name, pn5xx_dev);
    if (ret) {
        dev_err(&client->dev, "request_irq failed\n");
        goto err_request_irq_failed;
    }
    pn5xx_disable_irq(pn5xx_dev);
    i2c_set_clientdata(client, pn5xx_dev);
    return 0;

err_request_irq_failed:
    misc_deregister(&pn5xx_dev->pn5xx_device);
err_misc_register:
    mutex_destroy(&pn5xx_dev->read_mutex);
    mutex_destroy(&pn5xx_dev->p61_state_mutex);
    kfree(pn5xx_dev);
err_exit:
    if (gpio_is_valid(pdata->clkreq_gpio))
        gpio_free(pdata->clkreq_gpio);
    return ret;
}

static int pn5xx_remove(struct i2c_client *client)
{
    struct pn5xx_dev *pn5xx_dev;

    pn5xx_dev = i2c_get_clientdata(client);
    free_irq(client->irq, pn5xx_dev);
    misc_deregister(&pn5xx_dev->pn5xx_device);
    mutex_destroy(&pn5xx_dev->read_mutex);
    pn5xx_dev->nfc_ven_enabled = false;
    pn5xx_dev->spi_ven_enabled = false;
    gpio_free(pn5xx_dev->firm_gpio);
    gpio_free(pn5xx_dev->clkreq_gpio);
    kfree(pn5xx_dev);
    return 0;
}

static struct i2c_driver pn5xx_driver = {
    .probe        = pn5xx_probe,
    .remove        = pn5xx_remove,
    .driver        = {.owner = THIS_MODULE, .name = "pn544"},
};

static int __init pn5xx_dev_init(void) {return i2c_add_driver(&pn5xx_driver);}
static void __exit pn5xx_dev_exit(void) {i2c_del_driver(&pn5xx_driver);}

module_init(pn5xx_dev_init);
module_exit(pn5xx_dev_exit);

MODULE_AUTHOR("Sylvain Fonteneau");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
