/**
 *   @file  queueAlloc.c
 *
 *   @brief   
 *      This is the Queue Allocation test file.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  \par
*/

/* XDC includes */ 
#include <xdc/std.h>
#include <xdc/runtime/IHeap.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Memory.h>
#include <xdc/runtime/Error.h>
#include <xdc/cfg/global.h>
#include <string.h>

/* sysbios includes */ 
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>

/* CPPI LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>

/* CSL RL includes */
#include <ti/csl/csl_chip.h>
#include <ti/csl/csl_cacheAux.h>

/************************ USER DEFINES ********************/

/* Define to test AIF, FFTC A and FFTC B */
#undef  TEST_ALL

#define SYSINIT                     0

#define CPPI_RX_CHANNEL_0_NUM       1
#define CPPI_TX_CHANNEL_0_NUM       1

#define NUM_HOST_DESC               32
#define SIZE_HOST_DESC              48
#define NUM_MONOLITHIC_DESC         32
#define SIZE_MONOLITHIC_DESC        160
#define NUM_DATA_BUFFER             32
#define SIZE_DATA_BUFFER            64

#define MAPPED_VIRTUAL_ADDRESS      0x81000000

/* MPAX segment 2 registers */
#define XMPAXL2                     0x08000010 
#define XMPAXH2                     0x08000014

/************************ GLOBAL VARIABLES ********************/
/* linking RAM */
#pragma DATA_ALIGN (linkingRAM0, 16)
UInt64                  linkingRAM0[NUM_HOST_DESC + NUM_MONOLITHIC_DESC];
/* Descriptor pool [Size of descriptor * Number of descriptors] */
#pragma DATA_ALIGN (hostDesc, 16)
UInt8                   hostDesc[SIZE_HOST_DESC * NUM_HOST_DESC];
#pragma DATA_ALIGN (monolithicDesc, 16)
UInt8                   monolithicDesc[SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC];
#pragma DATA_ALIGN (dataBuff, 16)
UInt8                   dataBuff[SIZE_DATA_BUFFER * NUM_DATA_BUFFER];
/* QMSS configuration */
Qmss_InitCfg            qmssInitConfig;
/* Memory region configuration information */
Qmss_MemRegInfo         memInfo;

#pragma DATA_SECTION (isSysInitialized, ".qmss");
volatile Uint32             isSysInitialized;

/************************ EXTERN VARIABLES ********************/

/* QMSS device specific configuration */
extern Qmss_GlobalConfigParams  qmssGblCfgParams;

/*************************** FUNCTIONS ************************/

/**
 *  @b Description
 *  @n  
 *      Utility function which converts a local GEM L2 memory address 
 *      to global memory address.
 *
 *  @param[in]  addr
 *      Local address to be converted
 *
 *  @retval
 *      Computed L2 global Address
 */
#ifndef INTERNAL_LINKING_RAM 
static UInt32 l2_global_address (UInt32 addr)
{
	UInt32 corenum;

	/* Get the core number. */
	corenum = CSL_chipReadReg (CSL_CHIP_DNUM); 

	/* Compute the global address. */
	return (addr + (0x10000000 + (corenum * 0x1000000)));
}
#endif

/**
 *  @b Description
 *  @n  
 *      Entry point for the test code.
 *      This is an example code that shows CPPI LLD API usage.
 *
 *      It performs the following
 *          - Initializes the Queue Manager low level driver.
 *          - Initializes the CPPI low level driver.
 *          - Opens the CPPI CPDMA in queue manager
 *          - Initializes descriptors and pushes to free queue
 *          - Opens Rx and Tx channel
 *          - Pushes packet on Tx channel. Diverts the packet to Tx channel.
 *          - Process the Rx packet
 *          - Closes Rx and Tx channel
 *          - Closes all open queues
 *          - Closes CPDMA instance
 *          - Deinitializes CPPI LLD
 *
 *  @retval
 *      Not Applicable.
 */
Void queueAllocation (Void)
{
    Int32                   result;
    UInt32                  i;
    UInt8                   isAllocated;
    Qmss_QueueHnd           txQueHnd;
    UInt32                  corenum ;

#ifdef L2_CACHE
    uint32_t                *xmpaxPtr;
#endif

    System_printf ("**************************************************\n");
    System_printf ("************ QMSS Queue Allocate testing**********\n");
    System_printf ("**************************************************\n");

    /* Get the core number. */
	corenum = CSL_chipReadReg(CSL_CHIP_DNUM);

    System_printf ("*******Test running on Core %d *******************\n", corenum);

    if (corenum == SYSINIT)
    {
        /* Reset the variable to indicate to other cores system init is not yet done */
        isSysInitialized = 0;
#ifdef L2_CACHE
        /* Set L2 cache to 512KB */
        CACHE_setL2Size (CACHE_512KCACHE);
#endif
        System_printf ("Core %d : L1D cache size %d. L2 cache size %d.\n", corenum, CACHE_getL1DSize(), CACHE_getL2Size());
    
#ifdef L2_CACHE
        /* Define an MPAX segment in the virtual (CGEM) address space. 
         * Map MSMC physical address to virtual address.
         * Configure all +rwx permissions.
         */

        /* Phy address base: 0x0C00 0000
         * Size: 1MB (0x13 according to encoding)
         * Virtual address: 0x8100 0000 
         * Permission: 0xFF
         * MAR used: (0x8100 0000 >> 24) = 129
         */

        /* map using MPAX segment 2 registers */
        xmpaxPtr  = (uint32_t *)(XMPAXH2);
        *xmpaxPtr = ((MAPPED_VIRTUAL_ADDRESS >> 12) << 12) | (0x13);

        xmpaxPtr  = (uint32_t *)(XMPAXL2);
        *xmpaxPtr = ((0x0c000000 >> 12) << 8) | (0xFF);
    
        /* Enable caching for MAR 129. CSL does not define these MARS. Define a macro */
        CACHE_enableCaching ((MAPPED_VIRTUAL_ADDRESS) >> 24);
#endif 
        memset ((Void *) &linkingRAM0, 0, sizeof (linkingRAM0));
        memset ((Void *) &qmssInitConfig, 0, sizeof (Qmss_InitCfg));

        /* Set up the linking RAM. Use the internal Linking RAM. 
         * LLD will configure the internal linking RAM address and default size if a value of zero is specified.
         * Linking RAM1 is not used */
#ifdef INTERNAL_LINKING_RAM 
        qmssInitConfig.linkingRAM0Base = 0;
        qmssInitConfig.linkingRAM0Size = 0;
        qmssInitConfig.linkingRAM1Base = 0;
        qmssInitConfig.maxDescNum      = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
#else
        qmssInitConfig.linkingRAM0Base = l2_global_address ((UInt32) linkingRAM0);
        qmssInitConfig.linkingRAM0Size = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
        qmssInitConfig.linkingRAM1Base = 0;
        qmssInitConfig.maxDescNum      = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
#endif    

        /* Initialize Queue Manager SubSystem */
        result = Qmss_init (&qmssInitConfig, &qmssGblCfgParams);
        if (result != QMSS_SOK)
        {
            System_printf ("Error Core %d : Initializing Queue Manager SubSystem error code : %d\n", corenum, result);
            return;
        }
    
        /* Start Queue Manager SubSystem */
        result = Qmss_start ();
        if (result != QMSS_SOK)
        {
            System_printf ("Core %d : Error starting Queue Manager error code : %d\n", corenum, result);
        }
        else
            System_printf ("\nCore %d : QMSS initialization done\n", corenum);

        /* Indicate to other cores system init is done */
        isSysInitialized = 1;

        /* Writeback L1D */
        CACHE_wbL1d ((void *) &isSysInitialized, 4, CACHE_WAIT);   
    }
    else
    {
        /* Start Queue Manager SubSystem */
        System_printf ("Core %d : Waiting for QMSS to be initialized...\n\n", corenum);

        /* Synchronize all consumer cores. They must wait for the producer core to finish initialization. */
   
        do{
            CACHE_invL1d ((void *) &isSysInitialized, 4, CACHE_WAIT);
        } while (isSysInitialized == 0);

        /* Start Queue Manager SubSystem */
        result = Qmss_start ();
        if (result != QMSS_SOK)
        {
            System_printf ("Core %d : Error starting Queue Manager error code : %d\n", corenum, result);
        }
        else
            System_printf ("\nCore %d : QMSS initialization done\n", corenum);
    }

    System_printf ("Core %d : Allocating Low priority Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_LOW_PRIORITY_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_LOW_PRIORITY_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_LOW_PRIORITY_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_LOW_PRIORITY_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_LOW_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening Low priority Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

#ifdef TEST_ALL
    System_printf ("Core %d : Allocating AIF Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_AIF_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_AIF_QUEUE].startIndex;
		   i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_AIF_QUEUE].startIndex + 
		   qmssGblCfgParams.maxQueueNum[Qmss_QueueType_AIF_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_AIF_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening AIF Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating PASS Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_PASS_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_PASS_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_PASS_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_PASS_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_PASS_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening PASS Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }
#endif

    System_printf ("Core %d : Allocating Intc Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INTC_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INTC_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INTC_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INTC_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_INTC_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening Intc Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating SRIO Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_SRIO_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_SRIO_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_SRIO_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_SRIO_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_SRIO_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening SRIO Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

#ifdef TEST_ALL

    System_printf ("Core %d : Allocating FFTC A Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_A_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_A_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_A_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_A_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_FFTC_A_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening FFTC A *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating FFTC B Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_B_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_B_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_B_QUEUE].startIndex +
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_FFTC_B_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_FFTC_B_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening FFTC B Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }
#endif

    System_printf ("Core %d : Allocating High Priority Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_HIGH_PRIORITY_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_HIGH_PRIORITY_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_HIGH_PRIORITY_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_HIGH_PRIORITY_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_HIGH_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening High priority Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating Starvation Cnt Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_STARVATION_COUNTER_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_STARVATION_COUNTER_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_STARVATION_COUNTER_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_STARVATION_COUNTER_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_STARVATION_COUNTER_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening Starvation Cnt Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating InfraStructure Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INFRASTRUCTURE_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INFRASTRUCTURE_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INFRASTRUCTURE_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_INFRASTRUCTURE_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_INFRASTRUCTURE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening InfraStructure *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating Traffic Shaping Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_TRAFFIC_SHAPING_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_TRAFFIC_SHAPING_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_TRAFFIC_SHAPING_QUEUE].startIndex + 
		    qmssGblCfgParams.maxQueueNum[Qmss_QueueType_TRAFFIC_SHAPING_QUEUE].maxNum; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_TRAFFIC_SHAPING_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening Traffic Shaping Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }

    System_printf ("Core %d : Allocating GP Queues Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 20; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *******************Done opening GP Queues *********************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
    }
    
    System_printf ("Core %d : Allocating Queues within a specified range Starting at %d Queue Number\n", 
                    corenum, qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 50);

    for (i = qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 50; 
		    i < qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 61; i++)
    {
    	/* Opens transmit queue */
	    if ((txQueHnd = Qmss_queueOpenInRange (qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 50, 
                                        qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 60, &isAllocated)) < 0)
    	{
	    	System_printf ("Core %d : *****************Done opening Queues in Range*******************\n", corenum);
		    break;
    	}
        else
            System_printf ("Core %d : Queue Number : %d opened. Ref count %d\n", corenum, txQueHnd, isAllocated);
        if (i ==  qmssGblCfgParams.maxQueueNum[Qmss_QueueType_GENERAL_PURPOSE_QUEUE].startIndex + 55)
            if (Qmss_queueClose (txQueHnd) < 0 )
    	    	System_printf ("Core %d : Error closing Queue Number : %d \n", corenum, txQueHnd);
    }

    System_printf ("*******************************************************\n");
    System_printf ("*********** QMSS Queue Allocate testing Done **********\n");
    System_printf ("*******************************************************\n");
}

Void run_test (Void)
{
    queueAllocation ();
}

