/*
 *
 * Copyright (C) 2010-2012 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



#include "../pautest.h"
#include "test3pkts.h"

#ifdef __LINUX_USER_SPACE
#include "fw_test.h"
#include "fw_mem_allocator.h"
#endif


/* Format tx route and checksum verification test
 * This test tests the LLD Pa_formatTxRoute and Pa_formatTxCmd APIs, as well as the
 * checksum verification for IPv4 header, UDP, UDP-lite and TCP headers on reception.
 * This test also tests the transmit timestamp report and timestamp insertion of incoming
 * packets
 * 
 * This test covers the following sub-tests:
 *  - Test the LLD for formatting the checksum and routing commands
 *  - Test the LLD for formatting the CRC, routing and report timestamp commands
 *  - Test the firmware for correct packet routing after processing the checksum command
 *  - Test the PA sub-system for correct IPv4 header checksum calculation and verification
 *  - Test the PA sub-system for correct TCP/UDP/UDP-lite payload checksum calculation
 *  - Test the firmware for the ability to process the report timestamp command
 */

static char *tfName = "paTestTxFormatRoute";

#undef PA_T3_SHOW_TIMESTAMP

/* General purpose queue usage */
#define Q_CMD_RECYCLE	    0	/* Command descriptors/buffers recycle */
#define Q_CMD_REPLY			1   /* Replies from PA */
#define Q_MATCH				2	/* Packets from PA which match a lookup */
#define Q_NFAIL             3   /* Packet from PA which failed a lookup */
#define Q_TIMESTAMP         4   /* Packet from PA which contains the tx timestamp */

/* Handles managed by this test */
#define TEST_NUM_L2_HANDLES     1
#define TEST_NUM_L3_HANDLES     1  /* 2 */
#define TEST_NUM_L4_HANDLES     1  /* 2 */


/* Command reply SW Info identifiers */
#define TEST_SWINFO0_CMD_ID     0x12340000
#define TEST_SWINFO0_STATS_REQ  0x12340001
#define TEST_SWINFO0_TIMESTAMP  0x12340002

static paSysStats_t paTestTxFmtExptStats;    /* Expected stats results */

/* The IPv4 packet is sent four times. The first time with no
 * checksum calculation. This packet should return with the IP header 
 * and udp checksum flag showing invalid checksums. This is followed
 * by a command to do the IPv4 header checksum only, then the
 * UDP checksum only, and then both checksums. */
#define CMD_BUF_SIZE  (sizeof(pasahoNextRoute_t) + (3 * sizeof(pasahoComChkCrc_t)) + sizeof(pasahoReportTimestamp_t))
//static uint8_t paT3pkt[4][sizeof(t3pkt)];
#ifdef __LINUX_USER_SPACE
static uint8_t* paT3pkt[4] = {NULL, NULL, NULL, NULL};
#else
static uint8_t paT3pkt0[sizeof(t3pkt)];
static uint8_t paT3pkt1[sizeof(t3pkt)];
static uint8_t paT3pkt2[sizeof(t3pkt)];
static uint8_t paT3pkt3[sizeof(t3pkt)];
static uint8_t* paT3pkt[4] = {paT3pkt0, paT3pkt1 ,paT3pkt2, paT3pkt3};
#endif

static paTestStatus_t testSendIpv4  (tFramework_t *tf, paTest_t *pat)
{
    Cppi_HostDesc *hd;

    uint32_t   cmdStack[4][(sizeof(pasahoNextRoute_t) + (3 * sizeof(pasahoComChkCrc_t)) + sizeof(pasahoReportTimestamp_t)) / sizeof (uint32_t)];
    uint16_t   cmdStackSize[4] = {CMD_BUF_SIZE, CMD_BUF_SIZE, CMD_BUF_SIZE, CMD_BUF_SIZE};
    uint32_t   rxTimestamp[4], txTimestamp[4];
    paCmdInfo_t cmdInfo[5];
	uint32_t        *swinfo;
    int32_t      i;
    uint16_t   len;
	volatile int32_t    mdebugWait = 0;
    Qmss_Queue   q;
    uint16_t       tsQueue; 

    /* Because the UDP checksum must be zero, it will not be checked by default 
     * but only when the checksum is generated */
    uint32_t errFlagsExptd[4] = { 0xc, 0x4, 0x8, 0x0 }; 
    uint32_t eflags;
    
    paReturn_t paret;

    pasahoNextRoute_t  *panr;
    pasahoReportTimestamp_t    *pats;


    paRouteInfo_t  route =  {  pa_DEST_HOST,  /* Route - host         */
                               0,             /* flow Id              */
                               0,  			  /* Queue                */
                               -1,            /* Multi route disabled */
                               0,             /* SWInfo 0             */
                               0,             /* SWInfo 1 */       
                               0,             /* customType : not used */         
                               0,             /* customIndex: not used */     
                               0,             /* pkyType: for SRIO only */    
                               NULL};         /* No commands */
                               
    paCmdNextRoute_t routeCmd = {
                                    0,              /*  ctrlBitfield */
                                    pa_DEST_HOST,   /* Route - host       */
                                    0,              /* pktType don't care */
                                    0,              /* flow Id              */
                                    0,  			/* Queue                */
                                    0,              /* SWInfo 0             */
                                    0,              /* SWInfo 1 */       
                                    0               /* multiRouteIndex (not used) */
    
                                };                              

                        

    /* Create the command stack for the four packet sends */
    memset(cmdStack, 0, sizeof(cmdStack));
    memset(cmdInfo, 0, sizeof(cmdInfo));
    
	/* Route the packet to PA Tx 0 which makes it appear to have arrived over the network */
	route.queue = tf->QPaTx[0];
	routeCmd.queue = tf->QPaTx[0];
    tsQueue = tf->QGen[Q_TIMESTAMP];
    
    /* Packet 1 - send to PA, route back to QM into PA 0 as a received packet */
    /* report timestamp */
    memcpy(paT3pkt[0], t3pkt, sizeof(t3pkt));
    /* corrupt UDP checksum */
    paT3pkt[0][40] = 0xbe;
    paT3pkt[0][41] = 0xef;
    pats = (pasahoReportTimestamp_t *)&cmdStack[0][0];
    PASAHO_SET_CMDID(pats, PASAHO_PAMOD_REPORT_TIMESTAMP);
    PASAHO_SET_REPORT_FLOW(pats, 0);
    PASAHO_SET_REPORT_QUEUE(pats,tsQueue);
    pats->swInfo0 = TEST_SWINFO0_TIMESTAMP;
    
    panr = (pasahoNextRoute_t *)&cmdStack[0][sizeof(pasahoReportTimestamp_t)/sizeof(uint32_t)];

    paret = Pa_formatTxRoute (  NULL,                /* IPv4 header checksum */
                                NULL,                /* No second checksum   */
                                &route,              /* Internal routing     */
                                (Ptr)panr,           /* Command buffer       */
                                &cmdStackSize[0]);   /* Command size         */

    if (paret != pa_OK)  {
        System_printf ("%s (%s:%d): Pa_formatTxRoute returned error code %d\n", tfName, __FILE__, __LINE__, paret);
        return (PA_TEST_FAILED);
    }

    cmdStackSize[0] += sizeof(pasahoReportTimestamp_t);

    /* Packet 2 - send to PA for IPv4 header checksum calculation, then back to
     * QM in the PA 0 out queue */
    /* insert timestamp */
    memcpy(paT3pkt[1], t3pkt, sizeof(t3pkt));
    /* corrupt UDP checksum */
    paT3pkt[1][40] = 0xba;
    paT3pkt[1][41] = 0xbe;
    pats = (pasahoReportTimestamp_t *)&cmdStack[1][0];
    PASAHO_SET_CMDID(pats, PASAHO_PAMOD_REPORT_TIMESTAMP);
    PASAHO_SET_REPORT_FLOW(pats, 0);
    PASAHO_SET_REPORT_QUEUE(pats,tsQueue);
    pats->swInfo0 = TEST_SWINFO0_TIMESTAMP;
     
    panr = (pasahoNextRoute_t *)&cmdStack[1][sizeof(pasahoReportTimestamp_t)/sizeof(uint32_t)];

    paret = Pa_formatTxRoute (  &t3pktIpChksum,      /* IPv4 header checksum */
                                NULL,                /* No second checksum   */
                                &route,              /* Internal routing     */
                                (Ptr)panr,           /* Command buffer       */
                                &cmdStackSize[1]);   /* Command size         */


    if (paret != pa_OK)  {
        System_printf ("%s (%s:%d): Pa_formatTxRoute returned error code %d\n", tfName, __FILE__, __LINE__, paret);
        return (PA_TEST_FAILED);
    }
    
    cmdStackSize[1] += sizeof(pasahoReportTimestamp_t);
    
    /* Packet 3 - send to PA for UDP checksum calculation, then back to QM
     * in the PA 0 out queue */
    /* insert timestamp */
    memcpy(paT3pkt[2], t3pkt, sizeof(t3pkt));
    pats = (pasahoReportTimestamp_t *)&cmdStack[2][0];
    PASAHO_SET_CMDID(pats, PASAHO_PAMOD_REPORT_TIMESTAMP);
    PASAHO_SET_REPORT_FLOW(pats, 0);
    PASAHO_SET_REPORT_QUEUE(pats,tsQueue);
    pats->swInfo0 = TEST_SWINFO0_TIMESTAMP;
    
    panr = (pasahoNextRoute_t *)&cmdStack[2][sizeof(pasahoReportTimestamp_t)/sizeof(uint32_t)];

    paret = Pa_formatTxRoute (  &t3pktUdpChksum,     /* IPv4 header checksum */
                                NULL,                /* No second checksum   */
                                &route,              /* Internal routing     */
                                (Ptr)panr,           /* Command buffer       */
                                &cmdStackSize[2]);   /* Command size         */

    if (paret != pa_OK)  {
        System_printf ("%s (%s:%d): Pa_formatTxRoute returned error code %d\n", tfName, __FILE__, __LINE__, paret);
        return (PA_TEST_FAILED);
    }

    cmdStackSize[2] += sizeof(pasahoReportTimestamp_t);
    
    /* Packet 4 - both checksums, then back to QM for output */
    /* Commad 0: insert timestamp */
    memcpy(paT3pkt[3], t3pkt, sizeof(t3pkt));
    cmdInfo[0].cmd = pa_CMD_REPORT_TX_TIMESTAMP;
    cmdInfo[0].params.txTs.destQueue = tsQueue;
    cmdInfo[0].params.txTs.flowId    = 0;
    cmdInfo[0].params.txTs.swInfo0   = TEST_SWINFO0_TIMESTAMP;
    
    /* Command 1: IP checksum */
    cmdInfo[1].cmd = pa_CMD_TX_CHECKSUM;
    cmdInfo[1].params.chksum = t3pktIpChksum;
    
    /* Command 2: UDP checksum */
    //cmdInfo[2].cmd = pa_CMD_TX_CHECKSUM;
    //cmdInfo[2].params.chksum = t3pktUdpChksum;
    
    /* Command 3: CRC */
    cmdInfo[2].cmd = pa_CMD_CRC_OP;
    cmdInfo[2].params.crcOp = t3pktCrc;
    
    /* Command 3: Next route */
    cmdInfo[3].cmd = pa_CMD_NEXT_ROUTE;
    cmdInfo[3].params.route = routeCmd;
    
    paret = Pa_formatTxCmd (  4,        /* nCmd */
                              cmdInfo,  /* command info */
                              0,        /* offset */
                              (Ptr)&cmdStack[3][0], /* Command buffer       */
                              &cmdStackSize[3]);    /* Command size         */
                              
    if (paret != pa_OK)  {
        System_printf ("%s (%s:%d): Pa_formatTxCmd returned error code %d\n", tfName, __FILE__, __LINE__, paret);
        return (PA_TEST_FAILED);
    }

    /* Send the packets. Each packet transmission will result in the popping of two free
     * descriptors/buffers since the packet loops back into the QM twice - once with
     * the packet and checksums (and this arrives in Q 640 for transmit into PA) and
     * once again with the resulting packet. The descriptors popped off for inter-pa 
     * transmit will wind up in the default return Q */

    /* Recycle the free descriptors right back to the free descriptor queue */
    q.qMgr = 0;
    q.qNum = tf->QfreeDesc;

    for (i = 0; i < 4; i++)   {

        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QfreeDesc)) & ~15);
        if (hd == NULL)  {
            System_printf ("%s (%s:%d): Failed to pop a descriptor off the free descriptor Q (%d)\n", tfName, __FILE__, __LINE__, tf->QfreeDesc);
            return (PA_TEST_FAILED);
        }

        /* Setup the return for the descriptor */
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);

        /* Attach the data and set the length */
        //Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)utilgAddr((uint32_t)&paT3pkt[i]), sizeof(t3pkt));
        Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)utilgAddr((uint32_t)paT3pkt[i]), sizeof(t3pkt));
        Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, sizeof(t3pkt));

        /* Attach the command in PS data */
        Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)cmdStack[i], cmdStackSize[i]);

        /* Send the packet to PDSP 5 */
        //mdebugWait = 1;
        //if(i == 3)
  		//    mdebugHaltPdsp(5);  

        Qmss_queuePush (tf->QPaTx[5], (Ptr)hd, sizeof(t3pkt), TF_SIZE_DESC, Qmss_Location_TAIL);
        paTestTxFmtExptStats.classify1.nPackets     += 2;     /* Passes once through mac, once through IP */
        paTestTxFmtExptStats.classify1.nTableMatch  += 2;
        paTestTxFmtExptStats.classify2.nPackets     += 1;     
        paTestTxFmtExptStats.classify2.nUdp         += 1;
        paTestTxFmtExptStats.classify1.nIpv4Packets += 1;
        
        //if(i == 3)
        //    while(mdebugWait);
            
        //utilCycleDelay (3000);
                    

    }

    /* Wait for 4 packets to arrive in the match queue */
    for (i = 0; (i < 100) && (Qmss_getQueueEntryCount(tf->QGen[Q_MATCH]) < 4); i++) 
        utilCycleDelay (100);

    if (i == 100)  {
        System_printf ("%s (%s:%d): Did not find 4 packets in the match queue\n", tfName, __FILE__, __LINE__);
        return (PA_TEST_FAILED);
    }

    /* Verify the checksum fields in the CPPI descriptor. These must arrive in order */
    for (i = 0; i < 4; i++)  {
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_MATCH])) & ~15);
        if (hd == NULL)  {
            System_printf ("%s (%s:%d): Error - popped a NULL descriptor\n", tfName, __FILE__, __LINE__);
            return (PA_TEST_FAILED);
        }

        eflags = Cppi_getDescError (Cppi_DescType_HOST, (Cppi_Desc *)hd) & 0xf;
        Cppi_getTimeStamp (Cppi_DescType_HOST, (Cppi_Desc *)hd, &rxTimestamp[i]);

        testCommonRecycleLBDesc (tf, hd); 

        if (eflags != errFlagsExptd[i])  {
            System_printf ("%s (%s:%d): Expected error flags = %d, found %d\n", tfName, __FILE__, __LINE__, errFlagsExptd[i], eflags);
            return (PA_TEST_FAILED);
        }

        /* Also recycle the descriptor with linked buffer used in the intermediate transfer */
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QDefRet)) & ~15);
        if (hd == NULL)  {
            System_printf ("%s (%s:%d): Did not find descriptor for transferred packet\n", tfName, __FILE__, __LINE__);
            return (PA_TEST_FAILED);
        }
        testCommonRecycleLBDesc (tf, hd);
        
        /* Also verify the tx timestamp report packet */
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_TIMESTAMP])) & ~15);
        if (hd == NULL)  {
            System_printf ("%s (%s:%d): Did not find Tx timestamp report packet\n", tfName, __FILE__, __LINE__);
            return (PA_TEST_FAILED);
        }
        
        /* verify the swInfo for the received packet */
        Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swinfo);
        
        if (swinfo[0] != TEST_SWINFO0_TIMESTAMP)  {
            System_printf ("%s (%s:%d): Expected timestamp swInfo = 0x%08x, found 0x%08x\n", tfName, __FILE__, __LINE__, TEST_SWINFO0_TIMESTAMP,swinfo[0]);
            return (PA_TEST_FAILED);
        }
        
        /* record and compare timestamp */
        Cppi_getTimeStamp (Cppi_DescType_HOST, (Cppi_Desc *)hd, &txTimestamp[i]);
        
#ifdef PA_T3_SHOW_TIMESTAMP        
        System_printf ("pkt %d: Rx timestamp 0x%08x; Tx timestamp 0x%08x\n", i, rxTimestamp[i],txTimestamp[i]);
        System_flush();
#endif        
        
        if (txTimestamp[i] > rxTimestamp[i])
        {
            System_printf ("%s (%s:%d): Rx timestamp 0x%08x is ealier than Tx timestamp 0x%08x\n", tfName, __FILE__, __LINE__, rxTimestamp[i],txTimestamp[i]);
            return (PA_TEST_FAILED);
        }
        
        
        testCommonRecycleLBDesc (tf, hd);

    }
    
    
    /* Test the packet adjustment macros. The packet payload is reduced by 40 bytes,
     * the IP and UDP lengths are changed, and the packet checksum commands are updated to
     * reflect the new length. */
    /* IPv4 length */
    len = ((uint16_t)(t3pkt[T3_PKT_OFFSET_IP_LEN+0]) << 8) | t3pkt[T3_PKT_OFFSET_IP_LEN+1];
    len = len - 40;
    t3pkt[T3_PKT_OFFSET_IP_LEN+0] = len >> 8;
    t3pkt[T3_PKT_OFFSET_IP_LEN+1] = len & 0xff;
    
    /* UDP length */
    len = ((uint16_t)(t3pkt[T3_PKT_OFFSET_UDP_LEN+0]) << 8) | t3pkt[T3_PKT_OFFSET_UDP_LEN+1];
    len = len - 40;
    t3pkt[T3_PKT_OFFSET_UDP_LEN+0] = len >> 8;
    t3pkt[T3_PKT_OFFSET_UDP_LEN+1] = len & 0xff;
    
    pa_SET_TX_CHKSUM_LENGTH (&cmdStack[3][sizeof(pasahoReportTimestamp_t)/sizeof(uint32_t)], 1, len);                   /* Update the UDP checksum length */
    pa_SET_TX_CHKSUM_INITVAL(&cmdStack[3][sizeof(pasahoReportTimestamp_t)/sizeof(uint32_t)], 1, testCommonOnesCompAdd(len, T3_PKT_PSEUDO_HDR_CHKSUM_SANS_LEN));
    
    /*
     * Write back the entire cache to make sure that the test packets are updated. 
     * Note: It may be more efficient to call CACHE_wbL1d(blockPtr, byteCnt, wait) only for
     *       the portion of packet which is updated.
     *
     */
#ifndef __LINUX_USER_SPACE
    CACHE_wbAllL1d(CACHE_WAIT);
    //CACHE_wbAllL2(CACHE_WAIT);
#endif

	/* Send the packet for checksum generation, loop it back and verify the checksum */
    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QfreeDesc)) & ~15);
    if (hd == NULL)  {
    	System_printf ("%s (%s:%d): Failed to pop a descriptor off the free descriptor Q (%d)\n", tfName, __FILE__, __LINE__, tf->QfreeDesc);
        return (PA_TEST_FAILED);
    }

    /* Setup the return for the descriptor */
    Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);

    /* Attach the data and set the length */
#ifdef __LINUX_USER_SPACE
    {
        memcpy (paT3pkt[3], t3pkt, sizeof(t3pkt));
        Cppi_setOriginalBufInfo(Cppi_DescType_HOST, (Cppi_Desc *)hd, paT3pkt[3], sizeof(t3pkt) - 40);
        Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, paT3pkt[3], sizeof(t3pkt) - 40);
    }
#else
    Cppi_setOriginalBufInfo(Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)utilgAddr((uint32_t)t3pkt), sizeof(t3pkt) - 40);
    Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)utilgAddr((uint32_t)t3pkt), sizeof(t3pkt) - 40);
#endif
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, sizeof(t3pkt) - 40);

    /* Attach the command in PS data */
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)cmdStack[3], cmdStackSize[3]);

    /* Send the packet to PDSP 5 */
    Qmss_queuePush (tf->QPaTx[5], (Ptr)hd, sizeof(t3pkt), TF_SIZE_DESC, Qmss_Location_TAIL);
    paTestTxFmtExptStats.classify1.nPackets     += 2;     /* Passes once through mac, once through IP */
    paTestTxFmtExptStats.classify1.nTableMatch  += 2;
    paTestTxFmtExptStats.classify2.nPackets     += 1;     
    paTestTxFmtExptStats.classify2.nUdp         += 1;
    paTestTxFmtExptStats.classify1.nIpv4Packets += 1;
    
    
 /* Wait for the packet to arrive in the match queue */
    for (i = 0; (i < 100) && (Qmss_getQueueEntryCount(tf->QGen[Q_MATCH]) < 1); i++) 
        utilCycleDelay (100);

    if (i == 100)  {
        System_printf ("%s (%s:%d): Did not find the length modified packet in the match queue\n");
        return (PA_TEST_FAILED);
    }

    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_MATCH])) & ~15);
    if (hd == NULL)  {
        System_printf ("%s (%s:%d): Error - popped a NULL descriptor\n", tfName, __FILE__, __LINE__);
        return (PA_TEST_FAILED);
    }

    eflags = Cppi_getDescError (Cppi_DescType_HOST, (Cppi_Desc *)hd) & 0xf;

    testCommonRecycleLBDesc (tf, hd); 

    if (eflags != 0)  {
        System_printf ("%s (%s:%d): Expected error flags = 0, found %d\n", tfName, __FILE__, __LINE__, eflags);
        return (PA_TEST_FAILED);
    }

    /* Also recycle the descriptor with linked buffer used in the intermediate transfer */
    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QDefRet)) & ~15);
    if (hd == NULL)  {
        System_printf ("%s (%s:%d): Did not find descriptor for transferred packet\n", tfName, __FILE__, __LINE__);
        return (PA_TEST_FAILED);
    }
    testCommonRecycleLBDesc (tf, hd);

    /* Also verify the tx timestamp report packet */
    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_TIMESTAMP])) & ~15);
    if (hd == NULL)  {
        System_printf ("%s (%s:%d): Did not find Tx timestamp report packet\n", tfName, __FILE__, __LINE__);
        return (PA_TEST_FAILED);
    }
    
    /* verify the swInfo for the received packet */
    Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swinfo);
    
    if (swinfo[0] != TEST_SWINFO0_TIMESTAMP)  {
        System_printf ("%s (%s:%d): Expected timestamp swInfo = 0x%08x, found 0x%08x\n", tfName, __FILE__, __LINE__, TEST_SWINFO0_TIMESTAMP, swinfo[0]);
        return (PA_TEST_FAILED);
    }
    
    testCommonRecycleLBDesc (tf, hd);

    return (PA_TEST_PASSED);

}    

static int32_t testWaitCmdReply (tFramework_t *tf, paTest_t *pat, int32_t Qcmd, uint32_t swinfo0, int32_t line)
{
	Cppi_HostDesc *hd;
	uint32_t        *swinfo;
    paEntryHandle_t	reth;
    paReturn_t      paret;
    int32_t			    htype;
    int32_t             cmdDest;
	int32_t 		    i;
	
	for (i = 0; i < 100; i++)  {
		if (Qmss_getQueueEntryCount(Qcmd) > 0)  {
			
			hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (Qcmd)) & ~15);

            Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swinfo);
            
            if (swinfo[0] != swinfo0)  {
                System_printf ("%s (%s:%d): found packet in command reply queue without reply (found 0x%08x)\n", tfName, __FILE__, line, swinfo[0]);
                testCommonRecycleLBDesc (tf, hd);
                continue;
            }

            paret = Pa_forwardResult (tf->passHandle, (void *)hd->buffPtr, &reth, &htype, &cmdDest);
            if (paret != pa_OK)  
                System_printf ("%s (%s:%d): Pa_forwardResult returned error %d\n", tfName, __FILE__, line, paret);
            
            testCommonRecycleLBDesc (tf, hd); 
                
            return (0);

        }  else  {
            
          utilCycleDelay (100);

        }
	}
	
	return (-1);

}	


/* Check the stats  */
static paTestStatus_t t3CheckStats (tFramework_t *tf, paTest_t *pat, Bool clear)
{ 	
	Cppi_HostDesc  *hd;
	uint8_t		   *bp;
	paSysStats_t     *paStats;
 	uint32_t			blen;
	int32_t             i;
	paTestStatus_t  status;

	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
 	cmdReply.queue   = (uint16_t) tf->QGen[Q_CMD_REPLY];
 							   
 	
 	/* Check the PA stats to make sure they are set as expected */
 	cmdReply.replyId = TEST_SWINFO0_STATS_REQ;
 	if (testCommonRequestPaStats (tfName, tf, clear, tf->QLinkedBuf1, tf->QGen[Q_CMD_RECYCLE],  &cmdReply))  {
 		System_printf ("%s (%s:%d): testCommonRequestPaStats failed\n", tfName, __FILE__, __LINE__);
 		return (PA_TEST_FAILED);
 	}
 	
 	/* Wait for the stats reply */
	for (i = 0; i < 100; i++)  {
		utilCycleDelay (1000);
		if (Qmss_getQueueEntryCount (tf->QGen[Q_CMD_REPLY]) > 0)
			break;
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Did not find response from PA to stats request command\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
 	
	/* Recycle the descriptor/buffer returned from the stats request */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Did not find returned descriptor/buffer from stats request\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
	
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats request\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
 		
 	/* Format the stats response and compare to the expected results */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_REPLY])) & ~15);
	Cppi_getData (Cppi_DescType_HOST, (Cppi_Desc *)hd, &bp, &blen);
	paStats = (paSysStats_t *)Pa_formatStatsReply (tf->passHandle, (paCmd_t)bp);
	  
    if (testCommonCompareStats (tfName, (paSysStats_t *)&paTestTxFmtExptStats, paStats))
    	status = PA_TEST_FAILED;
    else
    	status = PA_TEST_PASSED;   
    	  
    if (clear)
        memset (&paTestTxFmtExptStats, 0, sizeof(paTestTxFmtExptStats));
    	  
     /* Recycle the descriptor and associated buffer back to queue from which it came */
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats response\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
	
	return (status);
}

static void testCleanup (tFramework_t *tf, paTest_t *pat, paHandleL2L3_t *l2Handles, paHandleL2L3_t *l3Handles, paHandleL4_t *l4Handles, paTestStatus_t testStatus)
{
    Cppi_HostDesc *hd;
    paReturn_t     paret;
    int32_t            cmdDest;
    uint16_t         cmdSize;
    int32_t            cmdCount  = 0;
    int32_t            i;


    paCmdReply_t   reply  =  { pa_DEST_HOST,           /* Destination */
                               TEST_SWINFO0_CMD_ID,    /* SW Info 0   */
                               0,  					   /* Reply queue */
                               0  };                   /* Flow ID     */


	reply.queue = tf->QGen[Q_CMD_REPLY];

    /* Close any open handles */
    for (i = 0; i < TEST_NUM_L2_HANDLES; i++)  {
        if (l2Handles[i] != 0)  {
            hd = testCommonDelHandle (tf, &l2Handles[i], tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, &reply, &cmdDest, &cmdSize, &paret);
            if (hd == NULL)  {
                System_printf ("%s (%s:%d): testCommonDelHandle returned NULL pointer, paret = %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            if (paret != pa_OK)  {
                System_printf ("%s (%s:%d): testCommonDelHandle return PA error %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
            paTestTxFmtExptStats.classify1.nPackets += 1;
            cmdCount += 1;

        }
    }

    for (i = 0; i < TEST_NUM_L3_HANDLES; i++)  {
        if (l3Handles[i] != 0)  {
            hd = testCommonDelHandle (tf, &l3Handles[i], tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, &reply, &cmdDest, &cmdSize, &paret);
            if (hd == NULL)  {
                System_printf ("%s (%s:%d): testCommonDelHandle returned NULL pointer, paret = %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            if (paret != pa_OK)  {
                System_printf ("%s (%s:%d): testCommonDelHandle return PA error %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
            paTestTxFmtExptStats.classify1.nPackets += 1;
            cmdCount += 1;

        }
    }

    for (i = 0; i < TEST_NUM_L4_HANDLES; i++)  {
        if ((l4Handles[i][0] != 0) || (l4Handles[i][1] != 0))  {
            hd = testCommonDelL4Handles (tf, l4Handles[i], tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, &reply, &cmdDest, &cmdSize, &paret);
            if (hd == NULL)  {
                System_printf ("%s (%s:%d): testCommonDelHandle returned NULL pointer, paret = %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            if (paret != pa_OK)  {
                System_printf ("%s (%s:%d): testCommonDelHandle return PA error %d\n", tfName, __FILE__, __LINE__, paret);
                testStatus = PA_TEST_FAILED;
                continue;
            }

            Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
            paTestTxFmtExptStats.classify2.nPackets += 1;
            cmdCount += 1;

        }
    }


    /* Wait for responses from the PA subsystem */
    for (i = 0; i < cmdCount; i++)  {
    	if (testWaitCmdReply (tf, pat, tf->QGen[Q_CMD_REPLY], TEST_SWINFO0_CMD_ID, __LINE__))  {
    		System_printf ("%s (%s:%d): Missing replies to PA commands (expected in queue %d)\n", tfName, __FILE__, __LINE__, tf->QGen[Q_CMD_REPLY]);	
    		testStatus = PA_TEST_FAILED;
    	}
    }
    		

    /* Restore any data on queues used */
    while (Qmss_getQueueEntryCount(tf->QGen[Q_CMD_RECYCLE]) > 0) {
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
        testCommonRecycleLBDesc (tf, hd);
    }

    while (Qmss_getQueueEntryCount(tf->QGen[Q_MATCH]) > 0)  {
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
        testCommonRecycleLBDesc (tf, hd);
    }

    while (Qmss_getQueueEntryCount(tf->QGen[Q_NFAIL]) > 0)  {
        hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
        testCommonRecycleLBDesc (tf, hd);
    }
            
#ifndef __LINUX_USER_SPACE
    if (t3CheckStats (tf, pat, TRUE) == PA_TEST_FAILED)
		testStatus = PA_TEST_FAILED;
#endif
            
    /* Return result */                
    pat->testStatus = testStatus;
    
    /* Return */
    Task_exit();
}            

        
static void testHandlePaError (tFramework_t *tf, paTest_t *pat, paReturn_t paret, paHandleL2L3_t *l2Handles, paHandleL2L3_t *l3Handles, paHandleL4_t *l4Handles, Cppi_HostDesc  *hd, int32_t line)
{
	if (paret == pa_OK)
		return;
		
	System_printf ("%s (%s:%d): Error - PA LLD returned error code %d\n", tfName, __FILE__, line, paret); 
	testCommonRecycleLBDesc (tf, hd);
	testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, PA_TEST_FAILED);  /* no return */
}      

#ifdef __LINUX_USER_SPACE
void* paTestTxFmtRt (void *args)
{
 	tFramework_t  *tf  = ((paTestArgs_t *)args)->tf;
 	paTest_t      *pat = ((paTestArgs_t *)args)->pat;
#else
void paTestTxFmtRt (UArg a0, UArg a1)
{
    tFramework_t   *tf  = (tFramework_t *)a0;
    paTest_t       *pat = (paTest_t *)a1;
#endif
    Cppi_HostDesc  *hd;
    paReturn_t      paret;

    paTestStatus_t  testStatus = PA_TEST_PASSED;

    int32_t     i;
    int32_t     cmdDest;
    uint16_t  cmdSize;


    paHandleL2L3_t  l2Handles[TEST_NUM_L2_HANDLES] = { 0 };
    paHandleL2L3_t  l3Handles[TEST_NUM_L3_HANDLES] = { 0 /*,  0  */};
    paHandleL4_t    l4Handles[TEST_NUM_L4_HANDLES] = {  {0,0} /*,  {0,0} */ };


    paRouteInfo_t  macMatchRoute =  {  pa_DEST_CONTINUE_PARSE_LUT1, /* Dest - keep parsing    */
                                       0,                           /* Flow ID                */
                                       0,                           /* Queue                  */
                                       -1,                          /* Multi route disabled   */
                                       0,                           /* SW Info 0              */
                                       0,                           /* SW Info 1              */       
                                       0,                           /* customType : pa_CUSTOM_TYPE_NONE  */         
                                       0,                           /* customIndex: not used  */     
                                       0,                           /* pkyType: for SRIO only */    
                                       NULL};                       /* No commands            */

    paRouteInfo_t  ipMatchRoute  =  {  pa_DEST_CONTINUE_PARSE_LUT2, /* Dest - keep parsing    */
                                       0,                           /* Flow ID                */
                                       0,                           /* Queue                  */
                                       -1,                          /* Multi route disabled   */
                                       0,                           /* SW Info 0              */
                                       0,                           /* SW Info 1              */       
                                       0,                           /* customType : pa_CUSTOM_TYPE_NONE  */         
                                       0,                           /* customIndex: not used  */     
                                       0,                           /* pkyType: for SRIO only */    
                                       NULL};                       /* No commands            */


    paRouteInfo_t  udpMatchRoute =  {  pa_DEST_HOST,                /* Dest - send to host    */
                                       0,                           /* Flow ID                */
                                       0,                           /* Queue, filled in below */
                                       -1,                          /* Multi route disabled   */
                                       0,                           /* SW Info 0              */
                                       0,                           /* SW Info 1              */       
                                       0,                           /* customType : not used  */         
                                       0,                           /* customIndex: not used  */     
                                       0,                           /* pkyType: for SRIO only */    
                                       NULL};                       /* No commands            */

    paRouteInfo_t  nfailRoute    =  {  pa_DEST_HOST,                /* Dest - send to host    */
                                       0,                           /* Flow ID                */
                                       0,                           /* Queue, filled in below */
                                       -1,                          /* Multi route disabled   */
                                       0,                           /* SW Info 0              */
                                       0,                           /* SW Info 1              */       
                                       0,                           /* customType : not used  */         
                                       0,                           /* customIndex: not used  */     
                                       0,                           /* pkyType: for SRIO only */    
                                       NULL};                       /* No commands            */

    paCmdReply_t cmdReply        =  {  pa_DEST_HOST,                /* Dest - send to host    */
                                       0,                           /* Reply ID (SWInfo 0)    */
                                       0,                           /* Queue                  */
                                       0,  };                       /* Flow ID                */
                                

#ifdef __LINUX_USER_SPACE
    for (i = 0; (i < (sizeof(paT3pkt)/sizeof(paT3pkt[0])) ); i++)  {

        /* Allocate memory for the Packet buffers */
        paT3pkt[i] = (uint8_t *)fw_memAlloc(sizeof(t3pkt), CACHE_LINESZ);
        if(paT3pkt[i] == NULL) {
  	        printf ("%s: memAlloc failed for pkt %d\n", tfName, i);
 		    pat->testStatus = PA_TEST_FAILED;
  	        return (void *)0;
        }
    }
#endif


    /* Runtime initial values */
    udpMatchRoute.queue = (uint16_t) tf->QGen[Q_MATCH];
    nfailRoute.queue    = (uint16_t) tf->QGen[Q_NFAIL];
    cmdReply.queue      = (uint16_t) tf->QGen[Q_CMD_REPLY];


    /* Zero the expected stats. They are updated as the test progresses */
    memset (&paTestTxFmtExptStats, 0, sizeof(paTestTxFmtExptStats));


    /* Make table entries for the MAC/IP and L4 port numbers. The packets that are
     * generated are routed back to the PA to be received. This will verify the
     * receive checksum generation as well */

    /* L2 (MAC) entries */
    for (i = 0; i < TEST_NUM_L2_HANDLES; i++)  {

        cmdReply.replyId = TEST_SWINFO0_CMD_ID;
        cmdReply.queue   = tf->QGen[Q_CMD_REPLY];
        hd = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &testPaEthInfo[i], &macMatchRoute, &nfailRoute,
                               &l2Handles[i], tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2,
                               &cmdReply, &cmdDest, &cmdSize, &paret);
        

        testHandlePaError (tf, pat, paret, l2Handles, l3Handles, l4Handles, hd, __LINE__); /* no return on error */


        /* Send the command to the PA */
        Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
        paTestTxFmtExptStats.classify1.nPackets += 1;

        /* Wait for a PA reply, and recycle the command descriptor/buffer */
        if (testWaitCmdReply (tf, pat, tf->QGen[Q_CMD_REPLY], TEST_SWINFO0_CMD_ID, __LINE__))
            testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, PA_TEST_FAILED);  /* no return */

    }


    /* L3 (IP) entries. In this test all IPs link to MAC index 0 */
    cmdReply.replyId = TEST_SWINFO0_CMD_ID;
    cmdReply.queue   = tf->QGen[Q_CMD_REPLY];

    for (i = 0; i < TEST_NUM_L3_HANDLES; i++)  {

        hd = testCommonAddIp (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &testPaIpInfo[i], &ipMatchRoute, &nfailRoute,
                              &l3Handles[i], l2Handles[0], tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf1,
                              &cmdReply, &cmdDest, &cmdSize, &paret);

        testHandlePaError (tf, pat, paret, l2Handles, l3Handles, l4Handles, hd, __LINE__);  /* no return on error */

        /* Send the command to the PA */
        Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
        paTestTxFmtExptStats.classify1.nPackets += 1;

        /* Wait for a PA reply, and recycle the command descriptor/buffer */
        if (testWaitCmdReply (tf, pat, tf->QGen[Q_CMD_REPLY], TEST_SWINFO0_CMD_ID, __LINE__))
            testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, PA_TEST_FAILED);

    }


    /* L4 (UDP/TCP) entries. There is a one to one link between ports and linked IP addresses */
    for (i = 0; i < TEST_NUM_L4_HANDLES; i++)  {

        hd = testCommonAddPort (tf, pa_LUT2_PORT_SIZE_16, (uint32_t)testPaPortInfo[i], &udpMatchRoute, &(l4Handles[i]), &l3Handles[i],
                                tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf1, &cmdReply, &cmdDest,
                                &cmdSize, &paret);
                               

        testHandlePaError (tf, pat, paret, l2Handles, l3Handles, l4Handles, hd, __LINE__);

        /* Send the command to the PA */
        Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
        paTestTxFmtExptStats.classify2.nPackets += 1;

        /* Wait for a PA reply, and recycle the command descriptor/buffer */
        if (testWaitCmdReply (tf, pat, tf->QGen[Q_CMD_REPLY], TEST_SWINFO0_CMD_ID, __LINE__))
            testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, PA_TEST_FAILED);

    }

     /* 
     * Configure the CRC engine for SCTP CRC-32C checksum 
     * The CRC-engine connected to PDSP2 should be configured since the SCTP is within the
     * inner IP paylaod which is parased and lookup at PDSP2
     */
 	cmdReply.replyId = TEST_SWINFO0_CMD_ID;
    cmdReply.queue   = tf->QGen[Q_CMD_REPLY];
    
    hd = testCommonConfigCrcEngine(tf, 5, &t3CrcCfg,  tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf3,
                                      &cmdReply, &cmdDest, &cmdSize, &paret);

        testHandlePaError (tf, pat, paret, l2Handles, l3Handles, l4Handles, hd, __LINE__);  /* no return on error */
    
 	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
                            
    /* Wait for a PA reply, and recycle the command descriptor/buffer */
        if (testWaitCmdReply (tf, pat, tf->QGen[Q_CMD_REPLY], TEST_SWINFO0_CMD_ID, __LINE__))
        {
		    System_printf ("%s (%s:%d): testCommonConfigCrcEngine failed\n", tfName, __FILE__, __LINE__);
            testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, PA_TEST_FAILED);
        }
    
    /* Pass the IPv4 packet, verify returned packets */
    testStatus = testSendIpv4 (tf, pat); 

#ifndef __LINUX_USER_SPACE
	if (t3CheckStats (tf, pat, TRUE) == PA_TEST_FAILED)
		testStatus = PA_TEST_FAILED;
#endif


    /* Close all handles and exit */
    testCleanup (tf, pat, l2Handles, l3Handles, l4Handles, testStatus);

#ifdef __LINUX_USER_SPACE
    return (void *)0;
#endif

}







