/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "../pautest.h"
#include "test2pkts.h"

#ifdef __LINUX_USER_SPACE
#include "fw_test.h"
#include "fw_mem_allocator.h"
#endif

/* Add/Delete MAC and PDSP routing test
 * This test tests the LLD Pa_addMac and Pa_delHandle functions, as well as the
 * PDSP firmware for routing mac packets with or without VLAN and MPLS. 
 * This test also verifies the MAC router scenario in which the MAC header of 
 * ingress packet is replaced with pre-defined MAC header.
 * This test has the following sub-tests
 *   - Test the LLD for the ability to determine if a new entry invalidates a previous entry
 *   - Test the LLD to enter increasingly strict match requirements (dst, src, vlan, ethertype, mpls)
 *   - Test the LLD when entering more entries than configured for
 *   - Test the LLD and firmware for the ability to configure command set
 *   - Test the firmware for the ability to take a burst of Pa_addMac commands
 *   - Test the firmware for the routing of matches and next step fail routes
 *   - Test the firmware for the ability to detect LLC/SNAP errors
 *   - Test the firmware for the ability to handle inner and outer VLAN tags
 *   - Test the firmware for the ability to break out of too many nested VLAN tags
 *   - Test the firmware for the ability to handle MPLS packets with a single tag
 *   - Test the firmware for the ability to handle MPLS packets with multiple tags
 *   - Test the firmware for the ability to parse unsupported protocol such as ARP
 *   - Test the firmware for the ability to take a burst of data packets
 * 	 - Test the firmware for correct header offset calculation
 *   - Test the LLD/firmware for the ability to delete entries
 *   - Test the firmware for the ability to execute command set with blind patch command
 */

 #define TEST_ARP
 
 static char *tfName = "paTestL2Routing";
 
 /* General purpose queue usage */
 #define Q_CMD_RECYCLE		  0		/* Command descriptors/buffers recycled here after sent to PA */
 #define Q_CMD_REPLY  		  1		/* Replies from PA routed here */
 #define Q_MATCH		  	  2		/* Packets from PA which match a lookup criteria */
 #define Q_NFAIL		      3		/* Packets from PA which matches a mac lookup, but failed an L3 lookup */
 #define Q_PARSE_ERR		  4		/* Packets which resulted in a parse error */
 #define Q_MULTI_0			  5		/* Multi route queue 0 */
 #define Q_MULTI_1			  6	    /* Multi route queue 1 */
 #define Q_MULTI_2			  7		/* Multi route queue 2 */
 #define Q_MULTI_3			  8		/* Multi route queue 3 */
 #define Q_MULTI_4			  9		/* Multi route queue 4 */
 #define Q_MULTI_5			 10		/* Multi route queue 5 */
 #define Q_MULTI_6           11		/* Multi route queue 6 */
 #define Q_MULTI_7			 12		/* Multi route queue 7 */
 
/* The number of PA handles maintained by this test */
#ifdef __LINUX_USER_SPACE
#define T2_NUM_LOCAL_HANDLES	63
#else
#define T2_NUM_LOCAL_HANDLES	64
#endif

/* The total number of buffers with linked descriptors */
#define TOTAL_BUFS   (TF_LINKED_BUF_Q1_NBUFS + TF_LINKED_BUF_Q2_NBUFS + TF_LINKED_BUF_Q3_NBUFS)
 
 /* Commands to the PA are verified through the value in swinfo0.
  * The 16 ms bits are used as verification, the 16 lbs are for local handle id */
#define T2_CMD_SWINFO0_ADD_ID  		    0x11110000  /* Identifies add mac command */
#define T2_CMD_SWINFO0_DEL_ID  		    0x22220000  /* Identifies del mac command */
#define T2_CMD_SWINFO0_STATS_REQ_ID	    0x33330000	/* Identifies the req stats command */
#define T2_CMD_SWINFO0_CMDSET_CFG_ID    0x44440000  /* Identifies the cmd set command */
#define T2_CMD_SWINFO0_PKT_ID		    0x55550000  /* Identifies the packet as a data packet */
 
/* Cycle delay count for PA reply */
#define T2_PA_CMD_REPLY_DELAY   10000

paSysStats_t paTestL2ExpectedStats;  /* Expected stats results */
 
 /* 32 L2 handles are managed. This structure is used to track the handle and
  * the activation state state of the handle */
enum  {
	T2_L2_HANDLE_UNCONFIGURED = 0,
	T2_L2_HANDLE_PENDING_ACK,
	T2_L2_HANDLE_ACTIVE,
	T2_L2_HANDLE_DISABLED
};

 typedef struct t2Handles_s  {
 
  	paHandleL2L3_t  paHandle;     /* The handle returned by the PA LLD */

 	uint32_t			state;		  /* T2_L2_HANDLE_UNCONFIGURED = handle not configured
 								   * T2_L2_HANDLE_PENDING_ACK = handle configured and sent to pa
 								   * T2_L2_HANDLE_ACTIVE = handle creation acknowledged by pa */
 	
 } t2Handles_t;
 
 
 /* Static test configuration - routing of matching packets to the same queue and 
  * distinguished by swinfo0 */
 typedef struct t2EthAndRoute_s  {
 	
 	paEthInfo_t  eth;
 	uint32_t       swinfo0;

 } t2EthAndRoute_t;
 
const t2EthAndRoute_t  t2EthAndSwinfo[] =  {
	
	{ {  {  0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },  	/* Entry 0: Route on dest mac only */
		 {  0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },	/* PA entry 0 */
		    0, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID },
		
	{ {  {  0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },  	/* Entry1: Route on dest/src mac */
		 {  0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },	/* PA entry 1 */
		   0, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID+1 },
		
	{ {  {  0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },	/* Entry 2: route on dest/src mac, vlan */
		 {  0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },	/* PA entry 2 */
		   0x888, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID+2 },
		
	{ {  {  0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },	/* Entry 3: route on dest/src mac, vlan */
		 {  0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },   /* and ethertype = MPLS unicast */
		   0x888, 0x8847, 0, 0}, T2_CMD_SWINFO0_PKT_ID+3 },		/* PA entry 3 */
		
	{ {  {  0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },	/* Entry 4: route on dest/src mac, vlan */
		 {  0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },   /* and ethertype = MPLS multicast */
		   0x888, 0x8848, 0, 0}, T2_CMD_SWINFO0_PKT_ID+4 },		/* PA entry 4 */
		
	{ {  { 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },	/* Entry 5: route on dest/src mac, vlan */
		 { 0x11, 0x22, 0x33, 0x44, 0x55, 0x66 },    /* ethertype = MPLS unicast, and MPLS tag */
		   0x888, 0x8847, 0x012345, 0}, T2_CMD_SWINFO0_PKT_ID+5 },	/* PA entry 5 */
		   
	{ {  { 0x44, 0x44, 0x44, 0x44, 0x44, 0x44 },    /* Entry 6: more specific entry then 7 */
		 { 0x33, 0x33, 0x33, 0x33, 0x33, 0x33 },	/* PA entry 6 */
		   0, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID+6 },
		 	 
	{ {  { 0x88, 0x88, 0x88, 0x88, 0x88, 0x88 },    /* Entry 7 route on source mac only */
		 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* PA entry 7 */
		   0, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID+7 },
		 
	{ {  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* Entry 8 route on vlan only */
		 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* PA entry 8 */
		   0x999, 0, 0, 0}, T2_CMD_SWINFO0_PKT_ID+8 },
		 
	{ {  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* Entry 9 route on ethertype only */
		 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* PA entry 9 */
		   0, 0x999, 0, 0}, T2_CMD_SWINFO0_PKT_ID+9 },
		 
         
#ifndef TEST_ARP         
	{ {  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },    /* Entry 10 route on mpls tag only */
		 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* PA entry 10 */
		   0, 0, 0x11111, 0}, T2_CMD_SWINFO0_PKT_ID+10 },
#else           
	{ {  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },    /* Entry 10 route on ARP only */
		 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },	/* PA entry 10 */
		   0x0, 0x806, 0, 0}, T2_CMD_SWINFO0_PKT_ID+10 },
#endif           
 };
 
/*
 *  Use both pre-determined and system-allocated LUT1 entry index 
 *  Note: The pre-determined LUT1 entry index should be consistent with the system-allocated one.
 *        It is not recommended to mix those two modes in the application.
 */ 
 
const int  t2EthIndex[] =  {
                pa_LUT1_INDEX_NOT_SPECIFIED,    /* entry 0 */
                62,                             /* entry 1 */
                pa_LUT1_INDEX_NOT_SPECIFIED,    /* entry 2 */
                60,                             /* entry 3 */
                59,                             /* entry 4 */
                pa_LUT1_INDEX_NOT_SPECIFIED,    /* entry 5 */
                pa_LUT1_INDEX_NOT_SPECIFIED,    /* entry 6 */
                56,                             /* entry 7 */
                pa_LUT1_INDEX_NOT_SPECIFIED,    /* entry 8 */
                54,                             /* entry 9 */
                40                              /* entry 10 */
};                
 
const t2EthAndRoute_t  t2EthAndSwinfoFail = {
		 { { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },			/* This would steal matches from entry 6 if allowed */
		   { 0x33, 0x33, 0x33, 0x33, 0x33, 0x33 },	  		/* This entry is not passed to PA */
		     0, 0, 0, 0}, 0xDEADDEAD };						/* !!! No PA Entry !!!! */
	
 
/* A single non-const entry is used to test handling entering too many table elements
 * as well as delete and add */
t2EthAndRoute_t  t2VarEthAndRoute =  { {  { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
									      { 0x45, 0x45, 0x45, 0x45, 0x45, 0x00 },
                        			        0, 0, 0, 0},  0 };
/*
 * Command Set to test MAC router
 *
 */         
#define T2_CMDSET_INDEX         20
#define T2_CMDSET_NUM_CMDS      1
 
static uint8_t t2MacRouteHdr[] = 
{
    0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
    0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
    0x08, 0x00
}; 
                                    
static  paPatchInfo_t t2PatchCmd = 
        {
            pa_PATCH_OP_MAC_HDR,    /* ctrlBitfield */  
            sizeof(t2MacRouteHdr),  /* nPatchBytes */
            sizeof(t2MacRouteHdr),  /* totalPatchSize */
            0,                      /* offset */
            t2MacRouteHdr           /* Pointer to the patch data */
        };
      
static paCmdInfo_t t2CmdSet20[] =
{
    
    /* Command 0: Insert Bytes */
    {
        pa_CMD_PATCH_DATA,
        {
            {
                0,                            /* ctrlBitfield */
                0,                            /* nPatchBytes */
                0,                            /* totalPatchSize */
                0,                            /* offset */  
                0                             /* Pointer to the patch data */
            }
        }
    }
    
}; 

static paCmdInfo_t t2CmdSetCmd =
    {
        pa_CMD_CMDSET,
        {
             
            {
                T2_CMDSET_INDEX               /* Command set index */
            }                   
        }    
    };
  
                                            
									        
/* Prototype required due to circular function calling */									        
paTestStatus_t t2CheckStats (tFramework_t *tf, paTest_t *pat, Bool clear, t2Handles_t *l2Handles);			


Cppi_HostDesc *formDataPacket (tFramework_t *tf, paTest_t *pat, int32_t pktIdx, uint8_t *expectedPktCount)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	
	/* Pop a descriptor off of the free queue list, format the return queue, and attach the packet */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QfreeDesc)) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Failed to pop a descriptor off the free descriptor Q (%d)\n", tfName, __FILE__, __LINE__, tf->QfreeDesc);
		return (NULL);
	}
	
	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = tf->QfreeDesc;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Make sure there is no control info.  */
  	Cppi_setPSLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, 0);
  	
  	/* Attach the data and set the length */
  	Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)utilgAddr((uint32_t)(t2PktTestInfo[pktIdx].pkt)), t2PktTestInfo[pktIdx].pktLen);
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, t2PktTestInfo[pktIdx].pktLen);
  	
  	return (hd);
}
  
								        
/* Recycle delete commands and command recycles  */
int32_t t2StateDel (tFramework_t *tf, paTest_t *pat, t2Handles_t *l2Handles)
{
	Cppi_HostDesc    *hd;
	paEntryHandle_t   reth;
	paReturn_t        paret;
	int32_t				  htype;
	int32_t  			  cmdDest;
	int32_t 			  i;
		
	/* Don't send the command until half of the rx buffers are available. The command replies will always be
	 * sourced from QLinkedBuf1 and arrive in Q_CMD_REPLY. Delay the command until there are enough buffers available
	 * to send and leave a 50% overhead.*/
	for (i = 0; (i < 100) && (Qmss_getQueueEntryCount (tf->QLinkedBuf1) < (TF_LINKED_BUF_Q1_NBUFS >> 1)); i++)  {		
		if ((Qmss_getQueueEntryCount (tf->QGen[Q_CMD_REPLY]) + Qmss_getQueueEntryCount(tf->QLinkedBuf1)) >= TF_LINKED_BUF_Q1_NBUFS)
		  break;
	}
			
	while (Qmss_getQueueEntryCount (tf->QGen[Q_CMD_REPLY]) > 0)  {
				
		/* Recycle the command descriptor/buffer */
		hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_REPLY])) & ~15);
		if ((hd->softwareInfo0 & 0xffff0000u) != T2_CMD_SWINFO0_DEL_ID)  {
			System_printf ("%s (%s:%d): Found packet in PA command reply queue without delete ID (found 0x%08x)", tfName, __FILE__, __LINE__, hd->softwareInfo0);
			return (-1);
		}
				
		/* Send the reply back to PA to let the driver know the command has been accepted */
		paret = Pa_forwardResult (tf->passHandle, (void *)hd->buffPtr, &reth, &htype, &cmdDest);
				
		if (paret != pa_OK)  {
			System_printf ("%s (%s:%d): paForwardResult returned error %d in response to pa_DelHandle reply from PA for handle #%d\n", tfName, __FILE__, __LINE__, hd->softwareInfo0 & 0xffff);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			return (-1);
		}
			
		l2Handles[hd->softwareInfo0 & 0xffff].state = T2_L2_HANDLE_DISABLED;
				
		if (testCommonRecycleLBDesc (tf, hd))  {
			System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
			return (-1);
		}
			
	}
		
		utilCycleDelay (100);
	
	
				
	/* The command recycle descriptor/buffer */
	while (Qmss_getQueueEntryCount (tf->QGen[Q_CMD_RECYCLE]) > 0)  {
		hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
		if (testCommonRecycleLBDesc (tf, hd))  {
			System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
			return (-1);
		}
	}
		
	
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Timeout waiting for free descriptor/buffer queue to fill to halfway point\n", tfName, __FILE__, __LINE__);
		return (-1);
	}
	
	return (0);
		
}
														        									        									
	
static void paTestL2RecoverAndExit (tFramework_t *tf, paTest_t *pat, t2Handles_t *l2Handles, paTestStatus_t status, Bool doStats)
{
	Cppi_HostDesc  *hd;
	paReturn_t      paret;
	int32_t  			cmdDest;
 	uint16_t			cmdSize;
	int32_t 			i, j, m;
	volatile int32_t    mdebugWait = 0;
	
	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
	cmdReply.queue   = (uint16_t) tf->QGen[Q_CMD_REPLY];

	
	/* Delete all the handles */
	for (i = 0; i < T2_NUM_LOCAL_HANDLES; i++)  {
		
		cmdReply.replyId = T2_CMD_SWINFO0_DEL_ID + i;
		hd = testCommonDelHandle (tf, &l2Handles[i].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, &cmdReply, &cmdDest, &cmdSize, &paret);
		
		if (paret != pa_OK)  {
			System_printf ("%s (%s:%d): PA LLD returned error code %d on handle deletion\n", tfName, __FILE__, __LINE__, paret);
			status = PA_TEST_FAILED;
			break;
		}
		
		if (hd == NULL)  {
			System_printf ("%s (%s:%d): No descriptor available for del handle command\n", tfName, __FILE__, __LINE__);
			status = PA_TEST_FAILED;
			break;
		}
		
		/* Wait to send the command until half of the rx buffers are available */
		if (t2StateDel (tf, pat, l2Handles))  {
			status = PA_TEST_FAILED;
			break;
		}
		
		/* mdebugHaltPdsp (0); */
		/* Send the command to the PA */
		Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
		paTestL2ExpectedStats.classify1.nPackets += 1;
		
		while (mdebugWait);
	}
	
	/* Give some time for remaining commands to complete */
	for (i = 0; i < 100; i++)  {
		
		for (j = m = 0; j < T2_NUM_LOCAL_HANDLES; j++)
			if (l2Handles[j].state != T2_L2_HANDLE_DISABLED)
				m += 1;
				
		if (m)  {
			if (t2StateDel (tf, pat, l2Handles))  {
				status = PA_TEST_FAILED;
				break;
			}
			utilCycleDelay (100);
		} else
			break;
		
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Unable to delete all L2 handles. %d remain undeleted\n", tfName, __FILE__, __LINE__, m);
		status = PA_TEST_FAILED;
	}
	
#ifndef __LINUX_USER_SPACE
	/* Verify the stats are as expected */
	if (t2CheckStats (tf, pat, TRUE, l2Handles) == PA_TEST_FAILED)
	  status = PA_TEST_FAILED;
#endif
	
	/* Test result */
	pat->testStatus = status;
	
	/* Return */
	Task_exit();
}		
				
/* Look for command replies from PA */					     
void t2L2CmdRep (tFramework_t *tf, paTest_t *pat, t2Handles_t *localHandles, int *cmdReply)
{
	Cppi_HostDesc  *hd;
	uint32_t         *swInfo0;
	uint32_t 			swInfoCmd;
	uint32_t			lid;
	paReturn_t      paret;
	paEntryHandle_t reth;
	int32_t				htype;
	int32_t				cmdDest;
	
	while (Qmss_getQueueEntryCount ((tf->QGen)[Q_CMD_REPLY]) > 0)  {
		
		hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop ((tf->QGen[Q_CMD_REPLY]))) & ~15);
		if (Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **) &swInfo0) == CPPI_EPIB_NOT_PRESENT)  {
			System_printf ("%s (%s:%d): Found descriptor in PA command reply queue without EIPB present, failing\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		swInfoCmd = (*swInfo0 & 0xffff0000u); 
        
        /* Add General Command Processing */
        /* Command Set Command */
        if( swInfoCmd == T2_CMD_SWINFO0_CMDSET_CFG_ID)
        {
            
		    /* Recycle the descriptor and buffer */
		    if (testCommonRecycleLBDesc (tf, hd))  {
			    System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
			    paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		    }
            *cmdReply = TRUE;
            break;
        }
        
		/* Verify expected value in swinfo0 16 msbs */
		if ( (swInfoCmd != T2_CMD_SWINFO0_ADD_ID) && (swInfoCmd != T2_CMD_SWINFO0_DEL_ID) ) {
			System_printf ("%s (%s:%d): Found descriptor in PA command reply queue without command reply swinfo0\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
		/* Extract the local instance value */
		lid = *swInfo0 & 0xffffu;
		if (lid >= T2_NUM_LOCAL_HANDLES)  {
			System_printf ("%s (%s:%d): Received PA command reply for out of range local handle %d (max %d)\n", tfName, __FILE__, __LINE__, lid, T2_NUM_LOCAL_HANDLES);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
		/* Send the reply back to PA to let the driver know the command has been accepted */
		paret = Pa_forwardResult (tf->passHandle, (void *)hd->buffPtr, &reth, &htype, &cmdDest);
		if (paret != pa_OK)  {
			System_printf ("%s (%s:%d): paForwardResult returned error %d in response to paAddMac reply from PA\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
		/* Make sure the handle returned by PA matches the local copy */
		if (localHandles[lid].paHandle != reth.l2l3Handle)  {
			System_printf ("%s (%s:%d): paForwardResult returned handle (0x%08x) that did match internal table value (0x%08x)\n", tfName, __FILE__, __LINE__, (uint32_t)(localHandles[lid].paHandle), (uint32_t) reth.l2l3Handle);
			testCommonRecycleLBDesc (tf, hd);  /* Ignore return code */
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
		/* Recycle the descriptor and buffer */
		if (testCommonRecycleLBDesc (tf, hd))  {
			System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
		if (swInfoCmd == T2_CMD_SWINFO0_ADD_ID)
			localHandles[lid].state = T2_L2_HANDLE_ACTIVE;
		else
			localHandles[lid].state = T2_L2_HANDLE_UNCONFIGURED;
	}
	
	while (Qmss_getQueueEntryCount ((tf->QGen)[Q_CMD_RECYCLE]) > 0)  {
		
		/* Recycle the command descriptor/buffer */
		hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop ((tf->QGen[Q_CMD_RECYCLE]))) & ~15);
		if (testCommonRecycleLBDesc (tf, hd))  {
			System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
			paTestL2RecoverAndExit (tf, pat, localHandles, PA_TEST_FAILED, TRUE);  /* No Return */
		}
		
	}
			
		
}
								     
									     
								     
									     
void paL2HandleError (tFramework_t *tf, paTest_t *pat, t2Handles_t *l2Handles, paReturn_t paret, Cppi_HostDesc *hd)
{	 	  
    /* Check paret before the descriptor. If paret indicates failure the descriptor will be NULL */                    
 	if (paret != pa_OK)  {
 		System_printf ("%s (%s:%d): testCommonAddMac failed, PA LLD error code = %d\n", tfName, __FILE__, __LINE__, paret);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No return */
 	}
 	
 	if (hd == NULL)  {
 		System_printf ("%s (%s:%d): testCommonAddMac failed due to unavailable free linked buffer descriptor\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No return */
 	}
}


/* Check the stats */

paTestStatus_t t2CheckStats (tFramework_t *tf, paTest_t *pat, Bool clear, t2Handles_t *l2Handles)
{ 	
	Cppi_HostDesc  *hd;
	uint8_t		   *bp;
	paSysStats_t     *paStats;
 	uint32_t			blen;
	int32_t             i;
	paTestStatus_t  status;

	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
 	cmdReply.queue   = (uint16_t) tf->QGen[Q_CMD_REPLY];
 							   
 	
 	/* Check the PA stats to make sure they are set as expected */
 	cmdReply.replyId = T2_CMD_SWINFO0_STATS_REQ_ID;
 	if (testCommonRequestPaStats (tfName, tf, clear, tf->QLinkedBuf1, tf->QGen[Q_CMD_RECYCLE],  &cmdReply))  {
 		System_printf ("%s (%s:%d): testCommonRequestPaStats failed\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, FALSE);  /* No Return */
 	}
 	
 	/* Wait for the stats reply */
	for (i = 0; i < 100; i++)  {
		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
		if (Qmss_getQueueEntryCount (tf->QGen[Q_CMD_REPLY]) > 0)
			break;
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Did not find response from PA to stats request command\n", tfName, __FILE__, __LINE__);
		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, FALSE);  /* No Return */
	}
	
	/* Recycle the descriptor/buffer returned from the stats request */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_RECYCLE])) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Did not find returned descriptor/buffer from stats request\n", tfName, __FILE__, __LINE__);
		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, FALSE);  /* No Return */
	}
	
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats request\n", tfName, __FILE__, __LINE__);
		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, FALSE);  /* No Return */
	}
 		
 	/* Format the stats response and compare to the expected results */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_CMD_REPLY])) & ~15);
	Cppi_getData (Cppi_DescType_HOST, (Cppi_Desc *)hd, &bp, &blen);
	paStats = (paSysStats_t *)Pa_formatStatsReply (tf->passHandle, (paCmd_t)bp);
	  
    if (testCommonCompareStats (tfName, (paSysStats_t *)&paTestL2ExpectedStats, paStats))
    	status = PA_TEST_FAILED;
    else
    	status = PA_TEST_PASSED;   
    	  
     /* Recycle the descriptor and associated buffer back to queue from which it came */
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats response\n", tfName, __FILE__, __LINE__);
		status = PA_TEST_FAILED;
	}
	
	return (status);
}
	
	
/* Search the receive data packet queue for received data packets. Remain in 
 * this function until all buffers are restored to their respective queues */
int32_t t2ReceiveDataPkts (tFramework_t *tf, paTest_t *pat, uint8_t *actualPktCount)
{
	Cppi_HostDesc    *hd;
	uint32_t		     *swInfo;
	pktTestInfo_t    *tinfo;
	pasahoLongInfo_t *pinfo;
	uint32_t		      infoLen;
	int32_t               i, j;
	uint32_t		      chan;
	
	for (i = 0; i < 100; i++)  {
		
        utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
		while (Qmss_getQueueEntryCount (tf->QGen[Q_MATCH]) > 0) {
			
			hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QGen[Q_MATCH])) & ~15);
			if (hd == NULL)  {
				System_printf ("%s (%s:%d): Popped a NULL descriptor off of match queue\n", tfName, __FILE__, __LINE__);
				return (-1);
			}
			
			/* Verify swInfo0 for packet match and packet ID number */
			Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swInfo);
			
			if ((*swInfo & 0xffff0000) != T2_CMD_SWINFO0_PKT_ID)  {
				System_printf ("%s (%s:%d): Found a packet in the receive packet queue but with incorrect swinfo0 Id: 0x%08x\n",
							   tfName, __FILE__, __LINE__, *swInfo);
			 	testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			
			chan = *swInfo & 0xffff;
			if (chan < T2_NUM_LOCAL_HANDLES)
			  actualPktCount[chan] += 1;
			  
			/* locate the associated test information based on the channel value */
			for (j = 0, tinfo = NULL; j < sizeof(t2PktTestInfo) / sizeof(pktTestInfo_t); j++)  {
				if (t2PktTestInfo[j].idx == chan)  {
					tinfo = &t2PktTestInfo[j];
					break;
				}
			}
			
			if (tinfo == NULL)  {
				System_printf ("%s (%s:%d): Found a packet in the receive packet queue for channel %d, but found no matching packet info\n",
								tfName, __FILE__, __LINE__, chan);
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
				
			  
			/* Verify the parse information is correct */
			if (Cppi_getPSData (Cppi_DescType_HOST, Cppi_PSLoc_PS_IN_DESC, (Cppi_Desc *)hd, (uint8_t **)&pinfo, &infoLen) != CPPI_SOK)  {
				System_printf ("%s (%s:%d): Error getting control info from received data packet\n", tfName, __FILE__, __LINE__);
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			
			if (testCommonComparePktInfo (tfName, tinfo->info, pinfo))  {
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
            
            /* Verify the replaced MAC header */
            if(memcmp((void*)hd->buffPtr, t2MacRouteHdr, sizeof(t2MacRouteHdr)))
            {
				System_printf ("%s (%s:%d): MAC header does not match!\n", tfName, __FILE__, __LINE__);
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
            }
			
			/* Return the descriptor/buffer */
			testCommonRecycleLBDesc (tf, hd);								
			
		}
		
		if ( (Qmss_getQueueEntryCount(TF_LINKED_BUF_Q1) == TF_LINKED_BUF_Q1_NBUFS)  &&
		     (Qmss_getQueueEntryCount(TF_LINKED_BUF_Q2) == TF_LINKED_BUF_Q2_NBUFS)  &&
		     (Qmss_getQueueEntryCount(TF_LINKED_BUF_Q3) == TF_LINKED_BUF_Q3_NBUFS)  )
			break;
		
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Error - unable to recover all descriptors with associated buffers\n", tfName, __FILE__, __LINE__);
        System_flush();
		return (-1);
	}
	
	return (0);

}
	
	
			
			
 
 
#ifdef __LINUX_USER_SPACE
void* paTestL2Routing (void *args)
{
 	tFramework_t  *tf  = ((paTestArgs_t *)args)->tf;
 	paTest_t      *pat = ((paTestArgs_t *)args)->pat;
#else
void paTestL2Routing (UArg a0, UArg a1)
{
 	tFramework_t   *tf  = (tFramework_t *)a0;
 	paTest_t       *pat = (paTest_t *)a1;
#endif
 	Cppi_HostDesc  *hd[8];
 	paReturn_t      paret;
 	t2Handles_t     l2Handles[T2_NUM_LOCAL_HANDLES];
 	paHandleL2L3_t  fHandle;
 	int32_t				i, j, k, l;
 	int32_t				state;
 	int32_t				count;
 	int32_t  			cmdDest;
 	Bool			halt;
 	uint16_t			cmdSize;
 	paTestStatus_t  testStatus = PA_TEST_PASSED;
 	uint8_t			expectedPktCount[T2_NUM_LOCAL_HANDLES];
 	uint8_t			actualPktCount[T2_NUM_LOCAL_HANDLES];
    int32_t             fCmdReply;

 	
 	volatile int32_t mdebugWait = 1;
 	
 	paRouteInfo_t   matchRoute = {  pa_DEST_HOST,		/* Dest */
 								    0,					/* Flow ID */
 								    0,					/* queue */
 								    -1,					/* Multi route */
 								    0,					/* sw Info 0 */
                                    0,                  /* sw Info 1 */       
                                    0,                  /* customType : not used */         
                                    0,                  /* customIndex: not used */     
                                    0,                  /* pkyType: for SRIO only */    
                                    &t2CmdSetCmd};      /* command set command */
 								    
 	paRouteInfo_t   nfailRoute = {  pa_DEST_HOST,		/* Dest */
 									0,					/* Flow ID */
 									0,					/* queue */
 									-1,					/* Multi route */
 									0,					/* sw Info 0 */
                                    0,                  /* sw Info 1 */       
                                    0,                  /* customType : not used */         
                                    0,                  /* customIndex: not used */     
                                    0,                  /* pkyType: for SRIO only */    
                                    NULL};              /* No commands */
 									
 	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
 		
#ifdef __LINUX_USER_SPACE
    for (i = 0; (i < (sizeof(t2PktTestInfo) / sizeof(pktTestInfo_t))); i++)  {
        int pktSz;
        uint8_t *pkt; 

        switch (i) {
          case 0:
              pktSz = sizeof(pkt2);
              pkt   = (uint8_t *)pkt2;
              break;

          case 1:
              pktSz = sizeof(pkt5);
              pkt   = (uint8_t *)pkt5;
              break;

          case 2:
              pktSz = sizeof(pkt10);
              pkt   = (uint8_t *)pkt10;
              break;
        }

        /* Allocate memory for the Packet buffers */
        t2PktTestInfo[i].pkt = (uint8_t *)fw_memAlloc(pktSz, CACHE_LINESZ);
        if(t2PktTestInfo[i].pkt == NULL) {
  	        printf ("%s: memAlloc failed for pkt %d\n", tfName, i);
 		    pat->testStatus = PA_TEST_FAILED;
  	        return (void *)0;
        }
        memcpy(t2PktTestInfo[i].pkt, pkt, pktSz);
    }
#endif

    /* Runtime initial values */
    matchRoute.queue = (uint16_t) tf->QGen[Q_MATCH];
    nfailRoute.queue = (uint16_t) tf->QGen[Q_NFAIL];
    cmdReply.queue   = (uint16_t) tf->QGen[Q_CMD_REPLY];
	cmdReply.flowId  = tf->tfFlowNum;
    
    t2CmdSet20[0].params.patch  = t2PatchCmd;  
    t2CmdSetCmd.params.cmdSet.index = T2_CMDSET_INDEX;  
    
    /* Zero out the l2Handle array and packet counts */
    memset (l2Handles, 0, sizeof(l2Handles));
    memset (expectedPktCount, 0, sizeof(expectedPktCount));
    memset (actualPktCount, 0, sizeof(actualPktCount));
    
    /* Zero out the expected stats. The stats will be updated as packets are sent into PA */
    memset (&paTestL2ExpectedStats, 0, sizeof(paTestL2ExpectedStats));
    
    /* Issue the command set command */
    cmdReply.replyId  = T2_CMD_SWINFO0_CMDSET_CFG_ID;
 	hd[0] = testCommonConfigCmdSet (tf, T2_CMDSET_INDEX, T2_CMDSET_NUM_CMDS, t2CmdSet20,  
 	                                tf->QGen[Q_CMD_RECYCLE],tf->QLinkedBuf2, 
 	                                &cmdReply, &cmdDest, &cmdSize, &paret);
                                    
    /* Check paret before the descriptor. If paret indicates failure the descriptor will be NULL */                    
 	if (paret != pa_OK)  {
 		System_printf ("%s (%s:%d): testCommonConfigCmdSet failed, PA LLD error code = %d\n", tfName, __FILE__, __LINE__, paret);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No return */
 	}
 	
 	if (hd[0] == NULL)  {
 		System_printf ("%s (%s:%d): testCommonConfigCmdSet failed due to unavailable free linked buffer descriptor\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No return */
 	}
                                         
    /* Send command */
 	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[0], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
    
 	/* Wait for a PA reply */
    fCmdReply = FALSE;
 	for (i = 0; i < 100; i++)  {
 		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 		t2L2CmdRep (tf, pat, l2Handles, &fCmdReply);
 		if (fCmdReply)
 			break;
 	}
 	
 	if (i == 100)  {
 		System_printf ("%s (%s:%d): Reply to Pa_configCmdSet not found\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* no return */
 	}
 								  
 	/* Initialize the first entry in the table */
 	cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + 0;  /* T2_CMD_SWINFO0_ADD_ID identifies command, 16 LS bits identify the local handle number */
 	cmdReply.queue = tf->QGen[Q_CMD_REPLY];
 	matchRoute.swInfo0 = nfailRoute.swInfo0 = t2EthAndSwinfo[0].swinfo0;
 	hd[0] = testCommonAddMac (tf, t2EthIndex[0], (paEthInfo_t *)&t2EthAndSwinfo[0].eth, &matchRoute, &nfailRoute,
 	                        &l2Handles[0].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, 
 	                        &cmdReply, &cmdDest, &cmdSize, &paret);
 	                        
 	                        
 	paL2HandleError (tf, pat, l2Handles, paret, hd[0]);  /* Will not return on error */
 	
 	
 	/* Send the command to PA. This will result in 1 packet in classify1 */
 	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[0], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 	l2Handles[0].state = T2_L2_HANDLE_PENDING_ACK;
 	paTestL2ExpectedStats.classify1.nPackets += 1;
 	 	
 	/* Wait for a PA reply */
 	for (i = 0; i < 100; i++)  {
 		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 		t2L2CmdRep (tf, pat, l2Handles, NULL);
 		if (l2Handles[0].state == T2_L2_HANDLE_ACTIVE)
 			break;
 	}
 	
 	if (i == 100)  {
 		System_printf ("%s (%s:%d): Reply to paAddMac not found\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* no return */
 	}
 	
 	
 	/* Add the next 6 entries in a burst. Take the memory from linked buffer descriptor area two
 	 * since the responses will come from area one. These are entries 1-6 in the local handle table */
 	for (i = 0; i < 6; i++)  {
 		cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + i + 1;
 		matchRoute.swInfo0 = nfailRoute.swInfo0 = t2EthAndSwinfo[i+1].swinfo0;
 		hd[i] = testCommonAddMac (tf, t2EthIndex[i+1], (paEthInfo_t *)&t2EthAndSwinfo[i+1].eth, &matchRoute, &nfailRoute, 
 	                        &l2Handles[i+1].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, 
 	                        &cmdReply, &cmdDest, &cmdSize, &paret);
 	    paL2HandleError (tf, pat, l2Handles, paret, hd[i]);  /* Will not return on error */
 	}
 	
 	
 	/* Send all the commands at once to test the ability to handle back to back commands */
 	for (i = 0; i < 6; i++)  {
 		/* if (mdebugWait) mdebugHaltPdsp(0); */ 
 		
 		Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[i], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 		
 		/* while (mdebugWait); */
 		
 		
 		l2Handles[i+1].state = T2_L2_HANDLE_PENDING_ACK;
 		paTestL2ExpectedStats.classify1.nPackets += 1;
 		
 	}
 	
 	
 	/* Wait for the PA to generate all the responses */
 	for (i = 0; i < 100; i++)  {
 		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 		t2L2CmdRep (tf, pat, l2Handles, NULL);
 		
 		state = 1;
 		for (j = 1; j < 7; j++)  {
 			if (l2Handles[i].state == T2_L2_HANDLE_PENDING_ACK)
 				state = 0;
 		}
 		
 		if (state == 1)
 			break;
 	}
 	
 	if ((i == 100) && (state == 0))  {
 		System_printf ("%s: (%s:%d): Burst of 5 addMac commands did not result in 5 acks from PA\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);
 	}

 	
 	/* The next entry should result in the PA generating an error */
 	cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + 7;  /* T2_CMD_SWINFO0_ADD_ID identfies command, 16 LS bits identify the local handle number */
 	matchRoute.swInfo0 = nfailRoute.swInfo0 = t2EthAndSwinfoFail.swinfo0;
 	hd[0] = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, (paEthInfo_t *)&t2EthAndSwinfoFail.eth, &matchRoute, &nfailRoute,
 	                        &l2Handles[7].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2,
 	                        &cmdReply, &cmdDest, &cmdSize, &paret);
 	                        
 	                         	/* If the descriptor came back non-null, restore it and return it */
 	if (hd[0] != NULL)  {
 		hd[0]->buffLen = hd[0]->origBufferLen;
 		Qmss_queuePush (tf->QLinkedBuf2, (Ptr)hd[0], hd[0]->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
 	                  
 	/* The test won't exit on this failure since no commands have been sent to PA */                  
 	if (paret != pa_INVALID_TABLE_MORE_SPECIFIC_ENTRY_PRESENT)  {
 		System_printf ("%s (%s:%d): function paAddMac did not detect an invalid entry order\n", tfName, __FILE__, __LINE__);
 		testStatus = PA_TEST_FAILED;
 	}
 		
 		
    /* Rapid fire the next 4 entries from the const table to the PA */
    /* Add the next 4 entries in a burst. Take the memory from linked buffer descriptor area two
 	 * since the responses will come from area one. These are entries 1-6 in the local handle table */
 	for (i = 0; i < 4; i++)  {
 		cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + i + 7;
 		matchRoute.swInfo0 = nfailRoute.swInfo0 = t2EthAndSwinfo[i+7].swinfo0;
 		hd[i] = testCommonAddMac (tf, t2EthIndex[i+7], (paEthInfo_t *)&t2EthAndSwinfo[i+7].eth, &matchRoute, &nfailRoute, 
 	                        &l2Handles[i+7].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2,
 	                        &cmdReply, &cmdDest, &cmdSize, &paret);
 	    paL2HandleError (tf, pat, l2Handles, paret, hd[i]);  /* Will not return on error */
 	}
 	
 	/* Send all the commands at once to test the ability to handle back to back commands */
 	for (i = 0; i < 4; i++)  {
 		Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[i], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 		l2Handles[i+7].state = T2_L2_HANDLE_PENDING_ACK;
 		paTestL2ExpectedStats.classify1.nPackets += 1;
 	}
 	
 	/* Wait for the PA to generate all the responses */
 	for (i = 0; i < 100; i++)  {
 		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 		t2L2CmdRep (tf, pat, l2Handles, NULL);
 		
 		state = 1;
 		for (j = 7; j < 11; j++)  {
 			if (l2Handles[i].state == T2_L2_HANDLE_PENDING_ACK)
 				state = 0;
 		}
 		
 		if (state == 1)
 			break;
 	}
 	
 	 if (i == 100)  {
 		System_printf ("%s (%s:%d): Failed to find responses from PA\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No Return */
 	}
 	
 	
 	/* 11 Entries into the table have been made. Make an additional 52 entries in 13 batches of 4.
 	 * These entries do not test any particular routing, simply table addition and overflow */
 	for (i = k = 0; i < 13; i++, k += 4)  {
 		for (j = 0; j < 4; j++)  {
 			cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + 11 + k + j;
 			t2VarEthAndRoute.eth.dst[5] = k + j;
 			matchRoute.swInfo0 = nfailRoute.swInfo0 = 0x5555000b + k + j + 11;
 			hd[j] = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &t2VarEthAndRoute.eth, &matchRoute, &nfailRoute, 
 	                       		 	  &l2Handles[k+j+11].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2,
 	                        		  &cmdReply, &cmdDest, &cmdSize, &paret);
 	    	paL2HandleError (tf, pat, l2Handles, paret, hd[j]);  /* Will not return on error */
 		}
 		
 		/* Send all the commands at once to test the ability to handle back to back commands */
 		for (j = 0; j < 4; j++)  {
 			Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[j], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 			l2Handles[k+j+11].state = T2_L2_HANDLE_PENDING_ACK;
 			paTestL2ExpectedStats.classify1.nPackets += 1;
 		}
 
 		/* Wait for the PA to generate all the responses */
 		for (l = 0; l < 100; l++)  {
 			utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 			t2L2CmdRep (tf, pat, l2Handles, NULL);
 		
 			state = 1;
 			for (j = 0; j < 4; j++)  {
 				if (l2Handles[11 + k + j].state == T2_L2_HANDLE_PENDING_ACK)
 					state = 0;
 			}
 		
 			if (state == 1)
 				break;
 		}
 		
 		if (l == 100)  {
 			System_printf ("%s (%s:%d): Failed to find responses from PA\n", tfName, __FILE__, __LINE__);
 			paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No Return */
 		}
 	}
 	
#ifndef __LINUX_USER_SPACE
 	/* There is one place left in the table. Try to make two more entries. The first one should succeed
 	 * but the next one should be rejected by the PA lld */
 	cmdReply.replyId = T2_CMD_SWINFO0_ADD_ID + 11 + k;
 	t2VarEthAndRoute.eth.dst[5] = k;
 	matchRoute.swInfo0 = nfailRoute.swInfo0 = T2_CMD_SWINFO0_ADD_ID + k + 11;
 	hd[0] = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &t2VarEthAndRoute.eth, &matchRoute, &nfailRoute, 
 	                    	  &l2Handles[k+11].paHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, 
 	                          &cmdReply, &cmdDest, &cmdSize, &paret);
 	paL2HandleError (tf, pat, l2Handles, paret, hd[j]);  /* Will not return on error */
 	
 	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[0], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 	l2Handles[k+11].state = T2_L2_HANDLE_PENDING_ACK;
 	paTestL2ExpectedStats.classify1.nPackets += 1;
 
 	
 	/* Wait for the PA to generate the response */
 	for (l = 0; l < 100; l++)  {
 		utilCycleDelay (T2_PA_CMD_REPLY_DELAY);
 		t2L2CmdRep (tf, pat, l2Handles, NULL);

 		if (l2Handles[11+k].state == T2_L2_HANDLE_ACTIVE)
 			break;
 	}
 	
 	if (l == 100)  {
 		System_printf ("%s (%s:%d): Failed to find responses from PA\n", tfName, __FILE__, __LINE__);
 		paTestL2RecoverAndExit (tf, pat, l2Handles, PA_TEST_FAILED, TRUE);  /* No Return */
	}
 
 	/* The next add mac command should fail */
 	k += 1;
 	t2VarEthAndRoute.eth.dst[5] = k;
 	matchRoute.swInfo0 = nfailRoute.swInfo0 = T2_CMD_SWINFO0_ADD_ID + k;
 	hd[0] = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &t2VarEthAndRoute.eth, &matchRoute, &nfailRoute, 
 	                    	  &fHandle, tf->QGen[Q_CMD_RECYCLE], tf->QLinkedBuf2, 
 	                          &cmdReply, &cmdDest, &cmdSize, &paret);
 		                          
 	/* If the descriptor came back non-null, restore it and return it */
 	if (hd[0] != NULL)  {
 		hd[0]->buffLen = hd[0]->origBufferLen;
 		Qmss_queuePush (tf->QLinkedBuf2, (Ptr)hd[0], hd[0]->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
 	
 	/* The test will continue on failure since the PA sub-system has not been changed */
 	if (paret != pa_HANDLE_TABLE_FULL)  {
 		System_printf ("%s (%s:%d): function paAddMac did not detect a full handle table\n", tfName, __FILE__, __LINE__);
 		testStatus = PA_TEST_FAILED;
 	}
 	
 	/* Check and clear the stats */
 	testStatus = t2CheckStats (tf, pat, TRUE, l2Handles);
	memset (&paTestL2ExpectedStats, 0, sizeof(paTestL2ExpectedStats));
	
	if (testStatus != PA_TEST_PASSED)
		paTestL2RecoverAndExit (tf, pat, l2Handles, testStatus, TRUE);
#endif
		
		
	/* Run packets through the system. the complete set of packets is run through three times. */
	for (j = 0, halt = FALSE; (j < 3) && (halt == FALSE); j++)   {
		
		for (i = 0; (i < sizeof(t2PktTestInfo) / sizeof(pktTestInfo_t)) && (halt == FALSE);  )  {
			
			/* Form up to 8 data packets to send */
			for (k = 0; ((k < 8) && (i < sizeof(t2PktTestInfo) / sizeof(pktTestInfo_t))); k++)  {
				hd[k] = formDataPacket (tf, pat, i, expectedPktCount);
				
				if (hd[k] == NULL)  {
					halt = TRUE;
					break;
				}
				
				/* Inc the count if the packet is passed back to the host */
  				if (t2PktTestInfo[i].idx >= 0)
	  				expectedPktCount[t2PktTestInfo[i].idx] += 1;
		
				/* Increment any expected stats */
				testCommonIncStats (t2PktTestInfo[i].statsMap, &paTestL2ExpectedStats);	
				
				/* Proceed to the next packet */
				i += 1;  				

			}
			
            //mdebugWait = 1;
 		    //if (mdebugWait) mdebugHaltPdsp(0);  
				
			for (l = 0; l < k; l++)
				Qmss_queuePush (tf->QPaTx[0], (Ptr)hd[l], hd[l]->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
                
	        //while (mdebugWait);
                
				
		}
		
		/* Wait for all descriptors associated with received packets to be recycled. The descriptors
		 * had to go somewhere so wait until the total count is accounted for */
		for (l = 0, count = 0; ((l < 100) && (count != TOTAL_BUFS)); l++)
        {
            utilCycleDelay (100);
			count = Qmss_getQueueEntryCount(tf->QGen[Q_MATCH]) + Qmss_getQueueEntryCount(tf->QGen[Q_NFAIL]) + 
			        Qmss_getQueueEntryCount(tf->QGen[Q_PARSE_ERR]) + Qmss_getQueueEntryCount(tf->QLinkedBuf1) + 
			        Qmss_getQueueEntryCount(tf->QLinkedBuf2) + Qmss_getQueueEntryCount(tf->QLinkedBuf3);
        }            
			
		if (l == 100)  {
			System_printf ("%s (%s:%d): Timeout waiting for packets from PA\n", tfName, __FILE__, __LINE__);
			testStatus = PA_TEST_FAILED;
			break;
		}
						
		if (t2ReceiveDataPkts (tf, pat, actualPktCount))
        {
            /* Error Handling */
			System_printf ("%s (%s:%d): t2ReceiveDataPkts timeout %d\n", tfName,
						   __FILE__, __LINE__);
            System_flush();               
			break;
        }    
	}
	
	/* Give a good delay for any more packets to come through, and examine th epackets. */
	//utilCycleDelay (2000);
	t2ReceiveDataPkts (tf, pat, actualPktCount);
	
	
	/* Verify that the expected and actual received packet counts match */
	for (i = 0; i < T2_NUM_LOCAL_HANDLES; i++)  {
		if (expectedPktCount[i] != actualPktCount[i])  {
			System_printf ("%s (%s:%d): Packet count mismatch for entry %d - expected %d, found %d\n", tfName,
						   __FILE__, __LINE__, i, expectedPktCount[i], actualPktCount[i]);
            System_flush();               
			testStatus = PA_TEST_FAILED;
		}
	}
			
	

	/* Clean up and return */
 	paTestL2RecoverAndExit (tf, pat, l2Handles, testStatus, TRUE);
 
#ifdef __LINUX_USER_SPACE
    return (void *)0;
#endif
}
 		
 
 		 
 
 	
 	


