/**  
 * @file multicore_example.h
 *
 * @brief 
 *  Holds all the constants and API definitions required by the example
 *  application to run.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009-2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef _MULTICORE_EXAMPLE_H_
#define _MULTICORE_EXAMPLE_H_

/* C Standard library Include */
#include <stdlib.h>
#include <string.h>

/* XDC types include */
#include <xdc/std.h>

/* XDC/BIOS includes */
#include <xdc/runtime/IHeap.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Memory.h>
#include <xdc/runtime/Error.h>

#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/hal/Hwi.h>
#include <ti/sysbios/knl/task.h>
#include <ti/sysbios/heaps/HeapBuf.h>
#include <ti/sysbios/heaps/HeapMem.h>
#include <ti/sysbios/knl/Semaphore.h>

#include <xdc/cfg/global.h>

/* Chip Level definitions include */
#include <ti/csl/csl_chip.h>

/* CSL EMAC include */
#include <ti/csl/csl_cpsw.h>
#include <ti/csl/csl_cpsgmii.h>
#include <ti/csl/csl_cpsgmiiAux.h>
#include <ti/csl/csl_mdio.h>
#include <ti/csl/csl_mdioAux.h>

/* BootCfg module include */
#include <ti/csl/csl_bootcfg.h>
#include <ti/csl/csl_bootcfgAux.h>

/* semaphor module */
#include <ti/csl/csl_semAux.h>

/* CSL Cache module includes */
#include <ti/csl/csl_cacheAux.h>

/* CSL XMC module includes */
#include <ti/csl/csl_xmcAux.h>

/* PA LLD include */
#include <ti/drv/pa/pa.h>

/* CPPI LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>

/** Number of cores in the test */
#define         NUM_CORES      4

/** Number of ports in the ethernet subsystem */
#define         NUM_PORTS                   3u

/** Number of MAC/GMII ports in the ethernet switch */
#define         NUM_MAC_PORTS               2u

/** Number of host descriptors used by the example program 
 * The number of descriptors needs to be in 2^X.
 * Therefore, this will be constant and not a function
 * of the number of cores in the test */  
#define         NUM_HOST_DESC               256

/** Host descriptor size. 
 *
 *  Big enough to hold the mandatory fields of the 
 *  host descriptor and PA Control Data
 * 
 *  = 32 bytes for Host desc + PA Control data
 */
#define         SIZE_HOST_DESC              80 

/** Number of PA Tx queues available */
#define         NUM_PA_TX_QUEUES            9

/** Number of PA Rx channels available */
#define         NUM_PA_RX_CHANNELS          24

/* Initial RX queue number */
#define         RX_QUEUE_NUM_INIT           900

#define CACHE_LINESZ    128
#define SYS_ROUND_UP(x,y)   ((x) + ((y) -1))/(y)*(y)

/**********************************************************************
 ****************************** Defines *******************************
 **********************************************************************/

/* Hardware Semaphore to synchronize access from
 * multiple applications (PA applications and non-PASS applications)
 * across different cores to the QMSS library.
 */
#define     QMSS_HW_SEM         3 

/* Hardware Semaphore to synchronize access from
 * multiple applications (PASS applications and non-PASS applications)
 * across different cores to the CPPI library.
 */
#define     CPPI_HW_SEM         4 

/* Hardware Semaphore to synchronize access from
 * multiple applications (PASS applications and non-PASS applications)
 * across different cores to the PA library.
 */
#define     PA_HW_SEM           5 

/* Hardware Semaphore to synchronize access from
 * multiple applications (PASS applications and non-PASS applications)
 * across different cores in this application.
 */
#define PA_APP_HW_SEM_SYS       6

#undef L2_CACHE
#ifdef L2_CACHE
    /* Invalidate L2 cache. This should invalidate L1D as well. 
     * Wait until operation is complete. */    
#define SYS_CACHE_INV(addr, size, code)    CACHE_invL2 (addr, size, code)

    /* Writeback L2 cache. This should Writeback L1D as well. 
     * Wait until operation is complete. */ 
#define SYS_CACHE_WB(addr, size, code)     CACHE_wbL2 (addr, size, code)

#else       
    /* Invalidate L1D cache and wait until operation is complete. 
     * Use this approach if L2 cache is not enabled */    
#define SYS_CACHE_INV(addr, size, code)    CACHE_invL1d (addr, size, code)
    /* Writeback L1D cache and wait until operation is complete. 
     * Use this approach if L2 cache is not enabled */    
#define SYS_CACHE_WB(addr, size, code)     CACHE_wbL1d (addr, size, code)

#endif


/* Define LoopBack modes */  
#define CPSW_LOOPBACK_NONE           0
#define CPSW_LOOPBACK_INTERNAL       1
#define CPSW_LOOPBACK_EXTERNAL       2

extern Qmss_QueueHnd gPaTxQHnd [], gTxFreeQHnd, gRxFreeQHnd, gRxQHnd[];
extern Pa_Handle gPAInstHnd;
extern paHandleL2L3_t gPaL2Handles[];
extern paHandleL2L3_t gPaL3Handles[];

extern Int cpswLpbkMode;
extern Int cpswSimTest;
extern Int cpswEvm6678;

extern Int32 Cpsw_SwitchOpen (void);
extern Int32 Mdio_Open (void);
extern Int32 Sgmii_Open (void);
extern Int32 Init_Qmss (void);
extern Int32 Init_Qmss_Local (Void);
extern Int32 Init_Cppi (void);
extern Int32 Init_Cpsw (void);
extern Int32 Init_PASS (void);
extern Int32 Setup_Tx (void);
extern Int32 Setup_Rx (void);
extern Int32 Setup_PASS (void);
extern UInt32 Convert_CoreLocal2GlobalAddr (UInt32  addr);
extern Int32 SendPacket (Void);
extern Int32 ReceivePacket (void);
extern Void  CycleDelay (Int32 count);
extern Int32 VerifyPacket (Cppi_Desc* pCppiDesc);
extern Void  ModifyPacket (Void);


#endif
