/**  
 * @file multicore_example.c
 *
 * @brief 
 *  Example to illustrate the usage of EMAC CPSW3G switch using CPPI, QMSS
 * 	low level drivers and CSL.
 *
 * 	This example application does the following:
 * 	    (1) Initializes:
 * 	            (a) Queue Manager (QM) Subsystem 
 * 	            (b) Packet Accelerator (PA) CPPI DMA 
 * 	            (c) Ethernet Subsystem (Ethernet switch + SGMII + MDIO)
 * 	            (d) PA Subsystem + PDSP
 *
 * 	    (2) Sets up the CPPI descriptors and Queues required for sending and
 * 	        receiving data using Ethernet.
 * 	            (a) Uses Host descriptors
 * 	            (b) Uses High Priority Accumulation interrupts
 *
 * 	    (3) Sets up the example application's configuration (MAC address
 * 	        it uses to send/recv data; IP address and port number it's listening
 * 	        on) in PA Subsystem so as to enable the PASS to forward all packets
 * 	        matching this configuration onto the application for processing.
 * 	            (a) Switch MAC address configured   =   0x10:0x11:0x12:0x13:0x14:0x15
 * 	            (b) Example's IP address            =   192.168.1.10
 * 	            (c) Example App's listening port    =   0x5678
 *
 * 	    (4) Sends packets onto wire 
 * 	        (constructed manually in code here with following settings):
 * 	            (a) Source MAC      =   0x00:0x01:0x02:0x03:0x04:0x05
 * 	                Destination MAC =   0x10:0x11:0x12:0x13:0x14:0x15
 *              (b) Source IP       =   192.168.1.1
 *                  Destination IP  =   192.168.1.10
 *              (c) Source Port     =   0x1234
 *                  Destination Port=   0x5678
 *              (d) Payload Data (80 bytes)
 *
 *          The packets sent by the application are sent onto wire and 
 *          since the destination MAC on the packet is the Ethernet Switch 
 *          MAC address, the packets are received by simulator and passed 
 *          back up to the example application for processing.
 *      
 *      (5) Application receives all packets using QM High priority interrupt
 *          registered; Validates received packet against data sent.
 *
 *  Example application Setup:
 *
 *          PC Running Simulator using CCS connected to a
 *          Switch/Hub. You could put another PC on the Hub to observe packets 
 *          being sent onto wire. 
 *
 *          Please consult the Readme.txt packaged with the example to 
 *          setup the CCS simulator configuration required to run this example 
 *          succesfully.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <multicore_example.h>
#include <ti/csl/cslr_device.h>
#include <ti/csl/csl_psc.h>
#include <ti/csl/csl_pscAux.h>

/* PA LLD include */
#include <ti/drv/pa/pa.h>


/**************************************************************
************************** DEFINITIONS ************************
***************************************************************/
/* Number of packets to be used for testing the example. */
#define                     MAX_NUM_PACKETS                         10u


/**************************************************************
************************* GLOBAL VARIABLES ********************
***************************************************************/
/* Counters to track number of packets sent/received by this application */
extern volatile UInt32		 gRxCounter, gTxCounter;

/*
 * Default test configuration for the silicon
 *
 * To run test at the CCS simulator
 *    cpswSimTest = 1
 *    cpswLpbkMode = CPSW_LOOPBACK_EXTERNAL
 */
#ifdef  SIMULATOR_SUPPORT
Int cpswSimTest = 1;
Int cpswLpbkMode = CPSW_LOOPBACK_EXTERNAL;
#else
Int cpswSimTest = 0;
Int cpswLpbkMode = CPSW_LOOPBACK_INTERNAL;
#endif

void mdebugHaltPdsp (Int pdspNum);
volatile Int mdebugWait = 1;

/* multicore sync up variables */
#pragma DATA_ALIGN   (globalCfgDone, 128)
#pragma DATA_SECTION(globalCfgDone,   ".sharedDDR")
volatile UInt32  globalCfgDone = FALSE;   

#pragma DATA_ALIGN   (localCfgDone, 128)
#pragma DATA_SECTION(localCfgDone,    ".sharedDDR")
/* number of local configuration completed */
volatile UInt32  localCfgDone = 0;        

#pragma DATA_ALIGN   (readyToSendPkts, 128)
#pragma DATA_SECTION(readyToSendPkts, ".sharedDDR")
volatile UInt32  readyToSendPkts = FALSE;      

/* multicore results */
#define TEST_NOT_COMPLETED 0
#define TEST_PASSED        1
#define TEST_FAILED        2
#pragma DATA_ALIGN   (testResult, 128)
#pragma DATA_SECTION(testResult,   ".sharedDDR")
volatile UInt32  testResult[NUM_CORES];    

/**************************************************************
**************** EXAMPLE APP FUNCTIONS ************************
***************************************************************/

/***************************************************************************************
 * FUNCTION PURPOSE: Power up PA subsystem
 ***************************************************************************************
 * DESCRIPTION: this function powers up the PA subsystem domains
 ***************************************************************************************/
void passPowerUp (void)
{

    /* PASS power domain is turned OFF by default. It needs to be turned on before doing any 
     * PASS device register access. This not required for the simulator. */

    /* Set PASS Power domain to ON */        
    CSL_PSC_enablePowerDomain (CSL_PSC_PD_PASS);

    /* Enable the clocks for PASS modules */
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_PKTPROC, PSC_MODSTATE_ENABLE);
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_CPGMAC,  PSC_MODSTATE_ENABLE);
    CSL_PSC_setModuleNextState (CSL_PSC_LPSC_Crypto,  PSC_MODSTATE_ENABLE);

    /* Start the state transition */
    CSL_PSC_startStateTransition (CSL_PSC_PD_PASS);

    /* Wait until the state transition process is completed. */
    while (!CSL_PSC_isStateTransitionDone (CSL_PSC_PD_PASS));
}


/** ============================================================================
 *   @n@b MultiCoreApp
 *
 *   @b Description
 *   @n Example application that sets up the application, sends, receives
 *      data.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return
 *   @n None
 *
 * =============================================================================
 */
Void MultiCoreApp (Void)
{
	Int32	   i;
  	UInt32     coreNum;
	Int32	   rxStatus=0;
    volatile UInt32     testComplete=FALSE;

    /* Get the core number. */
    coreNum = CSL_chipReadReg(CSL_CHIP_DNUM); 


    System_printf ("************************************************\n");
    System_printf ("*** PA Multi Core Example Started on Core %d ***\n",coreNum);
    System_printf ("************************************************\n");

    /* Init internal cycle counter */
    TSCL = 1; 
    
    /* Disable L1 and L2 Cache */
    //CACHE_wbAllL1d (CACHE_WAIT);
    //CACHE_setL1DSize(CACHE_L1_0KCACHE);
    //CACHE_setL1PSize(CACHE_L1_0KCACHE);
    #ifndef L2_CACHE
    CACHE_setL2Size(CACHE_0KCACHE);
    #endif
    
    /* Adjust the data packet as a function of the core number */
    ModifyPacket(); 

    /* All other cores wait for core 0 to finish the global config
       and setup the QMSS/CPPI/PASS */
    if(coreNum)
    {
        System_printf ("Waiting for global config...\n");
        while(!globalCfgDone)
            SYS_CACHE_INV ((void *) &globalCfgDone, 128, CACHE_WAIT);
    }
    
    /* Core 0 does the global initialization */
    if(!coreNum)
    {
        /* Initialize local sync flags for all cores
           Initialize test result for all cores */
        for(i=0; i<NUM_CORES; i++)
        {
            testResult[i]  =TEST_NOT_COMPLETED;
        }
        
        SYS_CACHE_WB ((void *) testResult, 128, CACHE_WAIT);
        
        /* Enable PASS power domain */
        passPowerUp();        
        
        /* Initialize the components required to run the example:
         *  (1) QMSS
         *  (2) CPPI
         *  (3) Ethernet switch subsystem + MDIO + SGMII
         */
        /* Initialize QMSS */
        if (Init_Qmss () != 0)
        {
            System_printf ("QMSS Global init failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("QMSS successfully initialized \n");            
        }

        /* Initialize CPPI */
        if (Init_Cppi () != 0)
        {
            System_printf ("CPPI init failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("CPPI successfully initialized \n");            
        }

        /* Init PA LLD */
        if (Init_PASS () != 0)
        {
            System_printf ("PASS init failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("PASS successfully initialized \n");            
        }

        /* Initialize the CPSW switch */
        if (Init_Cpsw () != 0)
        {
            System_printf ("Ethernet subsystem init failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("Ethernet subsystem successfully initialized \n");            
        }

        /* Setup Tx */
        if (Setup_Tx () != 0)
        {
            System_printf ("Tx setup failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("Tx setup successfully done \n");            
        }
    }
    else
    {
        /* Cores other than 0 do local QMSS initialization */
        if (Init_Qmss_Local () != 0)
        {
            System_printf ("QMSS Local init failed \n");
            BIOS_exit (-1);
        }
        else
        {
            System_printf ("QMSS Local successfully initialized \n");            
        }
        
        /* 
         * read the actual value from memory which have been initialized by core 0
         */
        SYS_CACHE_INV ((void *) &gRxFreeQHnd, 128, CACHE_WAIT);
        SYS_CACHE_INV ((void *) &gTxFreeQHnd, 128, CACHE_WAIT);
        SYS_CACHE_INV ((void *) gPaTxQHnd, 128, CACHE_WAIT);
        SYS_CACHE_INV ((void *) gPaL3Handles, 128, CACHE_WAIT);
        SYS_CACHE_INV ((void *) &gPAInstHnd, 128, CACHE_WAIT);
    }    

    /* Setup Rx */
    if (Setup_Rx () != 0)
    {
        System_printf ("Rx setup failed \n");
        BIOS_exit (-1);
    }
    else
    {
        System_printf ("Rx setup successfully done \n");            
    }

    /* Setup PA */
    if (Setup_PASS () != 0)
    {
        System_printf ("PASS setup failed \n");
        BIOS_exit (-1);
        
    }
    else
    {
        System_printf ("PASS setup successfully done \n");            
    }

    /* Core 0 finished the global config. Set flag so other
       cores can start their local config. */
    if (!coreNum)
    {
      globalCfgDone=TRUE;
      SYS_CACHE_WB ((void *) &globalCfgDone, 128, CACHE_WAIT);

    }  
    
    /* All cores update the counter informing that they finished their setup */
    /* The global variable is a shared resource which is being accessed from multiple cores. 
     * So here we need to protect it and ensure that there is only 1 core which is accessing 
     * it at a time. We use a Hardware Semaphore to protect this. */
    while ((CSL_semAcquireDirect (PA_APP_HW_SEM_SYS)) == 0);

    /* Invalidate the cache and make sure you get the latest from the memory. */
    SYS_CACHE_INV ((void *) &localCfgDone, 128, CACHE_WAIT);

    /* The core has completed local initialization */
    localCfgDone++;

    /* The SRIO Socket has been created. Writeback the contents to the cache. */
    SYS_CACHE_WB ((void *) &localCfgDone, 128, CACHE_WAIT);

    /* Release the hardware semaphore. */
    CSL_semReleaseSemaphore (PA_APP_HW_SEM_SYS);
    
    /* All cores wait here to sync up and send packets to PA
       at the same time. */
    System_printf ("Waiting for all cores to reach the barrier before transmission starts ... \n");
    
    while (localCfgDone != NUM_CORES)
        SYS_CACHE_INV ((void *) &localCfgDone, 128, CACHE_WAIT);
    
    /* Send data towards switch */
    System_printf ("Packet Transmission Start ... \n");
    for (i = 0; i < MAX_NUM_PACKETS; i ++)
    {
        if (SendPacket () != 0)
        {
            System_printf ("Packet %d send failed \n", i);
            BIOS_exit (-1);
        }
    }
    
    /* Wait until all packet reception is done */
   	System_printf ("Packet Transmission Done.\nWait for all packets to be Received ... \n");
    while (gRxCounter < gTxCounter)
    {
        if(ReceivePacket() != 0)
            rxStatus=-1;
    }
    
    System_printf ("Core %d: Packets Sent\t\t=\t%d \nCore %d: Packets Received\t=\t%d \n",coreNum, gTxCounter, coreNum,  gRxCounter);

    /* The global variable is a shared resource which is being accessed from multiple cores. 
     * So here we need to protect it and ensure that there is only 1 core which is accessing 
     * it at a time. We use a Hardware Semaphore to protect this. */
    while ((CSL_semAcquireDirect (PA_APP_HW_SEM_SYS)) == 0);
    
    /* Invalidate the cache and make sure you get the latest from the memory. */
    SYS_CACHE_INV ((void *) testResult, 128, CACHE_WAIT);

    if(rxStatus == 0)
       testResult[coreNum]=TEST_PASSED;
    else
       testResult[coreNum]=TEST_FAILED;
       
    /* Write the result back to the memory */   
    SYS_CACHE_WB ((void *) testResult, 128, CACHE_WAIT);
    
    /* Release the hardware semaphore. */
    CSL_semReleaseSemaphore (PA_APP_HW_SEM_SYS);

    /* Core 0 collects all the results and declare PASS or Fail */
    if(!coreNum)
    {
   	    System_printf ("Wait for all packets to be Received in all cores... \n");
        
        
        /* Wait until all cores have completed the test */
        while(!testComplete)
        {
            /* Give some time for the other cores to process the packet */
            CycleDelay (10000);

            testComplete=TRUE;
            
            /* Invalidate the cache and make sure you get the latest from the memory. */
            SYS_CACHE_INV ((void *) testResult, 128, CACHE_WAIT);
            
            for(i=0; i<NUM_CORES; i++)
            {
                if(testResult[i]==TEST_NOT_COMPLETED)
                    testComplete=FALSE;
            }
        }
        
        for(i=0; i<NUM_CORES; i++)
        {
            if(testResult[i]==TEST_PASSED)
                System_printf ("Test passed on core %d\n",i);
            else
                System_printf ("Test failed on core %d\n",i);
        }
    }

    System_printf ("**********************************************\n");
    System_printf ("*** PA Multi Core Example Ended on Core %d ***\n",coreNum);
    System_printf ("**********************************************\n");

    /* Example application done. Return success */
    BIOS_exit (0);
}

/** ============================================================================
 *   @n@b main
 *
 *   @b Description
 *   @n Entry point for single core example application.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return
 *   @n None
 * =============================================================================
 */
Int32 main (Void)
{
    Task_Params                	cpswTaskParams;
    
    /* Initialize the task params */
    Task_Params_init(&cpswTaskParams);

    /* Create the CPSW single core example task */
    Task_create((Task_FuncPtr)&MultiCoreApp, &cpswTaskParams, NULL);

    /* Start the BIOS Task scheduler */
	BIOS_start ();

	return 0;	
}
