/**
 *   @file  qmss_test.c
 *
 *   @brief   
 *      This is the QMSS unit test code.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2009-2011, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  \par
*/

#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <string.h>

#ifndef SIMULATOR_SUPPORT
/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>
#include <ti/drv/qmss/qmss_firmware.h>
#include <ti/drv/qmss/qmss_qos.h>

/* CSL RL includes */
#include <ti/csl/csl_chip.h>

/* OSAL includes */
#include <qmss_osal.h>

/************************ USER DEFINES ********************/
#define NUM_MONOLITHIC_DESC         2048
#define SIZE_MONOLITHIC_DESC        32
#define PROFILE_DESCS               400

#if (PROFILE_DESCS * QMSS_QOS_SRIO_MAX_TX_Q) > NUM_MONOLITHIC_DESC
#error NUM_MONOLITHIC_DESC is too small
#endif

/************************ GLOBAL VARIABLES ********************/
/* Descriptor pool [Size of descriptor * Number of descriptors] */
#pragma DATA_ALIGN (monolithicDesc, 16)
UInt8                   monolithicDesc[SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC];

/* Timestamp when SRIO model moved each descriptor */
struct 
{
    void    *descPtr;
    int      queueNum;
    uint32_t timestamp;
} timestamps[NUM_MONOLITHIC_DESC];

/* Global variable common to all test cases */

/* QMSS configuration */
Qmss_InitCfg            qmssInitConfig;
/* Memory region configuration information */
Qmss_MemRegInfo         memInfo;
/* QM descriptor configuration */
Qmss_DescCfg            descCfg;
/* Store the queue handle for destination queues on which allocated descriptors are stored */
Qmss_QueueHnd           QueHnd[QMSS_MAX_MEM_REGIONS];

/************************ EXTERN VARIABLES ********************/
/* Error counter */
extern UInt32                   errorCount;
/* QMSS device specific configuration */
extern Qmss_GlobalConfigParams  qmssGblCfgParams;

/**
 *  @b Description
 *  @n  
 *      Utility function which converts a local GEM L2 memory address 
 *      to global memory address.
 *
 *  @param[in]  addr
 *      Local address to be converted
 *
 *  @retval
 *      Computed L2 global Address
 */
static UInt32 l2_global_address (UInt32 addr)
{
        UInt32 corenum;

        /* Get the core number. */
        corenum = CSL_chipReadReg (CSL_CHIP_DNUM); 

        /* Compute the global address. */
        return (addr + (0x10000000 + (corenum * 0x1000000)));
}

/**
 *  @b Description
 *  @n  
 *      Utility function to allocate a contigous block of aligned queues
 *
 *  @param[out] queueHnds 
 *      Queue handles for each successfully opened Q.  Should be at least nQ long.
 *  @param[in]  nQ
 *      Number of queues to allocate as a block
 *  @param[in]  align
 *      Queue number to align to
 *
 *  @retval
 *      Queue number of first allocated queue, or -1 on failure
 */
int allocate_contig_queues (Qmss_QueueHnd *queueHnds, int nQ, int align)
{
    int queueNum, baseQueue;
    uint8_t isAllocated;
    int foundBase = -1;

    /* Allocate a contigous block of nQ queues aligned to align */
    for (baseQueue = ((QMSS_GENERAL_PURPOSE_QUEUE_BASE + align - 1) / align) * align;
         baseQueue < (QMSS_GENERAL_PURPOSE_QUEUE_BASE + QMSS_MAX_GENERAL_PURPOSE_QUEUE - nQ);
         baseQueue += align) 
    {
        for (queueNum = 0; queueNum < nQ; queueNum++) 
        {
            queueHnds[queueNum] = 
                Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, 
                                baseQueue + queueNum, &isAllocated);
            if (queueHnds[queueNum] < 0) 
            {
                errorCount++;
                System_printf("Queue open failed: %d\n", queueHnds[queueNum]);
                break;
            }
            if (! isAllocated) 
            {
                /* Somebody else got the queue. Close what we got, and try next range */
                for (queueNum--; queueNum >= 0; queueNum--) 
                {
                    Qmss_queueClose (queueHnds[queueNum]);
                }
                break;
            }
        }
        if (queueNum == nQ) 
        {
            foundBase = baseQueue;
            break;
        }
    }

    return foundBase;
}

/**
 *  @b Description
 *  @n  
 *      Utility function to sleep N cycles.  This is immune to timer rollover.
 *
 *  @param[in]  n
 *      Number of cycles to sleep
 *
 */
void delay (uint32_t cycles)
{
    uint32_t start = TSCL;

    while ( (TSCL - start) < cycles);
}

Void testSrioContext (Void)
{
    Qmss_Result             result;
    UInt32                  numAllocated, corenum;
    Qmss_QueueHnd           freeQ;
    int                     baseQueue;
    int                     numQueues;
    int                     queueNum;
    Qmss_QueueHnd           fwQHnds[32];
    Qmss_QueueHnd           srioSimQHnds[QMSS_QOS_SRIO_MAX_TX_Q];
    Qmss_QueueHnd           srioDivQHnds[QMSS_QOS_SRIO_MAX_TX_Q];
    Qmss_QosSrioCfg         srioCfg;
    uint8_t isAllocated;

    uint32_t startTime, endTime;

    /* Reset timer */
    TSCL = 0;

    /* Get the core number. */
    corenum = CSL_chipReadReg(CSL_CHIP_DNUM); 
    System_printf ("**********Core %d TESTING DESCRIPTOR ALLOCATION ************\n", corenum);

    memset ((Void *) &qmssInitConfig, 0, sizeof (Qmss_InitCfg));

    /* Set up the linking RAM. Use internal Linking RAM.  */
    qmssInitConfig.linkingRAM0Base = 0;
    qmssInitConfig.linkingRAM0Size = 0;
    qmssInitConfig.linkingRAM1Base = 0x0;
    qmssInitConfig.maxDescNum      = NUM_MONOLITHIC_DESC;

#ifdef xdc_target__bigEndian
    qmssInitConfig.pdspFirmware[0].pdspId = Qmss_PdspId_PDSP2;
    qmssInitConfig.pdspFirmware[0].firmware = &qos_be;
    qmssInitConfig.pdspFirmware[0].size = sizeof (qos_be);
#else
    qmssInitConfig.pdspFirmware[0].pdspId = Qmss_PdspId_PDSP2;
    qmssInitConfig.pdspFirmware[0].firmware = &qos_le;
    qmssInitConfig.pdspFirmware[0].size = sizeof (qos_le);
#endif

    /* Initialize Queue Manager SubSystem */
    result = Qmss_init (&qmssInitConfig, &qmssGblCfgParams);
    if (result != QMSS_SOK)
    {
        System_printf ("Error Core %d : Initializing Queue Manager SubSystem error code : %d\n", corenum, result);
        errorCount++;
        return;
    }

    /* Start Queue Manager SubSystem */
    result = Qmss_start ();
    if (result != QMSS_SOK)
    {
        System_printf ("Core %d : Error starting Queue Manager error code : %d\n", corenum, result);
    }

    /* Setup memory region for monolithic descriptors */
    memset ((Void *) &monolithicDesc, 0, SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC);
    memInfo.descBase = (UInt32 *) l2_global_address ((UInt32) monolithicDesc);
    memInfo.descSize = SIZE_MONOLITHIC_DESC;
    memInfo.descNum = NUM_MONOLITHIC_DESC;
    memInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    memInfo.memRegion = Qmss_MemRegion_MEMORY_REGION_NOT_SPECIFIED;
    memInfo.startIndex = 0;

    result = Qmss_insertMemoryRegion (&memInfo);
    if (result < QMSS_SOK)
    {
        System_printf ("Error Core %d : Inserting memory region %d error code : %d\n", corenum, memInfo.memRegion, result);
        errorCount++;
    }

    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION0;
    descCfg.descNum = NUM_MONOLITHIC_DESC;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_STARVATION_COUNTER_QUEUE;
    
    /* Initialize the descriptors and push to free Queue */
    if ((freeQ = Qmss_initDescriptor (&descCfg, &numAllocated)) < 0)
        {
        System_printf ("Error Core %d : Initializing descriptor error code: %d \n", corenum, freeQ);
        errorCount++;
        }
    else
    {
        if (descCfg.descNum != numAllocated)
        {
            errorCount++;
        }
            
        System_printf ("Core %d : Number of descriptors requested : %d. Number of descriptors allocated : %d \n",
            corenum, descCfg.descNum, numAllocated);
    }

    /* Allocate block of queues to be used by firmware */
    baseQueue = allocate_contig_queues (fwQHnds, 32, 32);
    if (baseQueue < 0) 
    {
        System_printf ("Core %d : Failed to open 32 contiguous queues\n");
        errorCount++;
    }
    srioCfg.queBase = baseQueue;

    /* Allocate queues to simulate SRIO HW */
    for (queueNum = 0; queueNum < QMSS_QOS_SRIO_MAX_TX_Q; queueNum++)
    {
        srioSimQHnds[queueNum] = 
            Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, 
                            QMSS_PARAM_NOT_SPECIFIED, &isAllocated);
        if (srioSimQHnds[queueNum] < 0) 
        {
            errorCount++;
            System_printf("Queue open failed: %d\n", srioSimQHnds[queueNum]);
            break;
        }
        srioCfg.TXQs[queueNum].threshold = 10;
        srioCfg.TXQs[queueNum].txQ = Qmss_getQueueNumber (srioSimQHnds[queueNum]);
    }

    /* Allocate queues to temporarily hold descriptors during test setup.
     * They will be diverted to shadow queues all at once
     */
    for (queueNum = 0; queueNum < QMSS_QOS_SRIO_MAX_TX_Q; queueNum++)
    {
        srioDivQHnds[queueNum] = 
            Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, 
                            QMSS_PARAM_NOT_SPECIFIED, &isAllocated);
        if (srioDivQHnds[queueNum] < 0) 
        {
            errorCount++;
            System_printf("Queue open failed: %d\n", srioDivQHnds[queueNum]);
            break;
        }
    }

#if 1
    if ((result = Qmss_configureQosTimer (437) != QCMD_RETCODE_SUCCESS))
    {
        errorCount++;
        System_printf("Core %d : Failed to configure QoS timer: %d\n", corenum, result);
    }
#endif

    if (errorCount) 
    {
        System_printf ("Test failed\n");
        return;
    }


    /* Iterate over each number of supported tx queues, and calculate throughput */
    for (numQueues = QMSS_QOS_SRIO_MIN_TX_Q; numQueues <= QMSS_QOS_SRIO_MAX_TX_Q; numQueues++)
    {
        int descID = 0;
        int descNum;

        /* Set up the cluster */
        srioCfg.queCount = numQueues;
        if ((result = Qmss_configureQosSrioCluster (0, &srioCfg)) != QCMD_RETCODE_SUCCESS)
        {
            System_printf ("Core %d : failed to configure SRIO cluster : %d\n", corenum, result);
            errorCount++;
            break;
        }
       
        /* Place PROFILE_DESCS descriptors into each input queue */
        for (queueNum = 0; queueNum < numQueues; queueNum++) 
        {
            for (descNum = 0; descNum < PROFILE_DESCS; descNum++) 
            {
                uint32_t *desc;
                desc = (uint32_t *)QMSS_DESC_PTR(Qmss_queuePop (freeQ));
                if (! desc) 
                {
                    System_printf ("Core %d : failed to pop a free desc\n", corenum);
                    errorCount++;
                    break;
                }
                /* No cache operation is needed since only the same CPU will read */
                *desc = descID ++; /* write unique ID to each descriptor */
                Qmss_queuePushDesc (srioDivQHnds[queueNum], desc);
            }
        }

        /* Enable the cluster */
        if ((result = Qmss_enableQosSrioCluster (0)) != QCMD_RETCODE_SUCCESS)
        {
            System_printf ("Core %d : failed to enable SRIO cluster : %d\n", corenum, result);
            errorCount++;
            break;
        }

        /* Start the clock */
        startTime = TSCL;
        /* Divert each queue of descriptors to a firmware monitored shadow queue */
        for (queueNum = 0; queueNum < numQueues; queueNum++) 
        {
            Qmss_queueDivert (srioDivQHnds[queueNum], 
                              fwQHnds[queueNum + QMSS_QOS_SRIO_SHADOW_TX_Q_OFFSET],
                              Qmss_Location_TAIL);
        }

        /* Run a simple model of the SRIO HW to move the descriptors placed on HW queues
         * to the shadow TX completion queues
         */
        queueNum = 0;
        for (descNum = 0; descNum < PROFILE_DESCS * numQueues; ) 
        {
           uint32_t *desc = (uint32_t *)QMSS_DESC_PTR(Qmss_queuePop (srioSimQHnds[queueNum]));
           if (desc) 
           {
                timestamps[descNum].timestamp = TSCL;
                timestamps[descNum].queueNum  = queueNum;
                timestamps[descNum].descPtr   = desc;
                descNum++;
                Qmss_queuePushDesc (fwQHnds[queueNum + QMSS_QOS_SRIO_SHADOW_TX_COMPLETION_Q_OFFSET],
                                   desc);
           } 
           else 
           {
               /* move to next queue if this queue is empty */
               queueNum++;
               if (queueNum >= numQueues) 
               {
                   queueNum = 0;
               }
           }
        }
        endTime = TSCL;
        System_printf ("Core %d: moved %d descriptors on %d queues (%d total) "
                       "in %d cycles (%d cycles per descriptor)\n",
                       corenum, 
                       PROFILE_DESCS, 
                       numQueues, 
                       PROFILE_DESCS * numQueues,
                       endTime - startTime, 
                       (endTime - startTime) / (PROFILE_DESCS * numQueues));


        /* Disable the cluster */
        if ((result = Qmss_disableQosSrioCluster (0)) != QCMD_RETCODE_SUCCESS)
        {
            System_printf ("Core %d : failed to disable SRIO cluster : %d\n", corenum, result);
            errorCount++;
            break;
        }

        /* Verify the results and return descriptors to free q */
        descID = 0;
        for (queueNum = 0; queueNum < numQueues; queueNum++) 
        {
            for (descNum = 0; descNum < PROFILE_DESCS; descNum++) 
            {
                uint32_t *desc;
                desc = (uint32_t *)QMSS_DESC_PTR(Qmss_queuePop (
                          fwQHnds[queueNum + QMSS_QOS_SRIO_TX_COMPLETION_Q_OFFSET]));
                if (! desc) 
                {
                    System_printf ("Core %d : failed to pop completed desc\n", corenum);
                    errorCount++;
                    break;
                }
                /* No cache operation is needed since only the same CPU wrote */
                if (*desc != descID) 
                {
                    System_printf ("Core %d : completed descriptors out of "
                                   "order got %d expect %d\n", 
                                   corenum,
                                   *desc,
                                   descID);
                    errorCount++;
                }
                descID = *desc + 1;
                Qmss_queuePushDesc (freeQ, desc);
            }
        }
    }

    if (errorCount == 0)
        System_printf ("\nCore %d : Descriptor allocation tests Passed\n");
}

Void run_test (Void)
{
    testSrioContext ();
}
#else
Void run_test (Void)
{
    System_printf ("Simulator doesn't support SRIO tracking functionality.");
}
#endif


