/*
 *
 * Copyright (C) 2010-2012 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/



/* Common test utilities */
#include "../pautest.h"

#define TEST_COMMON_STATS_REQ_ID	0x12123487

/* NULL terminated list of tests */
paTest_t  paTestList[] = {
	{ paTestUnconfigured, 	"Packet reception while unconfigured", PA_TEST_NOT_RUN },
	{ paTestSrioRouting,    "Pa_addSrio and SRIO routing",         PA_TEST_NOT_RUN },
	{ paTestL2Routing,    	"Pa_addMac and L2 routing",            PA_TEST_NOT_RUN },
#ifndef __LINUX_USER_SPACE 
	{ paTestL3Routing, 		"Pa_addIp and L3 Routing",			   PA_TEST_NOT_RUN },
	{ paTestL4Routing, 		"Pa_addPort and L4 Routing", 		   PA_TEST_NOT_RUN },
#endif
	{ paTestPatchRoute,     "Blind patch and route",               PA_TEST_NOT_RUN },
	{ paTestTxFmtRt,      	"Tx checksum and routing",             PA_TEST_NOT_RUN },
	{ paTestCustom,			"Custom routing",					   PA_TEST_NOT_RUN },
#ifndef __LINUX_USER_SPACE 
	{ paTestIPv4FragReassem,"IPv4 Fragmentation and Reassembly",   PA_TEST_NOT_RUN },
#endif
 	{ paTestUnconfigured, 	"Packet reception while unconfigured", PA_TEST_NOT_RUN }, 
#ifndef __LINUX_USER_SPACE 
	{ paTestMultiRouting,   "Multi-routing",					   PA_TEST_NOT_RUN },
#endif
	{ NULL,                 NULL,                                  PA_TEST_NOT_RUN }
};

/* Request stats from the PA */
int32_t testCommonRequestPaStats (char *fname, tFramework_t *tf, Bool reset, int32_t QSource, int32_t QRecycle,  paCmdReply_t *reply)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	paReturn_t     paret;
	uint16_t         cmdSize;
	int32_t			   cmdDest;
	uint32_t		   pktcmd = PASAHO_PACFG_CMD;
	volatile int32_t   mdebugWait = 0;
	
	/* Pop a descriptor with an associated buffer to use for the command request */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QSource)) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Failed to pop descriptor from queue %d\n", fname, __FILE__, __LINE__, QSource);
		return (-1);
	}
	
	/* Make sure there is a buffer attached and get the size */
	if (hd->buffPtr == (uint32_t)NULL)  {
		System_printf ("%s (%s:%d): Descriptor from queue %d had no associated buffer\n", fname, __FILE__, __LINE__, QSource);
		return (-1);
	}
	cmdSize = hd->origBufferLen;
	
	paret = Pa_requestStats (tf->passHandle, reset, (paCmd_t)hd->buffPtr, &cmdSize, reply, &cmdDest);

	
	if (paret != pa_OK)  {
		System_printf ("%s (%s:%d): Pa_requestStats returned error code %d\n", fname, __FILE__, __LINE__, paret);
		Qmss_queuePushDesc (QSource, (Ptr)hd);
		return (-1);
	}
	
	
	/* Set the packet length and the buffer length. The buffer length MUST NOT
	 * exceed the packet length */
	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, cmdSize);
  	hd->buffLen = cmdSize;
	
	/* Mark the packet as a configuration packet and setup for the descriptor return */
  	Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&pktcmd, 4);
  	
  	q.qMgr = 0;
  	q.qNum = QRecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);	
  	
/*  	mdebugHaltPdsp (4); */
	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
	while (mdebugWait);
	return (0);

}

/* Compare stats and report any differences */
int32_t testCommonCompareStats (char *fname, paSysStats_t *expected, paSysStats_t *actual)
{
	int32_t retval = 0;
	
	if (expected->classify1.nPackets != actual->classify1.nPackets)  {
		System_printf ("%s: Stat classify1.nPackets expected %d, found %d\n", fname, expected->classify1.nPackets, actual->classify1.nPackets);
		retval = 1;
	}
  	if (expected->classify1.nIpv4Packets != actual->classify1.nIpv4Packets)  {
  		System_printf ("%s: Stat classify1.nIpv4Packets expected %d, found %d\n", fname, expected->classify1.nIpv4Packets, actual->classify1.nIpv4Packets);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nIpv6Packets != actual->classify1.nIpv6Packets)  {
  		System_printf ("%s: Stat classify1.nIpv6Packets expected %d, found %d\n", fname, expected->classify1.nIpv6Packets, actual->classify1.nIpv6Packets);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nCustomPackets != actual->classify1.nCustomPackets)  {
  		System_printf ("%s: Stat classify1.nCustomPackets expected %d, found %d\n", fname, expected->classify1.nCustomPackets, actual->classify1.nCustomPackets);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nSrioPackets != actual->classify1.nSrioPackets)  {
  		System_printf ("%s: Stat classify1.nSrioPackets expected %d, found %d\n", fname, expected->classify1.nSrioPackets, actual->classify1.nSrioPackets);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nLlcSnapFail != actual->classify1.nLlcSnapFail)  {
  		System_printf ("%s: Stat classify1.nLlcSnapFail expected %d, found %d\n", fname, expected->classify1.nLlcSnapFail, actual->classify1.nLlcSnapFail);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nTableMatch != actual->classify1.nTableMatch)  {
  		System_printf ("%s: Stat classify1.nTableMatch expected %d, found %d\n", fname, expected->classify1.nTableMatch, actual->classify1.nTableMatch);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nNoTableMatch != actual->classify1.nNoTableMatch)  {
  		System_printf ("%s: Stat classify1.nNoTableMatch expected %d, found %d\n", fname, expected->classify1.nNoTableMatch, actual->classify1.nNoTableMatch);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nIpFrag != actual->classify1.nIpFrag)  {
  		System_printf ("%s: Stat classify1.nIpFrag expected %d, found %d\n", fname, expected->classify1.nIpFrag, actual->classify1.nIpFrag);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nIpDepthOverflow != actual->classify1.nIpDepthOverflow)  {
  		System_printf ("%s: Stat classify1.nIpDepthOverflow expected %d, found %d\n", fname, expected->classify1.nIpDepthOverflow, actual->classify1.nIpDepthOverflow);
  		retval = 1;
  	}
  	
 	if (expected->classify1.nVlanDepthOverflow != actual->classify1.nVlanDepthOverflow)  {
 		System_printf ("%s: Stat classify1.nVlanDepthOverflow expected %d, found %d\n", fname, expected->classify1.nVlanDepthOverflow, actual->classify1.nVlanDepthOverflow);
 		retval = 1;
 	}
 	
  	if (expected->classify1.nGreDepthOverflow != actual->classify1.nGreDepthOverflow)  {
  		System_printf ("%s: Stat classify1.nGreDepthOverflow expected %d, found %d\n", fname, expected->classify1.nGreDepthOverflow, actual->classify1.nGreDepthOverflow);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nMplsPackets != actual->classify1.nMplsPackets)  {
  		System_printf ("%s: Stat classify1.nMplsPackets expected %d, found %d\n", fname, expected->classify1.nMplsPackets, actual->classify1.nMplsPackets);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nParseFail != actual->classify1.nParseFail)  {
  		System_printf ("%s: Stat classify1.nParseFail expected %d, found %d\n", fname, expected->classify1.nParseFail, actual->classify1.nParseFail);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nInvalidIPv6Opt != actual->classify1.nInvalidIPv6Opt)  {
  		System_printf ("%s: Stat classify1.nInvalidIPv6Opt expected %d, found %d\n", fname, expected->classify1.nInvalidIPv6Opt, actual->classify1.nInvalidIPv6Opt);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nTxIpFrag != actual->classify1.nTxIpFrag)  {
  		System_printf ("%s: Stat classify1.nTxIpFrag expected %d, found %d\n", fname, expected->classify1.nTxIpFrag, actual->classify1.nTxIpFrag);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nSilentDiscard != actual->classify1.nSilentDiscard)  {
  		System_printf ("%s: Stat classify1.nSilentDiscard expected %d, found %d\n", fname, expected->classify1.nSilentDiscard, actual->classify1.nSilentDiscard);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nInvalidControl != actual->classify1.nInvalidControl)  {
  		System_printf ("%s: Stat classify1.nInvalidControl expected %d, found %d\n", fname, expected->classify1.nInvalidControl, actual->classify1.nInvalidControl);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nInvalidState != actual->classify1.nInvalidState)  {
  		System_printf ("%s: Stat classify1.nInvalidState expected %d, found %d\n", fname, expected->classify1.nInvalidState, actual->classify1.nInvalidState);
  		retval = 1;
  	}
  	
  	if (expected->classify1.nSystemFail != actual->classify1.nSystemFail)  {
  		System_printf ("%s: Stat classify1.nSystemFail expected %d, found %d\n", fname, expected->classify1.nSystemFail, actual->classify1.nSystemFail);
  		retval = 1;
  	}
  	
  	System_flush ();
  	
  	
  	
	if (expected->classify2.nPackets != actual->classify2.nPackets)  {
		System_printf ("%s: Stat classify2.nPackets expected %d, found %d\n", fname, expected->classify2.nPackets, actual->classify2.nPackets);
		retval = 1;
	}
  	
  	if (expected->classify2.nUdp != actual->classify2.nUdp)  {
  		System_printf ("%s: Stat classify2.nUdp expected %d, found %d\n", fname, expected->classify2.nUdp, actual->classify2.nUdp);
  		retval = 1;
  	}
  	
  	if (expected->classify2.nTcp != actual->classify2.nTcp)  {
  		System_printf ("%s: Stat classify2.nTcp expected %d, found %d\n", fname, expected->classify2.nTcp, actual->classify2.nTcp);
  		retval = 1;
  	}
  	
  	if (expected->classify2.nCustom != actual->classify2.nCustom)  {
  		System_printf ("%s: Stat classify2.nCustom expected %d, found %d\n", fname, expected->classify2.nCustom, actual->classify2.nCustom);
  		retval = 1;
  	}
  	
  	if (expected->classify2.nSilentDiscard != actual->classify2.nSilentDiscard)  {
  		System_printf ("%s: Stat classify2.nSilentDiscard expected %d, found %d\n", fname, expected->classify2.nSilentDiscard, actual->classify2.nSilentDiscard);
  		retval = 1;
  	}
  	
  	if (expected->classify2.nInvalidControl != actual->classify2.nInvalidControl)  {
  		System_printf ("%s: Stat classify2.nInvalidControl expected %d, found %d\n", fname, expected->classify2.nInvalidControl, actual->classify2.nInvalidControl);
  		retval = 1;
  	}
  	
  	if (expected->modify.nCommandFail != actual->modify.nCommandFail)  {
  		System_printf ("%s: Stat modify.nCommandFail expected %d, found %d\n", fname, expected->modify.nCommandFail, actual->modify.nCommandFail);
  		retval = 1;
  	}
  	
  	System_flush ();
  	return (retval);
}

/* Request User stats from the PA */
int32_t testCommonRequestUsrStats (char *fname, tFramework_t *tf, Bool reset, int32_t QSource, int32_t QRecycle,  paCmdReply_t *reply, paUsrStats_t  *usrStats)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	paReturn_t     paret;
	uint16_t       cmdSize;
	int32_t		   cmdDest;
	uint32_t	   pktcmd = PASAHO_PACFG_CMD;
	volatile int32_t   mdebugWait = 0;
	
	/* Pop a descriptor with an associated buffer to use for the command request */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QSource)) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Failed to pop descriptor from queue %d\n", fname, __FILE__, __LINE__, QSource);
		return (-1);
	}
	
	/* Make sure there is a buffer attached and get the size */
	if (hd->buffPtr == (uint32_t)NULL)  {
		System_printf ("%s (%s:%d): Descriptor from queue %d had no associated buffer\n", fname, __FILE__, __LINE__, QSource);
		return (-1);
	}
	cmdSize = hd->origBufferLen;
	
	paret = Pa_requestUsrStats (tf->passHandle, reset, (paCmd_t)hd->buffPtr, &cmdSize, reply, &cmdDest, usrStats);
	
	if (paret != pa_OK)  {
		System_printf ("%s (%s:%d): Pa_requestUsrStats returned error code %d\n", fname, __FILE__, __LINE__, paret);
		Qmss_queuePushDesc (QSource, (Ptr)hd);
		return (-1);
	}
	
    if (cmdSize)
    {
	    /* Set the packet length and the buffer length. The buffer length MUST NOT
	    * exceed the packet length */
	    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, cmdSize);
  	    hd->buffLen = cmdSize;
	
	    /* Mark the packet as a configuration packet and setup for the descriptor return */
  	    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&pktcmd, 4);
  	
  	    q.qMgr = 0;
  	    q.qNum = QRecycle;
  	    Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);	
  	
	    Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
    }
    else
    {
    
        /* Recycle the descriptor and associated buffer back to queue from which it came */
	    if (testCommonRecycleLBDesc (tf, hd))  {
		    System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats response\n", fname, __FILE__, __LINE__);
		    return (-1);
	    }
    
    }
	return (0);

}

/* Compare stats and report any differences */
int32_t testCommonCompareUsrStats (char *fname, paUsrStats_t *expected, paUsrStats_t *actual)
{
	int32_t retval = 0;
    int32_t i;
    
    for (i = 0; i < pa_USR_STATS_MAX_64B_COUNTERS; i++)
    {
	    if (expected->count64[i] != actual->count64[i])  {
		    System_printf ("%s: Usr Stat count64[%d] expected %d, found %d\n", fname, i, expected->count64[i], actual->count64[i]);
		    retval = 1;
	    }
    }
    
    for (i = 0; i < pa_USR_STATS_MAX_32B_COUNTERS; i++)
    {
	    if (expected->count32[i] != actual->count32[i])  {
		    System_printf ("%s: Usr Stat count32[%d] expected %d, found %d\n", fname, i, expected->count32[i], actual->count32[i]);
		    retval = 1;
	    }
    }
  	
  	System_flush ();
  	return (retval);
}


/* Compare long packet information */
int32_t testCommonComparePktInfo (Char *tfName, pasahoLongInfo_t *expected, pasahoLongInfo_t *actual)
{
	int32_t retval = 0;
    Bool fCustom = PASAHO_LINFO_READ_HDR_BITMASK(expected) & PASAHO_HDR_BITMASK_CUSTOM;
	
	if (PASAHO_LINFO_READ_START_OFFSET(expected) != PASAHO_LINFO_READ_START_OFFSET(actual))  {
		System_printf ("%s (%s:%d): expected start offset = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_START_OFFSET(expected), PASAHO_LINFO_READ_START_OFFSET(actual));
		retval = 1;
	}
	
	if ((PASAHO_LINFO_READ_END_OFFSET(expected)) &&
        (PASAHO_LINFO_READ_END_OFFSET(expected) != PASAHO_LINFO_READ_END_OFFSET(actual)))  {
		System_printf ("%s (%s:%d): expected end offset = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_END_OFFSET(expected), PASAHO_LINFO_READ_END_OFFSET(actual));
		retval = 1;
	}
	
	if (!fCustom && (PASAHO_LINFO_READ_EIDX(expected) != PASAHO_LINFO_READ_EIDX(actual)))  {
		System_printf ("%s (%s:%d): expected error index = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_EIDX(expected), PASAHO_LINFO_READ_EIDX(actual));
		retval = 1;
	}
	
	if (PASAHO_LINFO_READ_PMATCH(expected) != PASAHO_LINFO_READ_PMATCH(actual))  {
		System_printf ("%s (%s:%d): expected pmatch = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_PMATCH(expected), PASAHO_LINFO_READ_PMATCH(actual));
		retval = 1;
	}
	
	if (PASAHO_LINFO_READ_HDR_BITMASK(expected) != PASAHO_LINFO_READ_HDR_BITMASK(actual))  {
		System_printf ("%s (%s:%d): expected header bitmask = 0x%04x, found 0x%04x\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_HDR_BITMASK(expected), PASAHO_LINFO_READ_HDR_BITMASK(actual));
		retval = 1;
	}
    
	if ((PASAHO_LINFO_READ_HDR_BITMASK2(expected)) &&
        (PASAHO_LINFO_READ_HDR_BITMASK2(expected) != PASAHO_LINFO_READ_HDR_BITMASK2(actual))) {
		System_printf ("%s (%s:%d): expected header bitmask = 0x%012x, found 0x%012x\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_HDR_BITMASK(expected), PASAHO_LINFO_READ_HDR_BITMASK(actual));
		retval = 1;
	}
	
	if (PASAHO_LINFO_READ_VLAN_COUNT(expected) != PASAHO_LINFO_READ_VLAN_COUNT(actual))  {
		System_printf ("%s (%s:%d): expected vlan count = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_VLAN_COUNT(expected), PASAHO_LINFO_READ_VLAN_COUNT(actual));
		retval = 1;
	}
	
	if (PASAHO_LINFO_READ_IP_COUNT(expected) != PASAHO_LINFO_READ_IP_COUNT(actual))  {
		System_printf ("%s (%s:%d): expected IP count = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_IP_COUNT(expected), PASAHO_LINFO_READ_IP_COUNT(actual));
		retval = 1;
	}
	
	if (PASAHO_LINFO_READ_GRE_COUNT(expected) != PASAHO_LINFO_READ_GRE_COUNT(actual))  {
		System_printf ("%s (%s:%d): expected GRE count = %d, found %d\n", tfName, __FILE__, __LINE__, PASAHO_LINFO_READ_GRE_COUNT(expected), PASAHO_LINFO_READ_GRE_COUNT(actual));
		retval = 1;
	}
	
	System_flush ();
	
	return (retval);
}

/* Increment expected stats based on an input bit map */
void testCommonIncStats (paStatsBmap_t *map,  paSysStats_t *stats)	
{
	int32_t i;
	
	/* There are always three maps in the map list because stats can be incremented up to 3 times */
	for (i = 0; i < 3; i++)  {
		
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_PACKETS))
			stats->classify1.nPackets += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_TABLE_MATCH))
			stats->classify1.nTableMatch += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_NO_TABLE_MATCH))
			stats->classify1.nNoTableMatch += 1;
            
		if (map[i] & (1 << TF_STATS_BM_C1_IP_FRAG))
			stats->classify1.nIpFrag += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_TX_IP_FRAG))
			stats->classify1.nTxIpFrag += 1;
            
		if (map[i] & (1 << TF_STATS_BM_C1_VLAN_OVERFLOW))
			stats->classify1.nVlanDepthOverflow += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_MPLS))
			stats->classify1.nMplsPackets += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_PARSE_FAIL))
			stats->classify1.nParseFail += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_SILENT_DISCARD))
			stats->classify1.nSilentDiscard += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_IPV4))
			stats->classify1.nIpv4Packets += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_IPV6))
			stats->classify1.nIpv6Packets += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_CUSTOM))
			stats->classify1.nCustomPackets += 1;
            
		if (map[i] & (1 << TF_STATS_BM_C1_NUM_SRIO))
			stats->classify1.nSrioPackets += 1;
            
		if (map[i] & (1 << TF_STATS_BM_C2_NUM_PACKETS))
			stats->classify2.nPackets += 1;
            
		if (map[i] & (1 << TF_STATS_BM_C2_NUM_UDP))
			stats->classify2.nUdp += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C2_NUM_TCP))
			stats->classify2.nTcp += 1;
			
		if (map[i] & (1 << TF_STATS_BM_C2_NUM_CUSTOM))
			stats->classify2.nCustom += 1;
			
	}
	
}
			

/* Reset a buffer descriptor with linked buffer and place on the correct free buffer with 
 * descriptor queue */
int32_t testCommonRecycleLBDesc (tFramework_t *tf, Cppi_HostDesc *hd)
{
    Qmss_Queue q;

	hd->buffLen = hd->origBufferLen;
	hd->buffPtr = hd->origBuffPtr;
    hd->nextBDPtr = 0;
	
    q.qMgr = 0;
    q.qNum = tf->QDefRet;

	if (hd->buffLen == TF_LINKED_BUF_Q1_BUF_SIZE)  {
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
		Qmss_queuePush (tf->QLinkedBuf1, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);

	}  else if (hd->buffLen == TF_LINKED_BUF_Q2_BUF_SIZE)  {
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
		Qmss_queuePush (tf->QLinkedBuf2, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL); 

	}  else if (hd->buffLen == TF_LINKED_BUF_Q3_BUF_SIZE)  {
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
		Qmss_queuePush (tf->QLinkedBuf3, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);

	}  else
		return (-1);
		
	return (0);
}

/* Provide an add srio configuration to the PA sub-system */
Cppi_HostDesc *testCommonAddSrio (tFramework_t *tf, int index, paSrioInfo_t *srioInfo, uint16_t nextHdr, uint16_t nextHdrOffset,
                                 paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, paHandleL2L3_t *l2handle, int32_t Qrecycle, int32_t QCmdMem, 
 	                        	 paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_addSrio (tf->passHandle,
                         index,
					     srioInfo,
                         nextHdr,
                         nextHdrOffset,
					     matchRoute,
					     nfailRoute,
					     l2handle,
					     (paCmd_t) hd->buffPtr,
					     cmdSize,
					     repInfo,
					     cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}


/* Provide an add mac configuration to the PA sub-system */
Cppi_HostDesc *testCommonAddMac (tFramework_t *tf, int index, paEthInfo_t *ethInfo, paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, 
 	                       		 paHandleL2L3_t *l2handle, int32_t Qrecycle, int32_t QCmdMem, 
 	                        	 paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_addMac (tf->passHandle,
                        index,
					    ethInfo,
					    matchRoute,
					    nfailRoute,
					    l2handle,
					    (paCmd_t) hd->buffPtr,
					    cmdSize,
					    repInfo,
					    cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

Cppi_HostDesc *testCommonAddIp (tFramework_t *tf, int32_t index, paIpInfo_t *ipInfo, paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, 
 	                       		 paHandleL2L3_t *l3handle, paHandleL2L3_t linkedL2Handle, int32_t Qrecycle, int32_t QCmdMem, 
 	                        	 paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_addIp (tf->passHandle,
                       pa_LUT_INST_NOT_SPECIFIED,
                       index,
					   ipInfo,
					   linkedL2Handle,
					   matchRoute,
					   nfailRoute,
					   l3handle,
					   (paCmd_t) hd->buffPtr,
					   cmdSize,
					   repInfo,
					   cmdDest);
					   
    /* Restore the descriptor and return it on PA failure */
	if ((*paret != pa_OK) && (*paret != pa_DUP_ENTRY)) {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

Cppi_HostDesc *testCommonAddPort (tFramework_t *tf, int portSize, uint32_t destPort, paRouteInfo_t *matchRoute, paHandleL4_t *l4handle, paHandleL2L3_t *l3handle,
                                  int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;	                          
                                
    *paret =   Pa_addPort ( tf->passHandle, (uint16_t)portSize, destPort, *l3handle, FALSE, pa_PARAMS_NOT_SPECIFIED, matchRoute, *l4handle, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);
    
                    
     /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}  

Cppi_HostDesc *testCommonAddPort2 (tFramework_t *tf, int portSize, uint32_t destPort, Uint16 fReplace, uint16_t divertQ, paRouteInfo_t *matchRoute, paHandleL4_t *l4handle, paHandleL2L3_t *l3handle,
                                   int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;	                          
                                
    *paret =   Pa_addPort ( tf->passHandle, (uint16_t)portSize, destPort, *l3handle, fReplace, divertQ, matchRoute, *l4handle, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);
                    
     /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}  


Cppi_HostDesc *testCommonAddCustomL3 (tFramework_t *tf, uint16_t customIndex, uint8_t match[], paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute,
                                      paHandleL2L3_t *l3handle, paHandleL2L3_t linkedL2Handle,
									  int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_addCustomLUT1 (tf->passHandle, customIndex, pa_LUT_INST_NOT_SPECIFIED, pa_LUT1_INDEX_NOT_SPECIFIED, match, linkedL2Handle, matchRoute, nfailRoute, 
                               l3handle, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);	     

     /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

Cppi_HostDesc *testCommonAddCl3Config (tFramework_t *tf, uint16_t customIndex, uint16_t offset, uint16_t nextHdr, uint16_t nextHdrOffset, uint8_t masks[], 
									   int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;	     
		
	*paret = Pa_setCustomLUT1 (tf->passHandle, customIndex, offset, nextHdr, nextHdrOffset, masks, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);

    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}    
                         

Cppi_HostDesc *testCommonAddCustomL4 (tFramework_t *tf, uint16_t customIndex, uint8_t match[], paRouteInfo_t *matchRoute, paHandleL4_t l4handle, paHandleL2L3_t l3handle,
									  int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_addCustomLUT2 (tf->passHandle, customIndex,  match, l3handle, FALSE, pa_PARAMS_NOT_SPECIFIED, matchRoute, l4handle, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);	     
			
							

     /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

Cppi_HostDesc *testCommonAddCl4Config (tFramework_t *tf, Bool handleLink, uint16_t customIndex, uint16_t offsets[], uint8_t masks[], 
										int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;	     
		
	*paret = Pa_setCustomLUT2 (tf->passHandle, customIndex, handleLink, offsets, masks, 0, (paCmd_t) hd->buffPtr, cmdSize, repInfo, cmdDest);

    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}    

/* Provide an Multi-route configuration to the PA sub-system */
Cppi_HostDesc *testCommonConfigMultiRoute (tFramework_t *tf, paMultiRouteModes_e mode, uint16_t index, uint16_t nRoute, paMultiRouteEntry_t *routeEntry, 
 	                       		           int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_configMultiRoute (tf->passHandle,
                                  mode,
                                  index,
					              nRoute,
					              routeEntry,
					              (paCmd_t) hd->buffPtr,
					              cmdSize,
					              repInfo,
					              cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

/* Provide an Exception Route configuration to the PA sub-system */
Cppi_HostDesc *testCommonConfigExceptionRoute (tFramework_t *tf, int nRoute, int *routeTypes, paRouteInfo_t *eRoutes, 
 	                       		               int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_configExceptionRoute (tf->passHandle,
					                  nRoute,
                                      routeTypes,
					                  eRoutes,
					                  (paCmd_t) hd->buffPtr,
					                  cmdSize,
					                  repInfo,
					                  cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

/* Provide a Cmd Set configuration to the PA sub-system */
Cppi_HostDesc *testCommonConfigCmdSet (tFramework_t *tf, uint16_t index, int nCmd, paCmdInfo_t *cmdInfo, 
 	                       		       int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_configCmdSet (tf->passHandle,
                              index,
					          nCmd,
                              cmdInfo,
					          (paCmd_t) hd->buffPtr,
					          cmdSize,
					          repInfo,
					          cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

/* Provide a Usr Stats configuration to the PA subsystem */
Cppi_HostDesc *testCommonConfigUsrStats (tFramework_t *tf, paUsrStatsConfigInfo_t *cfgInfo, 
 	                       		         int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
    *paret = pa_OK;
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
    {
		return (NULL);
    }    
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_configUsrStats (tf->passHandle,
					            cfgInfo,
					            (paCmd_t) hd->buffPtr,
					            cmdSize,
					            repInfo,
					            cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}
                                       
                                       


/* Provide a CRC Engine configuration to the PA sub-system */
Cppi_HostDesc *testCommonConfigCrcEngine (tFramework_t *tf, uint16_t index, paCrcConfig_t *cfgInfo, 
 	                       		          int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_configCrcEngine (tf->passHandle,
                                 index,
					             cfgInfo,
					             (paCmd_t) hd->buffPtr,
					             cmdSize,
					             repInfo,
					             cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

/* Provide a Global configuration to the PA sub-system */
Cppi_HostDesc *testCommonGlobalConfig (tFramework_t *tf, paCtrlInfo_t *cfgInfo, 
 	                       		       int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *repInfo, int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_control (tf->passHandle,
					     cfgInfo,
					     (paCmd_t) hd->buffPtr,
					     cmdSize,
					     repInfo,
					     cmdDest);
			
    /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}


Cppi_HostDesc *testCommonDelHandle (tFramework_t *tf, paHandleL2L3_t *paHdl, int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *cmdReply, 
									int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_delHandle (tf->passHandle, paHdl, (paCmd_t)hd->buffPtr, cmdSize, cmdReply, cmdDest);
	
	 /* Restore the descriptor and return it on PA failure */
	if (*paret != pa_OK)  {
		hd->buffLen = hd->origBufferLen;
		Qmss_queuePush (QCmdMem, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
		return (NULL);
	}
	
#ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}

Cppi_HostDesc *testCommonDelL4Handles (tFramework_t *tf, paHandleL4_t l4Handle, int32_t Qrecycle, int32_t QCmdMem, paCmdReply_t *reply, 
									   int32_t *cmdDest, uint16_t *cmdSize, paReturn_t *paret)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	uint32_t		   psCmd;
	
	/* Pop a descriptor with a linked buffer to create the command */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (QCmdMem)) & ~15);
	if (hd == NULL)
		return (NULL);
		
	*cmdSize = hd->origBufferLen;
	
	*paret = Pa_delL4Handle (tf->passHandle, l4Handle, (paCmd_t)hd->buffPtr, cmdSize, reply, cmdDest);
	
 #ifdef PA_SIM_BUG_4BYTES
  	*cmdSize = (*cmdSize+3)& ~3;
#endif

	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = Qrecycle;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Mark the packet as a configuration packet */
  	psCmd = PASAHO_PACFG_CMD;
    Cppi_setPSData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t *)&psCmd, 4);
  	
    hd->buffLen = *cmdSize;
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, *cmdSize);
  	
  	return (hd);
}     


uint16_t testCommonOnesCompAdd (uint16_t v1, uint16_t v2)
{
	uint32_t v3;
	v3 = v1 + v2;
	v3 = (v3 & 0xffff) + (v3 >> 16);
	v3 = (v3 & 0xffff) + (v3 >> 16);
	return ((uint16_t)v3);
}
	     	
	     	

int32_t testCommonWaitCmdReply (tFramework_t *tf, paTest_t *pat, char *tname, int32_t Qcmd, uint32_t swinfo0, int32_t line)
{
	Cppi_HostDesc *hd;
	uint32_t        *swinfo;
    paEntryHandle_t	reth;
    paReturn_t     paret;
    int32_t			   htype;
    int32_t            cmdDest;
	int32_t 		   i;
	
	for (i = 0; i < 100; i++)  {
		if (Qmss_getQueueEntryCount(Qcmd) > 0)  {
			
			hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (Qcmd)) & ~15);

            Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swinfo);
            
            if (swinfo[0] != swinfo0)  {
                System_printf ("%s (%s:%d): found packet in command reply queue with swifo[0] = 0x%08x (expected 0x%08x)\n", tname, __FILE__, line, swinfo[0], swinfo0);
                testCommonRecycleLBDesc (tf, hd);
                continue;
            }

            paret = Pa_forwardResult (tf->passHandle, (void *)hd->buffPtr, &reth, &htype, &cmdDest);
            if (paret != pa_OK)  
                System_printf ("%s (%s:%d): Pa_forwardResult returned error %d\n", tname, __FILE__, line, paret);
            
            testCommonRecycleLBDesc (tf, hd); 
                
            return (0);

        }  else  {
            
          utilCycleDelay (500);

        }
	}
	
	return (-1);

}	

static paUsrStats_t  pauUsrStats;

/* Request the stats, compare to expected value */
paTestStatus_t testCommonCheckUsrStats (tFramework_t *tf, paTest_t *pat, char *tName, paUsrStats_t *expected, int32_t qSource, int32_t qCmdRecycle, int32_t qReply, Bool clear)
{ 	
	Cppi_HostDesc  *hd;
	paUsrStats_t   *usrStats = &pauUsrStats;
	Int             i;
	paTestStatus_t  status;

	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
 	cmdReply.queue   = (uint16_t) qReply;
    
    memset(usrStats, 0, sizeof(paUsrStats_t)); 
 	
 	/* Check the PA stats to make sure they are set as expected */
 	cmdReply.replyId = TEST_COMMON_STATS_REQ_ID;
 	if (testCommonRequestUsrStats (tName, tf, clear, qSource, qCmdRecycle, &cmdReply, usrStats))  {
 		System_printf ("%s (%s:%d): testCommonRequestPaStats failed\n", tName, __FILE__, __LINE__);
 		return (PA_TEST_FAILED);
 	}
 	
    if (clear)
    {
 	    /* Wait for the stats reply */
	    for (i = 0; i < 100; i++)  {
		    utilCycleDelay (1000);
		    if (Qmss_getQueueEntryCount (qReply) > 0)
			    break;
	    }
	
	    if (i == 100)  {
		    System_printf ("%s (%s:%d): Did not find response from PA to stats request command\n", tName, __FILE__, __LINE__);
		    return (PA_TEST_FAILED);
	    }
 	
	    /* Recycle the descriptor/buffer returned from the stats request */
	    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (qCmdRecycle)) & ~15);
	    if (hd == NULL)  {
		    System_printf ("%s (%s:%d): Did not find returned descriptor/buffer from stats request\n", tName, __FILE__, __LINE__);
		    return (PA_TEST_FAILED);
	    }
	
	    if (testCommonRecycleLBDesc (tf, hd))  {
		    System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats request\n", tName, __FILE__, __LINE__);
		    return (PA_TEST_FAILED);
	    }

 	    /* Format the stats response and compare to the expected results */
	    hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (qReply)) & ~15);
        
        /* Recycle the descriptor and associated buffer back to queue from which it came */
	    if (testCommonRecycleLBDesc (tf, hd))  {
		    System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats response\n", tName, __FILE__, __LINE__);
		    return (PA_TEST_FAILED);
	    }
    }
	  
    if (testCommonCompareUsrStats (tName, expected, usrStats))
    	status = PA_TEST_FAILED;
    else
    	status = PA_TEST_PASSED;   
    	
    if (clear)
        memset (expected, 0, sizeof(paUsrStats_t));
	
	return (status);
}	     	



/* Request the stats, compare to expected value */
paTestStatus_t testCommonCheckStats (tFramework_t *tf, paTest_t *pat, char *tName, paSysStats_t *expected, int32_t qSource, int32_t qCmdRecycle, int32_t qReply, Bool clear)
{ 	
	Cppi_HostDesc  *hd;
	uint8_t		   *bp;
	paSysStats_t     *paStats;
 	uint32_t			blen;
	int32_t             i;
	paTestStatus_t  status;

	paCmdReply_t cmdReply = {  pa_DEST_HOST,			/* Dest */
 							   0,						/* Reply ID (returned in swinfo0) */
 							   0,						/* Queue */
 							   0 };						/* Flow ID */
 							   
 	cmdReply.queue   = (uint16_t) qReply;
 							   
 	
 	/* Check the PA stats to make sure they are set as expected */
 	cmdReply.replyId = TEST_COMMON_STATS_REQ_ID;
 	if (testCommonRequestPaStats (tName, tf, clear, qSource, qCmdRecycle,  &cmdReply))  {
 		System_printf ("%s (%s:%d): testCommonRequestPaStats failed\n", tName, __FILE__, __LINE__);
 		return (PA_TEST_FAILED);
 	}
 	
 	/* Wait for the stats reply */
	for (i = 0; i < 100; i++)  {
		utilCycleDelay (1000);
		if (Qmss_getQueueEntryCount (qReply) > 0)
			break;
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Did not find response from PA to stats request command\n", tName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
 	
	/* Recycle the descriptor/buffer returned from the stats request */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (qCmdRecycle)) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Did not find returned descriptor/buffer from stats request\n", tName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
	
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats request\n", tName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}

 		
 	/* Format the stats response and compare to the expected results */
	hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (qReply)) & ~15);
	Cppi_getData (Cppi_DescType_HOST, (Cppi_Desc *)hd, &bp, &blen);
	paStats = Pa_formatStatsReply (tf->passHandle, (paCmd_t)bp);
	  
    if (testCommonCompareStats (tName, expected, paStats))
    	status = PA_TEST_FAILED;
    else
    	status = PA_TEST_PASSED;   
    	
    if (clear)
        memset (expected, 0, sizeof(paSysStats_t));
    	  
     /* Recycle the descriptor and associated buffer back to queue from which it came */
	if (testCommonRecycleLBDesc (tf, hd))  {
		System_printf ("%s (%s:%d): Failed to find original free buffer Q for stats response\n", tName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
	}
	
	return (status);
}	     	

int32_t commonFifoPushElement (pauFifo_t *fifo, uint32_t elem)
{
	int32_t nextIn;
	
	nextIn = fifo->inIdx + 1;
	if (nextIn >= fifo->size)
		nextIn = 0;
	
	/* Check if the fifo is full */
	if (nextIn == fifo->outIdx)
		return (-1);
		
	/* Add the element */
	fifo->data[fifo->inIdx] = elem;
	fifo->inIdx = nextIn;
	
	return (0);
}

int32_t commonFifoGetCount (pauFifo_t *fifo)
{
	if (fifo->inIdx >= fifo->outIdx)
		return (fifo->inIdx - fifo->outIdx);
	else
		return (fifo->size + fifo->inIdx - fifo->outIdx);
}

uint32_t commonFifoPopElement (pauFifo_t *fifo, int32_t *numInQBeforePop)
{
	uint32_t v;
	
	*numInQBeforePop = commonFifoGetCount (fifo);
	if (*numInQBeforePop == 0)
		return (0);
		
	v = fifo->data[fifo->outIdx];
	fifo->outIdx = fifo->outIdx + 1;
	if (fifo->outIdx >= fifo->size)
		fifo->outIdx = 0;
		
	return (v);
}
		
	
int32_t testCommonVerifyCmdResult (tFramework_t *tf, paSysStats_t *stats, char *tfName, char *fname, int32_t fline, char *tid, Cppi_HostDesc *hd, 
								paReturn_t exp, paReturn_t act, pauTestSetupStatus_t *status, int32_t cmdDest, int32_t cmdSize)
{
	
	if (act != exp)  {
		System_printf ("%s (%s:%d - from %s:%d): %s returned %d, expected %d\n", tfName, __FILE__, __LINE__, fname, fline, tid,
						act, exp);
	
		if (hd != NULL)  {
			if (testCommonRecycleLBDesc (tf, hd))  {
				System_printf ("%s (%s:%d - from %s:%d): Failed to find original free buffer Q for stats request\n", tfName, 
						__FILE__, __LINE__, fname, fline);
			}
		}
		
		/* Don't change the status of the entry */
		return (-1);
		
	}  else  {
		
		if (act == pa_OK)  {
			if (hd == NULL)  {
				System_printf ("%s (%s:%d - from %s:%d): %s returned OK but no buffer descriptor found\n",
						tfName, __FILE__, __LINE__, fname, fline), tid;
				return (-1);
			}
			
			Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
			*status = PAU_TEST_SETUP_STATUS_CMD_SENT;
			if ((cmdDest == pa_CMD_TX_DEST_0) || (cmdDest == pa_CMD_TX_DEST_1) || (cmdDest == pa_CMD_TX_DEST_2))
				stats->classify1.nPackets += 1;
			if ((cmdDest == pa_CMD_TX_DEST_3))
				stats->classify2.nPackets += 1;
			return (0);
		
		}  else  {
			
			/* The descriptor was already returned if the pa result code was not OK */
			*status = PAU_TEST_SETUP_STATUS_CMD_NO_CMD;  /* Command is complete */
		
			return (0);
		}
	}
	
}
	


int32_t testCommonMacSetup (tFramework_t *tf, paSysStats_t *stats, pauTestMacSetup_t *maci, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	/* Set the reply ID to indicate the type and index of the setup */
	rep->replyId = TF_COMMON_CMD_ID_ADD_MAC + idx;
	
	hd = testCommonAddMac (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &maci->ethInfo, &maci->matchRoute, &maci->nFail, &maci->handle, 
						   maci->bufferQ, maci->bufferQ, rep, &cmdDest, &cmdSize, &paret);
						   
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_addMac", hd, maci->paret, paret, &maci->status, cmdDest, cmdSize));
}
						   
		

int32_t testCommonIpSetup (tFramework_t *tf, paSysStats_t *stats, pauTestIpSetup_t *ipi, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	paHandleL2L3_t linkHandle = NULL;
	
	/* Set the reply ID to indicate the type and index of the reply */
	rep->replyId = TF_COMMON_CMD_ID_ADD_IP + idx;
	
	
	/* Verify a valid previous link */
	if ((ipi->l2Link != NULL) && (ipi->l3Link == NULL))  {
		if (ipi->l2Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED)  {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to L2 handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = ipi->l2Link->handle;
		
	} else if ((ipi->l2Link == NULL) && (ipi->l3Link != NULL))  {
		if (ipi->l3Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED)  {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to L3 handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = ipi->l3Link->handle;
		
	}  else if ((ipi->l2Link != NULL) && (ipi->l3Link != NULL))  {
		System_printf ("%s: (%s:%d - from %s:%d): Invalid IP configuration - both L2 and L3 links specified\n",
					tfName, __FILE__, __LINE__, fname, fline);
		return (-1);
	}
	
	
	hd = testCommonAddIp (tf, pa_LUT1_INDEX_NOT_SPECIFIED, &ipi->ipInfo, &ipi->matchRoute, &ipi->nFail, &ipi->handle, linkHandle, ipi->bufferQ,
						  ipi->bufferQ, rep, &cmdDest, &cmdSize, &paret);
						  
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_addIp", hd, ipi->paret, paret, &ipi->status, cmdDest, cmdSize));
						  
}


int32_t testCommonL4Setup (tFramework_t *tf, paSysStats_t *stats, pauTestL4Setup_t *l4i, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	paHandleL2L3_t linkHandle = NULL;
	
	/* Set the reply ID to indicate the type and index of the reply */
	rep->replyId = TF_COMMON_CMD_ID_ADD_PORT + idx;
	
	/* Verify that the linked handle is valid */
	if (l4i->l3Link != NULL)  {
		if (l4i->l3Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED)  {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to UDP handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = l4i->l3Link->handle;
	}	

	hd = testCommonAddPort (tf, pa_LUT2_PORT_SIZE_16, (uint32_t)l4i->destPort, &l4i->matchRoute, &l4i->handle, &linkHandle, l4i->bufferQ, l4i->bufferQ,
							rep, &cmdDest, &cmdSize, &paret);
	
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_addPort", hd, l4i->paret, paret, &l4i->status, cmdDest, cmdSize));
	
	
}

int32_t testCommonCl3Config (tFramework_t *tf, paSysStats_t *stats, pauTestCl3Config_t *l3cfg, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	rep->replyId = TF_COMMON_CMD_ID_CFG_CL3 + idx;
	
	
	hd = testCommonAddCl3Config (tf, l3cfg->custIndex, l3cfg->offset, l3cfg->nextHdr, l3cfg->nextHdrOffset, l3cfg->byteMasks, l3cfg->bufferQ,
								 l3cfg->bufferQ, rep, &cmdDest, &cmdSize, &paret);
								
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_setCustomLUT1", hd, l3cfg->paret, paret, &l3cfg->status, cmdDest, cmdSize));

}	

int32_t testCommonCl3Setup (tFramework_t *tf, paSysStats_t *stats, pauTestCl3Setup_t *l3s, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	paHandleL2L3_t linkHandle = NULL;
	
	rep->replyId = TF_COMMON_CMD_ID_ADD_CL3 + idx;
	
	/* Verify that the linked handle is valid */
	/* Verify a valid previous link */
	if ((l3s->l2Link != NULL) && (l3s->l3Link == NULL))  {
		if (l3s->l2Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED)  {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to L2 handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = l3s->l2Link->handle;
		
	} else if ((l3s->l2Link == NULL) && (l3s->l3Link != NULL))  {
		if (l3s->l3Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED)  {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to L3 handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = l3s->l3Link->handle;
		
	}  else if ((l3s->l2Link != NULL) && (l3s->l3Link != NULL))  {
		System_printf ("%s: (%s:%d - from %s:%d): Invalid IP configuration - both L2 and L3 links specified\n",
					tfName, __FILE__, __LINE__, fname, fline);
		return (-1);
	}
    
	hd = testCommonAddCustomL3 (tf, l3s->custIndex, l3s->match, &l3s->matchRoute, &l3s->nFail, &l3s->handle,  linkHandle, l3s->bufferQ, l3s->bufferQ, rep,
								&cmdDest, &cmdSize, &paret);
	
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_addCustomLUT1", hd, l3s->paret, paret, &l3s->status, cmdDest, cmdSize));

}



	
int32_t testCommonCl4Config (tFramework_t *tf, paSysStats_t *stats, pauTestCl4Config_t *l4cfg, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	rep->replyId = TF_COMMON_CMD_ID_CFG_CL4 + idx;
	
	
	hd = testCommonAddCl4Config (tf, l4cfg->handleLink, l4cfg->custIndex, l4cfg->byteOffsets, l4cfg->byteMasks, l4cfg->bufferQ,
								l4cfg->bufferQ, rep, &cmdDest, &cmdSize, &paret);
								
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_setCustomLUT2", hd, l4cfg->paret, paret, &l4cfg->status, cmdDest, cmdSize));

}	

int32_t testCommonCl4Setup (tFramework_t *tf, paSysStats_t *stats, pauTestCl4Setup_t *l4s, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	paHandleL2L3_t linkHandle = NULL;
	
	rep->replyId = TF_COMMON_CMD_ID_ADD_CL4 + idx;
	
	/* Verify that the linked handle is valid */
	if (l4s->l3Link != NULL)  {
		if (l4s->l3Link->status != PAU_TEST_SETUP_STATUS_CMD_REPLIED) {
			System_printf ("%s: (%s:%d - from %s:%d): IP linked to Custom L4 handle that is not in CMD_REPLIED state\n",
							tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		linkHandle = l4s->l3Link->handle;
		
	}	
	
	hd = testCommonAddCustomL4 (tf, l4s->custIndex, l4s->match, &l4s->matchRoute, l4s->handle,  linkHandle, l4s->bufferQ, l4s->bufferQ, rep,
								&cmdDest, &cmdSize, &paret);
	
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_addCustomLUT2", hd, l4s->paret, paret, &l4s->status, cmdDest, cmdSize));

}


int32_t testCommonCmdSetSetup (tFramework_t *tf, paSysStats_t *stats, pauTestCmdSetSetup_t *cmdSet, paCmdReply_t *rep, char *tfName, char *fname, int32_t fline, int32_t idx)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	rep->replyId = TF_COMMON_CMD_ID_ADD_CMDSET + idx;
	
	hd =  testCommonConfigCmdSet(tf, cmdSet->index, cmdSet->nCmds, cmdSet->cmdInfo, cmdSet->bufferQ, cmdSet->bufferQ, rep,
								 &cmdDest, &cmdSize, &paret);
	
	return (testCommonVerifyCmdResult (tf, stats, tfName, fname, fline, "Pa_configCmdSet", hd, cmdSet->paret, paret, &cmdSet->status, cmdDest, cmdSize));

}

int32_t testCommonCloseL4 (tFramework_t *tf, paSysStats_t *stats, paHandleL4_t handle, pauTestSetupStatus_t *status, paCmdReply_t *rep, int32_t Q, char *tfName, char *fname, int32_t fline)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	hd = testCommonDelL4Handles (tf, handle, Q, Q, rep, &cmdDest, &cmdSize, &paret);
	
	if ((hd == NULL) || paret != pa_OK)  {
		System_printf ("%s (%s:%d from %s:%d): close L4 handle failed, paret = %d\n", tfName, __FILE__, __LINE__, fname, fline, paret);
		return (-1);
	}
	
	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
	stats->classify2.nPackets += 1;
	
	return (0);
}


int32_t testCommonCloseL2L3 (tFramework_t *tf, paSysStats_t *stats, paHandleL2L3_t *handle, pauTestSetupStatus_t *status, paCmdReply_t *rep, int32_t Q, char *tfName, char *fname, int32_t fline)
{
	Cppi_HostDesc *hd;
	int32_t 		   cmdDest;
	uint16_t 		   cmdSize;
	paReturn_t     paret;
	
	hd = testCommonDelHandle (tf, handle, Q, Q, rep, &cmdDest, &cmdSize, &paret);
	
	if ((hd == NULL) || paret != pa_OK)  {
		System_printf ("%s (%s:%d from %s:%d): close L2/L3 handle failed, paret = %d\n", tfName, __FILE__, __LINE__, fname, fline, paret);
		return (-1);
	}
	
	stats->classify1.nPackets += 1;
	
	Qmss_queuePush (tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
	
	return (0);
}
	
	
		
	
	
/* Examine the reply queue and change command state */
void testCommonCheckComplete (tFramework_t *tf, pauTestSetup_t *setup, char *tfName, char *fname, int32_t fline)
{
	Cppi_HostDesc  *hd;
	uint32_t 		   *swinfo;
	uint32_t			swinfoType;
	uint32_t			swinfoChan;
	paReturn_t      paret;
	paEntryHandle_t	reth;
    int32_t			    htype;
    int32_t             cmdDest;
	
	pauTestSetupStatus_t *status;
	int32_t					  maxChan;
	char				 *tid;
	
	while ((hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QCommonCmdRep)) & ~15)) != NULL)  {
		
		Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint8_t **)&swinfo);
		swinfoType = swinfo[0] & 0xffff0000;
		swinfoChan = swinfo[0] & 0x0000ffff;
		
		switch (swinfoType)  {
			case TF_COMMON_CMD_ID_ADD_MAC:  status = &setup->macs[swinfoChan].status;    maxChan = setup->nMac;     tid = "Add Mac";        break;
			case TF_COMMON_CMD_ID_ADD_IP:   status = &setup->ips[swinfoChan].status;     maxChan = setup->nIps;     tid = "Add Ip";         break;
			case TF_COMMON_CMD_ID_ADD_PORT: status = &setup->l4s[swinfoChan].status;     maxChan = setup->nL4s;     tid = "Add L4";         break;
			case TF_COMMON_CMD_ID_CFG_CL3:  status = &setup->cl3cfgs[swinfoChan].status; maxChan = setup->nCl3Cfgs; tid = "CL3 config";     break;
			case TF_COMMON_CMD_ID_ADD_CL3:  status = &setup->cl3s[swinfoChan].status;    maxChan = setup->nCl3s;    tid = "Add Cl3";        break;
			case TF_COMMON_CMD_ID_DEL_CL3:  status = &setup->cl3s[swinfoChan].status;    maxChan = setup->nCl3s;    tid = "Del Cl3";		break;
			case TF_COMMON_CMD_ID_CFG_CL4:  status = &setup->cl4cfgs[swinfoChan].status; maxChan = setup->nCl4Cfgs; tid = "CL4 config";     break;
			case TF_COMMON_CMD_ID_ADD_CL4:  status = &setup->cl4s[swinfoChan].status;    maxChan = setup->nCl4s;    tid = "Add Cl4";        break;
			case TF_COMMON_CMD_ID_DEL_CL4:  status = &setup->cl4s[swinfoChan].status;    maxChan = setup->nCl4s;    tid = "Del Cl4";		break;
			case TF_COMMON_CMD_ID_DEL_PORT: status = &setup->l4s[swinfoChan].status;     maxChan = setup->nL4s;     tid = "Del L4";         break;
			case TF_COMMON_CMD_ID_DEL_IP:   status = &setup->ips[swinfoChan].status;     maxChan = setup->nIps;     tid = "Del Ip";   		break;
			case TF_COMMON_CMD_ID_DEL_MAC:  status = &setup->macs[swinfoChan].status;    maxChan = setup->nMac;     tid = "Del Mac";		break;
			case TF_COMMON_CMD_ID_ADD_CMDSET:  status = &setup->cmdSet[swinfoChan].status;    maxChan = setup->nCmdSets;    tid = "AddCmd Set";        break;
						
			default:
				System_printf ("%s: (%s:%d - from %s:%d): Found descriptor in common command rep queue with unknown swinfo0 = 0x%08x\n",
						tfName, __FILE__, __LINE__, fname, fline, swinfo[0]);
				testCommonRecycleLBDesc (tf, hd);
				continue;
		}
		
		if (swinfoChan >= maxChan)  {
			System_printf ("%s: (%s:%d - from %s:%d): Found channel %d in %s setup, but max value is %d\n",
							tfName, __FILE__, __LINE__, fname, fline, swinfoChan, tid, maxChan);
			testCommonRecycleLBDesc (tf, hd);
			continue;
		}
	
		*status = PAU_TEST_SETUP_STATUS_CMD_REPLIED;
		
		if (swinfoType != TF_COMMON_CMD_ID_CFG_CL4)  {
			paret = Pa_forwardResult (tf->passHandle, (void *)hd->buffPtr, &reth, &htype, &cmdDest);
			
			if (paret != pa_OK)  {
				System_printf ("%s: (%s:%d - from %s:%d): Pa_forwardResult returned error code %d\n",
								tfName, __FILE__, __LINE__, fname, fline, paret);
				*status = PAU_TEST_SETUP_STATUS_CMD_REP_ERR;
				
			}
		}
		
		testCommonRecycleLBDesc (tf, hd);
	}
	
}		
		


/* Returns TRUE if all entries are complete */
Bool testCommonWaitComplete (tFramework_t *tf, pauTestSetup_t *setup, char *tfName, char *fname, int32_t fline)
{
	int  i, j;
	Bool done;
	
	for (i = 0, done = FALSE; (i < 100) && (done == FALSE); i++)  {
		done = TRUE;
		
		for (j = 0; (j < setup->nMac) && (done == TRUE); j++)  
			if (setup->macs[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)  
				done = FALSE;
		
		for (j = 0; (j < setup->nIps) && (done == TRUE); j++)
			if (setup->ips[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
				
		for (j = 0; (j < setup->nL4s) && (done == TRUE); j++)
			if (setup->l4s[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
				
		for (j = 0; (j < setup->nCl3Cfgs) && (done == TRUE); j++)
			if (setup->cl3cfgs[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
				
		for (j = 0; (j < setup->nCl3s) && (done == TRUE); j++)
			if (setup->cl3s[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
                
		for (j = 0; (j < setup->nCl4Cfgs) && (done == TRUE); j++)
			if (setup->cl4cfgs[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
				
		for (j = 0; (j < setup->nCl4s) && (done == TRUE); j++)
			if (setup->cl4s[j].status == PAU_TEST_SETUP_STATUS_CMD_SENT)
				done = FALSE;
				
		if (done == FALSE)
			utilCycleDelay (100);
		else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
	return (done);
}


/* Wait for a specific command to complete */
int32_t testCommonCheckWait (tFramework_t *tf, pauTestSetup_t *setup, char *tfName, char *fname, int32_t fline, pauTestSetupStatus_t *status)
{
	int32_t i;
	
	for (i = 0; i < 100; i++)  {
		testCommonCheckComplete (tf, setup, tfName, fname, fline);
		if (*status != PAU_TEST_SETUP_STATUS_CMD_SENT)
			break;
		else
			utilCycleDelay (100);
	}
	
	if (i == 100)
		return (-1);
		
	return (0);
	
}
		
				
 	                        	                     	 	         
		
		
/* Setup PA for a test */
int32_t testCommonSetupTest (tFramework_t *tf, paSysStats_t *stats, pauTestSetup_t *setup, char *tfName, char *fname, int32_t fline)
{
	int32_t i;
	paCmdReply_t cmdReply = {  pa_DEST_HOST,						/* Dest */
 							      0,								/* Reply ID (returned in swinfo0) */
 							   	  0,								/* Queue, set by common setup */
 							      0 };		 						/* Flow ID */
 							     
	cmdReply.queue = tf->QCommonCmdRep; 							  
	
	/* Initialize the test state */
	for (i = 0; i < setup->nMac; i++)
		setup->macs[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
	
	for (i = 0; i < setup->nIps; i++)
		setup->ips[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	for (i = 0; i < setup->nL4s; i++)
		setup->l4s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	for (i = 0; i < setup->nCl3Cfgs; i++)
		setup->cl3cfgs[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	for (i = 0; i < setup->nCl3s; i++)
		setup->cl3s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
        
	for (i = 0; i < setup->nCl4Cfgs; i++)
		setup->cl4cfgs[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	for (i = 0; i < setup->nCl4s; i++)
		setup->cl4s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
	
	/* MAC setup */
	for (i = 0; i < setup->nMac; i++)  {
		if (testCommonMacSetup (tf, stats, &(setup->macs[i]), &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (MAC) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->macs[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->macs[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): MAC setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else 
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
						
			
	
	/* IP Setup */
	for (i = 0; i < setup->nIps; i++)  {
		if (testCommonIpSetup (tf, stats, &(setup->ips[i]), &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (IP) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->ips[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->ips[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): IP setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else 
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
	/* Custom L3 config */
	for (i = 0; i < setup->nCl3Cfgs; i++)  {
		if (testCommonCl3Config (tf, stats, &(setup->cl3cfgs[i]), &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responces (L3 Custom Config) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl3cfgs[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl3cfgs[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L3 config failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
			
	}
	
	/* Custom L3 setup */
	for (i = 0; i < setup->nCl3s; i++)  {
		if (testCommonCl3Setup (tf, stats, &setup->cl3s[i], &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (L3 Custom Setup) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl3s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl3s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L3 setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
    
	/* L4 Setup */
	for (i = 0; i < setup->nL4s; i++)  {
		if (testCommonL4Setup (tf, stats, &(setup->l4s[i]), &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (L4) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->l4s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->l4s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): L4 setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
            
        /* Wait a while for the next L4 entry */
        utilCycleDelay (1000);    
	}
    
	/* Custom L4 config */
	for (i = 0; i < setup->nCl4Cfgs; i++)  {
		if (testCommonCl4Config (tf, stats, &(setup->cl4cfgs[i]), &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responces (Custom L4 Config) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl4cfgs[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl4cfgs[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L4 config failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
			
	}
	
	/* Custom L4 setup */
	for (i = 0; i < setup->nCl4s; i++)  {
		if (testCommonCl4Setup (tf, stats, &setup->cl4s[i], &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (Custom L4) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl4s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl4s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L4 setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
            
        /* Wait a while for the next L4 entry */
        utilCycleDelay (1000);    
            
	}
    
	/* Command Set setup */
	for (i = 0; i < setup->nCmdSets; i++)  {
		if (testCommonCmdSetSetup (tf, stats, &setup->cmdSet[i], &cmdReply, tfName, fname, fline, i))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses (Command Set Configuration) not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cmdSet[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cmdSet[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Command Set setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
            
        /* Wait a while for the next L4 entry */
        utilCycleDelay (1000);    
            
	}

	if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
		System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
	
	
	return (0);
}
	
/* Teardown a test which was setup using testCommonSetupTest */
int32_t testCommonTeardownTest (tFramework_t *tf, paSysStats_t *stats, pauTestSetup_t *setup, char *tfName, char *fname, int32_t fline)
{
	int32_t i;
	paCmdReply_t cmdReply = {  pa_DEST_HOST,						/* Dest */
 							      0,								/* Reply ID (returned in swinfo0) */
 							   	  0,								/* Queue, set by common setup */
 							      0 };		 						/* Flow ID */
 							     
	cmdReply.queue = tf->QCommonCmdRep; 							  
	
	/* Only teardown channels that were active */
	for (i = 0; i < setup->nMac; i++)
		if (setup->macs[i].status == PAU_TEST_SETUP_STATUS_CMD_REPLIED)
			setup->macs[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
	
	for (i = 0; i < setup->nIps; i++)
		if (setup->ips[i].status == PAU_TEST_SETUP_STATUS_CMD_REPLIED)
			setup->ips[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	for (i = 0; i < setup->nCl3s; i++)
		if (setup->cl3s[i].status == PAU_TEST_SETUP_STATUS_CMD_REPLIED)
			setup->cl3s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
            
	for (i = 0; i < setup->nL4s; i++)
		if (setup->l4s[i].status == PAU_TEST_SETUP_STATUS_CMD_REPLIED)
			setup->l4s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
		
	/* There is no teardown for custom L4 config */
		
	for (i = 0; i < setup->nCl4s; i++)
		if (setup->cl4s[i].status == PAU_TEST_SETUP_STATUS_CMD_REPLIED)
			setup->cl4s[i].status = PAU_TEST_SETUP_STATUS_CMD_NOT_SENT;
			
	/* Custom L4 teardown */
	for (i = 0; i < setup->nCl4s; i++)  {
		cmdReply.replyId = TF_COMMON_CMD_ID_DEL_CL4 + i;
		if (testCommonCloseL4 (tf, stats, setup->cl4s[i].handle, &setup->cl4s[i].status, &cmdReply, setup->cl4s[i].bufferQ, tfName, fname, fline))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl4s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl4s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L4 setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
	/* L4 Teardown */
	for (i = 0; i < setup->nL4s; i++)  {
		cmdReply.replyId = TF_COMMON_CMD_ID_DEL_PORT + i;
		if (testCommonCloseL4 (tf, stats, setup->l4s[i].handle, &setup->l4s[i].status, &cmdReply, setup->l4s[i].bufferQ, tfName, fname, fline))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->l4s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->l4s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): L4 setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
	/* IP Teardown */
	for (i = 0; i < setup->nIps; i++)  {
		cmdReply.replyId = TF_COMMON_CMD_ID_DEL_IP + i;
		if (testCommonCloseL2L3 (tf, stats, &setup->ips[i].handle, &setup->ips[i].status, &cmdReply, setup->ips[i].bufferQ, tfName, fname, fline))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->ips[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->ips[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): IP setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else 
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
    
	/* Custom L3 teardown */
	for (i = 0; i < setup->nCl3s; i++)  {
		cmdReply.replyId = TF_COMMON_CMD_ID_DEL_CL3 + i;
		if (testCommonCloseL2L3 (tf, stats, &setup->cl3s[i].handle, &setup->cl3s[i].status, &cmdReply, setup->cl3s[i].bufferQ, tfName, fname, fline))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->cl3s[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->cl3s[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): Custom L3 teardown failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
    
			
	/* MAC teardown */
	for (i = 0; i < setup->nMac; i++)  {
        if(setup->macs[i].paret != pa_OK)continue;
		cmdReply.replyId = TF_COMMON_CMD_ID_DEL_MAC + i;
		if (testCommonCloseL2L3 (tf, stats, &setup->macs[i].handle, &setup->macs[i].status, &cmdReply, setup->macs[i].bufferQ, tfName, fname, fline))  {
			if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
				System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
			return (-1);
		}
		
		if (setup->macs[i].waitDone == TRUE)  {
			if (testCommonCheckWait (tf, setup, tfName, fname, fline, &setup->macs[i].status) == -1)
				System_printf ("%s: (%s:%d - called from %s:%d): MAC setup failed waiting for item %d to complete\n",
								tfName, __FILE__, __LINE__, fname, fline, i);
		}  else 
			testCommonCheckComplete (tf, setup, tfName, fname, fline);
	}
	
	if (testCommonWaitComplete (tf, setup, tfName, fname, fline) == FALSE)
		System_printf ("%s: (%s:%d - called from %s:%d): PA responses not complete\n", tfName, __FILE__, __LINE__, fname, fline);
	
	
	return (0);
	
}
	

	
/* Packets are taken off of the free descriptor Q and automatically returned after the transfer */
int32_t testCommonSendPackets (tFramework_t *tf, char *tfName, paSysStats_t *stats, pktTestInfo_t *pktInfo, int32_t nPackets, int32_t dest)
{
	Cppi_HostDesc  *hd;
	Qmss_Queue      q;
	int32_t i;
	
	
	for (i = 0; i < nPackets; i++)  {
		
		hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (tf->QfreeDesc)) & ~15);
		if (hd == NULL)  {
			System_printf ("%s (%s:%d): Failed to pop a descriptor off free descriptor queue %d\n", 
							tfName, __FILE__, __LINE__, tf->QfreeDesc);
			return (-1);
		}
		
			/* Setup the return for the descriptor */
  		q.qMgr = 0;
  		q.qNum = tf->QfreeDesc;
  		Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
        
  	    /* Make sure there is no control info.  */
  	    Cppi_setPSLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, 0);
  		
  		Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, pktInfo[i].pkt, (uint32_t)pktInfo[i].pktLen);
  		Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, (uint32_t)pktInfo[i].pktLen);
  		
  		Qmss_queuePush (tf->QPaTx[dest], (Ptr)hd, pktInfo[i].pktLen, TF_SIZE_DESC, Qmss_Location_TAIL);
  		
  		testCommonIncStats (pktInfo[i].statsMap, stats);
	}
	
	
	return (0);
}
	
		
		

/* Function used for debugging firmware */
#include <ti/csl/cslr_device.h>
#include <ti/csl/cslr_pa_ss.h>

void mdebugHaltPdsp (int32_t pdspNum)
{
    CSL_Pa_ssRegs *passRegs = (CSL_Pa_ssRegs *)CSL_PA_SS_CFG_REGS; 
	passRegs->PDSP_CTLSTAT[pdspNum].PDSP_CONTROL &= ~(CSL_PA_SS_PDSP_CONTROL_PDSP_ENABLE_MASK);

}

void mdebugRunPdsp (int32_t pdspNum)
{
    CSL_Pa_ssRegs *passRegs = (CSL_Pa_ssRegs *)CSL_PA_SS_CFG_REGS; 
	passRegs->PDSP_CTLSTAT[pdspNum].PDSP_CONTROL |= (CSL_PA_SS_PDSP_CONTROL_PDSP_ENABLE_MASK);

}



  	
		
	

