/**  
 * @file cppi_qmss_mgmt.c
 *
 * @brief 
 *  This file holds all  required routines to configure CPPI/QMSS LLDs and 
 *  to send/receive data using PA/QM.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include "fw_test.h"
#include <ti/drv/qmss/qmss_qm.h>
#include <ti/drv/qmss/device/qmss_device.c>
#include <ti/drv/cppi/device/cppi_device.c>


/* QMSS device specific configuration */
extern Qmss_GlobalConfigParams qmssGblCfgParams[];

/* CPPI device specific configuration */
extern Cppi_GlobalConfigParams  cppiGblCfgParams[];




/* Host Descriptor Region - [Size of descriptor * Number of descriptors] 
 * MUST be 16 byte aligned.
 */
uint8_t *pHostDesc = 0;

/* Buffers to be used for TX */
uint8_t *pCppiMemTX = 0;


/* QMSS queue handles */

/* Queue with free descriptors */
Qmss_QueueHnd                           gGlobalFreeQHnd;   

/* TX queues used to send data to PA PDSP/CPSW.*/
Qmss_QueueHnd                           gPaTxQHnd [NUM_PA_TX_QUEUES];

/* TX queue with free decriptors attached to data buffers for transmission.*/
Qmss_QueueHnd                           gTxFreeQHnd;

/* RX queue with free decriptors attached to data buffers to be used
 * by the PASS CPDMA to hold the received data.
 */
Qmss_QueueHnd                           gRxFreeQHnd;

/* RX queue used by the application to receive packets from PASS/CPSW.
 * Each core has an independent RX queue. 
 */
Qmss_QueueHnd                           gRxQHnd;


/* CPPI Handles used by the application */
Cppi_Handle                             gCpdmaHnd;

Cppi_ChHnd                              gCpdmaTxChanHnd [NUM_PA_TX_QUEUES];

Cppi_ChHnd                              gCpdmaRxChanHnd [NUM_PA_RX_CHANNELS];  

Cppi_FlowHnd                            gRxFlowHnd;

/* Static data packet being used for TX and RX verification
 */
#define PACKET_UDP_DEST_PORT_SHIFT  36
#define PACKET_PAYLOAD_SHIFT        42
#pragma DATA_ALIGN(pktMatch, CACHE_LINESZ)
uint8_t pktMatch[] = {
							0x10, 0x11, 0x12, 0x13, 0x14, 0x15,                      /* Dest MAC */
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05,                      /* Src MAC  */
                            0x08, 0x00,                                              /* Ethertype = IPv4 */
                            0x45, 0x00, 0x00, 0x6c,                                  /* IP version, services, total length */
                            0x00, 0x00, 0x00, 0x00,                                  /* IP ID, flags, fragment offset */
                            0x05, 0x11, 0x32, 0x26,                                  /* IP ttl, protocol (UDP), header checksum */
                            0xc0, 0xa8, 0x01, 0x01,                                  /* Source IP address */
                            0xc0, 0xa8, 0x01, 0x0a,                                  /* Destination IP address */
                            0x12, 0x34, 0x56, 0x78,                                  /* UDP source port, dest port */
                            0x00, 0x58, 0x1d, 0x18,                                  /* UDP len, UDP checksum */
                            0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,          /* 80 bytes of payload data */
                            0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41,
                            0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
                            0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51,
                            0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
                            0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61,
                            0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
                            0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71,
                            0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
                            0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81  };

/* Tx/Rx packet counters */
volatile uint32_t   gTxCounter = 0, gRxCounter = 0;
/******************************************************************************
* Macro to convert to IP Register Virtual Address from a mapped base Virtual Address
* Input: virtBaseAddr: Virtual base address mapped using mmap for IP
*        phyBaseAddr: Physical base address for the IP
*        phyRegAddr:  Physical register address
******************************************************************************/
static inline void* FW_GET_REG_VADDR (void * virtBaseAddr, uint32_t phyBaseAddr, uint32_t phyRegAddr)
{
    return((void *)((uint8_t *)virtBaseAddr + (phyRegAddr - phyBaseAddr)));
}

/** ============================================================================
 *   @n@b Init_Qmss
 *
 *   @b Description
 *   @n This API initializes the QMSS LLD on core 0 only.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t Init_Qmss (void)
{
    int32_t                     result;
    Qmss_MemRegInfo             memCfg;
    Qmss_InitCfg                qmssInitConfig;
    Cppi_DescCfg                cppiDescCfg;
    uint32_t                    numAllocated;
    Qmss_GlobalConfigParams     fw_qmssGblCfgParams;

    /* Initialize QMSS */
    memset (&qmssInitConfig, 0, sizeof (Qmss_InitCfg));

    /* Set up QMSS configuration */

    /* Use internal linking RAM */
    qmssInitConfig.linkingRAM0Base  =   0;   
    qmssInitConfig.linkingRAM0Size  =   0;
    qmssInitConfig.linkingRAM1Base  =   0x0;
    qmssInitConfig.maxDescNum       =   NUM_HOST_DESC;

    /* Bypass hardware initialization as it is done within Kernel */
    qmssInitConfig.qmssHwStatus     =   QMSS_HW_INIT_COMPLETE;

    fw_qmssGblCfgParams = qmssGblCfgParams[0];

    /* Convert address to Virtual address */ 
    fw_qmssGblCfgParams.qmConfigReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CONFIG_STARVATION_COUNTER_REGS);

    fw_qmssGblCfgParams.qmDescReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_DESCRIPTION_REGS);

    fw_qmssGblCfgParams.qmQueMgmtReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueMgmtProxyReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_PROXY_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueStatReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QUE_PEEK_REGS);
    
    fw_qmssGblCfgParams.qmQueIntdReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_INTD_REGS);

    fw_qmssGblCfgParams.qmPdspCmdReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM1_REGS);
  
    fw_qmssGblCfgParams.qmPdspCmdReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM2_REGS);
        
    fw_qmssGblCfgParams.qmPdspCtrlReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP1_REGS);

    fw_qmssGblCfgParams.qmPdspCtrlReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP2_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP1_RAM_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP2_RAM_REGS);

    fw_qmssGblCfgParams.qmStatusRAM = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_STATUS_RAM_REGS);

    fw_qmssGblCfgParams.qmLinkingRAMReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_LINKING_RAM_REGS);

    fw_qmssGblCfgParams.qmMcDMAReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_MCDMA_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER1_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER2_REGS);

    fw_qmssGblCfgParams.qmQueMgmtDataReg = (void *)((uint32_t)fw_qmssDataVaddr);

    fw_qmssGblCfgParams.qmQueMgmtProxyDataReg = 
        FW_GET_REG_VADDR(fw_qmssDataVaddr,QMSS_DATA_BASE_ADDR,QMSS_DATA_BASE_QUEUE_PROXY_ADDR);

    /* Initialize the Queue Manager */
    result = Qmss_init (&qmssInitConfig, &fw_qmssGblCfgParams);
    if (result != QMSS_SOK)
    {
        System_printf ("Error initializing Queue Manager SubSystem, Error code : %d\n", result);
        return -1;
    }

    /* Start Queue manager on this core */
    Qmss_start ();

    /* Setup the descriptor memory regions. 
     *
     * The Descriptor base addresses MUST be global addresses and
     * all memory regions MUST be setup in ascending order of the
     * descriptor base addresses.
     */
    pHostDesc = (uint8_t*)fw_memAlloc((NUM_HOST_DESC *
                                       SIZE_HOST_DESC),
                                       CACHE_LINESZ);

    /* Initialize and setup CPSW Host Descriptors required for example */
    memset (pHostDesc, 0, SIZE_HOST_DESC * NUM_HOST_DESC);
    memCfg.descBase             =   (uint32_t *) pHostDesc;
    memCfg.descSize             =   SIZE_HOST_DESC;
    memCfg.descNum              =   NUM_HOST_DESC;
    memCfg.manageDescFlag       =   Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    memCfg.memRegion            =   Qmss_MemRegion_MEMORY_REGION0;
    memCfg.startIndex           =   0;

    /* Insert Host Descriptor memory region */
    result = Qmss_insertMemoryRegion(&memCfg);
    if (result == QMSS_MEMREGION_ALREADY_INITIALIZED)
    {
        System_printf ("Memory Region %d already Initialized \n", memCfg.memRegion);
    }
    else if (result < QMSS_SOK)
    {
        System_printf ("Error: Inserting memory region %d, Error code : %d\n", memCfg.memRegion, result);
        return -1;
    }    

    /* Initialize all the descriptors we just allocated on the
     * memory region above. Setup the descriptors with some well
     * known values before we use them for data transfers.
     */
    memset (&cppiDescCfg, 0, sizeof (cppiDescCfg));
    cppiDescCfg.memRegion       =   Qmss_MemRegion_MEMORY_REGION0;
    cppiDescCfg.descNum         =   NUM_HOST_DESC;
    cppiDescCfg.destQueueNum    =   QMSS_PARAM_NOT_SPECIFIED;     
    cppiDescCfg.queueType       =   Qmss_QueueType_GENERAL_PURPOSE_QUEUE;
    cppiDescCfg.initDesc        =   Cppi_InitDesc_INIT_DESCRIPTOR;
    cppiDescCfg.descType        =   Cppi_DescType_HOST;
    
    /* By default:
     *      (1) Return descriptors to tail of queue 
     *      (2) Always return entire packet to this free queue
     *      (3) Set that PS Data is always present in start of SOP buffer
     *      (4) Configure free q num < 4K, hence qMgr = 0
     *      (5) Recycle back to the same Free queue by default.
     */
    cppiDescCfg.returnPushPolicy            =   Qmss_Location_TAIL;    
    cppiDescCfg.cfg.host.returnPolicy       =   Cppi_ReturnPolicy_RETURN_ENTIRE_PACKET;    
    cppiDescCfg.cfg.host.psLocation         =   Cppi_PSLoc_PS_IN_DESC;         
    cppiDescCfg.returnQueue.qMgr            =   0;    
    cppiDescCfg.returnQueue.qNum            =   QMSS_PARAM_NOT_SPECIFIED; 
    cppiDescCfg.epibPresent                 =   Cppi_EPIB_EPIB_PRESENT;
    
    /* Initialize the descriptors, create a free queue and push descriptors to a global free queue */
    if ((gGlobalFreeQHnd = Cppi_initDescriptor (&cppiDescCfg, &numAllocated)) <= 0)
    {
        System_printf ("Error Initializing Free Descriptors, Error: %d \n", gGlobalFreeQHnd);
        return -1;
    }
    else
    {
        System_printf ("Initializing Free Descriptors. \n");
    }        
   
    /* Queue Manager Initialization Done */
    return 0;
}


/** ============================================================================
 *   @n@b Init_Qmss_Local
 *
 *   @b Description
 *   @n This API initializes the QMSS LLD in cores other than core 0.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t Init_Qmss_Local (void)
{
  int32_t            result;

  while(1)
  {
      /* Block until Qmss_init() has completed by core 0 */
      result = Qmss_start();
      if(result == QMSS_NOT_INITIALIZED)
      {
          System_printf ("QMSS Not yet Initialized\n");
          continue;
      }
      else if (result != QMSS_SOK)  {
        System_printf ("Qmss_start failed with error code %d\n", result);
        return (-1);
      }

      if (result == QMSS_SOK) 
      {
          break;
      }
  }
  return 0;
}


/** ============================================================================
 *   @n@b Init_Cppi
 *
 *   @b Description
 *   @n This API initializes the CPPI LLD, opens the PASS CPDMA and opens up
 *      the Tx, Rx channels required for data transfers.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t Init_Cppi (void)
{
    int32_t                     result, i;        
    Cppi_CpDmaInitCfg           cpdmaCfg;
    uint8_t                     isAllocated;        
    Cppi_TxChInitCfg            txChCfg;
    Cppi_RxChInitCfg            rxChInitCfg;
    Cppi_GlobalConfigParams     fw_cppiGblCfgParams[CPPI_MAX_CPDMA];

  for (i=0; i<CPPI_MAX_CPDMA; i++)
    fw_cppiGblCfgParams[i] = cppiGblCfgParams[i];


  /* Convert Physical address to Virtual address for LLD access */
  /* SRIO CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* PASS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxFlowRegs =
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* QMSS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);


    /* Initialize CPPI LLD */
    result = Cppi_init (fw_cppiGblCfgParams);
    if (result != CPPI_SOK)
    {
        System_printf ("Error initializing CPPI LLD, Error code : %d\n", result);
        return -1;
    }

    /* Initialize PASS CPDMA */
    memset (&cpdmaCfg, 0, sizeof (Cppi_CpDmaInitCfg));
    cpdmaCfg.dmaNum     = Cppi_CpDma_PASS_CPDMA;
    if ((gCpdmaHnd = Cppi_open (&cpdmaCfg)) == NULL)
    {
        System_printf ("Error initializing CPPI for PASS CPDMA %d \n", cpdmaCfg.dmaNum);
        return -1;
    }    

    /* Open all CPPI Tx Channels. These will be used to send data to PASS/CPSW */             
    for (i = 0; i < NUM_PA_TX_QUEUES; i ++)
    {
        txChCfg.channelNum      =   i;       /* CPPI channels are mapped one-one to the PA Tx queues */
        txChCfg.txEnable        =   Cppi_ChState_CHANNEL_DISABLE;  /* Disable the channel for now. */
        txChCfg.filterEPIB      =   0;
        txChCfg.filterPS        =   0;
        txChCfg.aifMonoMode     =   0;
        txChCfg.priority        =   2;
        if ((gCpdmaTxChanHnd[i] = Cppi_txChannelOpen (gCpdmaHnd, &txChCfg, &isAllocated)) == NULL)
        {
            System_printf ("Error opening Tx channel %d\n", txChCfg.channelNum);
            return -1;
        }
        Cppi_channelEnable (gCpdmaTxChanHnd[i]);
    }

    /* Open all CPPI Rx channels. These will be used by PA to stream data out. */
    for (i = 0; i < NUM_PA_RX_CHANNELS; i++)
    {
        /* Open a CPPI Rx channel that will be used by PA to stream data out. */
        rxChInitCfg.channelNum  =   i; 
        rxChInitCfg.rxEnable    =   Cppi_ChState_CHANNEL_DISABLE; 
        if ((gCpdmaRxChanHnd[i] = Cppi_rxChannelOpen (gCpdmaHnd, &rxChInitCfg, &isAllocated)) == NULL)
        {
            System_printf ("Error opening Rx channel: %d \n", rxChInitCfg.channelNum);
            return -1;
        }

        /* Also enable Rx Channel */
        Cppi_channelEnable (gCpdmaRxChanHnd[i]);    
    }
    
    /* Clear CPPI Loobpack bit in PASS CDMA Global Emulation Control Register */
    Cppi_setCpdmaLoopback(gCpdmaHnd, 0);   

    /* CPPI Init Done. Return success */
    return 0;
}    

/** ============================================================================
 *   @n@b Setup_Tx
 *
 *   @b Description
 *   @n This API sets up all relevant data structures and configuration required
 *      for sending data to PASS/Ethernet. It sets up a Tx free descriptor queue,
 *      PASS Tx queues required for send.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t Setup_Tx (void)
{
    uint8_t                     isAllocated, i;        
    Qmss_Queue                  qInfo;
    void*                       pCppiDesc;

    /* Open all Transmit (Tx) queues. 
     *
     * These queues are used to send data to PA PDSP/CPSW.
     */
    for (i = 0; i < NUM_PA_TX_QUEUES; i ++)
    {            
        if ((gPaTxQHnd[i] = Qmss_queueOpen (Qmss_QueueType_PASS_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
        {
            System_printf ("Error opening PA Tx queue \n");
            return -1;
        }            
    }
    
    /* Open a Tx Free Descriptor Queue (Tx FDQ). 
     *
     * This queue will be used to hold Tx free decriptors that can be filled
     * later with data buffers for transmission onto wire.
     */
    if ((gTxFreeQHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    {
        System_printf ("Error opening Tx Free descriptor queue \n");
        return -1;
    }       
             

    qInfo = Qmss_getQueueNumber (gTxFreeQHnd);

    /* Attach some free descriptors to the Tx free queue we just opened. */
    for (i = 0; i < NUM_TX_DESC; i++)
    {
        /* Get a free descriptor from the global free queue we setup 
         * during initialization.
         */
        if ((pCppiDesc = Qmss_queuePop (gGlobalFreeQHnd)) == NULL)
        {
            break;                
        }

        /* The descriptor address returned from the hardware has the 
         * descriptor size appended to the address in the last 4 bits.
         *
         * To get the true descriptor size, always mask off the last 
         * 4 bits of the address.
         */
        pCppiDesc = (void*) ((uint32_t) pCppiDesc & 0xFFFFFFF0);

        pCppiMemTX = (void *)fw_memAlloc(TX_BUF_SIZE,CACHE_LINESZ);
        if(pCppiMemTX == NULL)
            break;

        /* Populate the Tx free descriptor with the buffer. */
        Cppi_setData (Cppi_DescType_HOST, pCppiDesc, pCppiMemTX, TX_BUF_SIZE);

        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, pCppiDesc, pCppiMemTX, TX_BUF_SIZE);

        /* Setup the Completion queue:
         *
         * Setup the return policy for this desc to return to the free q we just
         * setup instead of the global free queue.
         */
        Cppi_setReturnQueue ((Cppi_DescType) Cppi_DescType_HOST, pCppiDesc, qInfo);
        Cppi_setPacketLen    (Cppi_DescType_HOST, pCppiDesc, TX_BUF_SIZE);
        
        SYS_CACHE_WB (pCppiDesc, SIZE_HOST_DESC, 0);

        /* Push descriptor to Tx free queue */
        Qmss_queuePushDescSize (gTxFreeQHnd, pCppiDesc, SIZE_HOST_DESC);           
    }
    if (i != NUM_TX_DESC)
    {
        System_printf ("Error allocating Tx free descriptors \n");            
        return -1;
    }

    /* All done with Rx configuration. Return success. */
    return 0;
}

/** ============================================================================
 *   @n@b Setup_Rx
 *
 *   @b Description
 *   @n This API sets up all relevant data structures and configuration required
 *      for receiving data from PASS/Ethernet. It sets up a Rx free descriptor queue
 *      with some empty pre-allocated buffers to receive data, and an Rx queue
 *      to which the Rxed data is streamed for the example application. 
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t Setup_Rx (void)
{
    uint8_t                     isAllocated, i;        
    Qmss_Queue                  rxFreeQInfo, rxQInfo;
    void*                       pCppiDesc;
    Cppi_RxFlowCfg              rxFlowCfg;
    void*                       pDataBuffer;
    uint32_t                    mySWInfo[] = {0x11112222, 0x33334444};

       
    /* Open a Receive (Rx) queue. 
     *
     * This queue will be used to hold all the packets received by PASS/CPSW
     *
     */
    if ((gRxQHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    {
        System_printf ("Error opening gRxQHnd queue \n");
        return -1;
    }            
    rxQInfo = Qmss_getQueueNumber (gRxQHnd);

      
    /* Open a Rx Free Descriptor Queue (Rx FDQ). 
     *
     * This queue will hold all the Rx free decriptors. These descriptors will be
     * used by the PASS CPDMA to hold data received via CPSW.
     */
    if ((gRxFreeQHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
    {
        System_printf ("Error opening Rx Free descriptor queue \n");
        return -1;
    } 
    
    rxFreeQInfo = Qmss_getQueueNumber (gRxFreeQHnd);

    /* Attach some free descriptors to the Rx free queue we just opened. */
    for (i = 0; i < NUM_RX_DESC; i++)
    {
        /* Get a free descriptor from the global free queue we setup 
         * during initialization.
         */
        if ((pCppiDesc = Qmss_queuePop (gGlobalFreeQHnd)) == NULL)
        {
            System_printf ("Error poping descriptor.\n");
            break;                
        }

        /* The descriptor address returned from the hardware has the 
         * descriptor size appended to the address in the last 4 bits.
         *
         * To get the true descriptor size, always mask off the last 
         * 4 bits of the address.
         */
        pCppiDesc = (void*) ((uint32_t) pCppiDesc & 0xFFFFFFF0);
        pDataBuffer = (uint8_t *)fw_memAlloc(RX_BUF_SIZE,CACHE_LINESZ);

        /* Populate the Rx free descriptor with the buffer we just allocated. */
        Cppi_setData (Cppi_DescType_HOST, pCppiDesc, (uint8_t *)pDataBuffer, RX_BUF_SIZE);

        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, pCppiDesc, (uint8_t *)pDataBuffer, RX_BUF_SIZE);            

        /* Setup the Completion queue:
         * Setup the return policy for this desc to return to the free q we just
         * setup instead of the global free queue.
         */
        Cppi_setReturnQueue (Cppi_DescType_HOST, pCppiDesc, rxFreeQInfo);
        Cppi_setSoftwareInfo (Cppi_DescType_HOST, pCppiDesc, (uint8_t *) mySWInfo);
        Cppi_setPacketLen    (Cppi_DescType_HOST, pCppiDesc, RX_BUF_SIZE);          
        
        SYS_CACHE_WB (pCppiDesc, SIZE_HOST_DESC, 0);

        /* Push descriptor to Tx free queue */
        Qmss_queuePushDescSize (gRxFreeQHnd, pCppiDesc, SIZE_HOST_DESC);           
    }        
    if (i != NUM_RX_DESC)
    {
        System_printf ("Error allocating Rx free descriptors \n");
        return -1;
    }
 
    
    /* Setup a Rx Flow.
     * A Rx flow encapsulates all relevant data properties that CPDMA would
     * have to know in order to succefully receive data.
     */
    /* Initialize the flow configuration */
    memset (&rxFlowCfg, 0, sizeof(Cppi_RxFlowCfg));
    rxFreeQInfo = Qmss_getQueueNumber (gRxFreeQHnd);

    /* Let CPPI pick the next available flow */
#if FLOW_WORKAROUND
    rxFlowCfg.flowIdNum             =   1; 
#else
    rxFlowCfg.flowIdNum             =   CPPI_PARAM_NOT_SPECIFIED; 
#endif

    rxFlowCfg.rx_dest_qmgr          =   rxQInfo.qMgr;    
    rxFlowCfg.rx_dest_qnum          =   rxQInfo.qNum;  
    rxFlowCfg.rx_desc_type          =   Cppi_DescType_HOST; 

    rxFlowCfg.rx_ps_location        =   Cppi_PSLoc_PS_IN_DESC;  
    rxFlowCfg.rx_psinfo_present     =   1;    /* Enable PS info */
    
    rxFlowCfg.rx_error_handling     =   0;    /* Drop the packet, do not retry on starvation by default */       
    rxFlowCfg.rx_einfo_present      =   1;    /* EPIB info present */       
    
    rxFlowCfg.rx_dest_tag_lo_sel    =   0;    /* Disable tagging */
    rxFlowCfg.rx_dest_tag_hi_sel    =   0;    
    rxFlowCfg.rx_src_tag_lo_sel     =   0;    
    rxFlowCfg.rx_src_tag_hi_sel     =   0;    

    rxFlowCfg.rx_size_thresh0_en    =   0;    /* By default, we disable Rx Thresholds */
    rxFlowCfg.rx_size_thresh1_en    =   0;    /* By default, we disable Rx Thresholds */
    rxFlowCfg.rx_size_thresh2_en    =   0;    /* By default, we disable Rx Thresholds */
    rxFlowCfg.rx_size_thresh0       =   0x0;
    rxFlowCfg.rx_size_thresh1       =   0x0;
    rxFlowCfg.rx_size_thresh2       =   0x0;

    rxFlowCfg.rx_fdq0_sz0_qmgr      =   rxFreeQInfo.qMgr; /* Setup the Receive free queue for the flow */
    rxFlowCfg.rx_fdq0_sz0_qnum      =   rxFreeQInfo.qNum;    
    rxFlowCfg.rx_fdq0_sz1_qnum      =   0x0; 
    rxFlowCfg.rx_fdq0_sz1_qmgr      =   0x0;
    rxFlowCfg.rx_fdq0_sz2_qnum      =   0x0;
    rxFlowCfg.rx_fdq0_sz2_qmgr      =   0x0;
    rxFlowCfg.rx_fdq0_sz3_qnum      =   0x0;
    rxFlowCfg.rx_fdq0_sz3_qmgr      =   0x0;

    rxFlowCfg.rx_fdq1_qnum          =   rxFreeQInfo.qNum;  /* Use the Rx Queue to pick descriptors */
    rxFlowCfg.rx_fdq1_qmgr          =   rxFreeQInfo.qMgr;
    rxFlowCfg.rx_fdq2_qnum          =   rxFreeQInfo.qNum;  /* Use the Rx Queue to pick descriptors */
    rxFlowCfg.rx_fdq2_qmgr          =   rxFreeQInfo.qMgr;
    rxFlowCfg.rx_fdq3_qnum          =   rxFreeQInfo.qNum;  /* Use the Rx Queue to pick descriptors */
    rxFlowCfg.rx_fdq3_qmgr          =   rxFreeQInfo.qMgr;

    /* Configure the Rx flow */
    if ((gRxFlowHnd = Cppi_configureRxFlow (gCpdmaHnd, &rxFlowCfg, &isAllocated)) == NULL)
    {
        System_printf ("Error configuring Rx flow \n");
        return -1;
    }

    /* All done with Rx configuration. Return success. */
    return 0;
}

/** ============================================================================
 *   @n@b SendPacket
 *
 *   @b Description
 *   @n This API is called to actually send out data onto wire using ethernet.
 *      On success, this API increments a global Tx counter to indicate the same.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t SendPacket (void)
{
    Cppi_HostDesc*      pCppiDesc;
    uint32_t            dataBufferSize;
    uint32_t            key;
    uint8_t*            pDataBuf;
    uint32_t            dataLen;
    

    /* Get a free descriptor from the global free queue we setup 
     * during initialization.
     */
    if ((pCppiDesc = Qmss_queuePop (gTxFreeQHnd)) == NULL)
    {
        System_printf ("No Tx free descriptor. Cant run send/rcv test \n");
        return -1;
    }
    
    /* The descriptor address returned from the hardware has the 
     * descriptor size appended to the address in the last 4 bits.
     *
     * To get the true descriptor size, always mask off the last 
     * 4 bits of the address.
     */
    pCppiDesc = (void*) ((uint32_t) pCppiDesc & 0xFFFFFFF0);    
    
    dataBufferSize  =   sizeof (pktMatch);
    
    SYS_CACHE_INV (pCppiDesc, SIZE_HOST_DESC, 0);

    Cppi_getData (Cppi_DescType_HOST, (Cppi_Desc*)pCppiDesc, &pDataBuf, &dataLen);
    if(dataLen < dataBufferSize)
    {
        System_printf ("Unexpected Error. Buffer Length in descriptor shorter than expected \n");
        while(1);
    }
    memcpy(pDataBuf,pktMatch,dataBufferSize);
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)pCppiDesc, dataBufferSize);
    
    /* Clear PS Data */
    Cppi_setPSLen (Cppi_DescType_HOST, (Cppi_Desc *)pCppiDesc, 0);
  
    SYS_CACHE_WB (pCppiDesc, SIZE_HOST_DESC, 0);
  
    if (passLpbkMode != PASS_LOOPBACK_INTERNAL)
    {
        /* Send the packet out the mac. It will loop back to PA if the mac/switch 
         * have been configured properly 
         */  
        Qmss_queuePushDescSize (gPaTxQHnd[8], pCppiDesc, SIZE_HOST_DESC);
    }
    else
    {
        /* Send the packet to be looped back at PA. 
         */  
        Qmss_queuePushDescSize (gPaTxQHnd[0], pCppiDesc, SIZE_HOST_DESC);
    }
    /* Increment the application transmit counter */
    gTxCounter ++;

    /* Give some time for the PA to process the packet */
    CycleDelay (10000);

    return 0; 
}

/** ============================================================================
 *   @n@b ReceivePacket
 *
 *   @b Description
 *   @n This API is called to Receive packets.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t ReceivePacket (void)
{
	Cppi_Desc           *hd;
	int32_t             j;
    int32_t             status=0;
    
	/* Wait for a data packet from PA */
    for (j = 0; j < 100; j++)  
    {
      CycleDelay (1000);
      if (Qmss_getQueueEntryCount (gRxQHnd) > 0)   
      {
        hd = (Cppi_Desc *)(((uint32_t)Qmss_queuePop (gRxQHnd)) & ~0xf);
        if(VerifyPacket(hd) != 0)
            status=-1;
      }
    } 
    
    return (status);
}

/** ============================================================================
 *   @n@b VerifyPacket
 *
 *   @b Description
 *   @n This API verifies a packet received against the expected data and 
 *      returns 0 to inidcate success and -1 to indicate a mismatch.
 *
 *   @param[in]  
 *   @n pCppiDesc           Packet descriptor received.
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t VerifyPacket (Cppi_Desc* pCppiDesc)
{
	Cppi_HostDesc               *pHostDesc;
	uint8_t                     *pDataBuffer;
	int32_t                       i;
	
    pHostDesc = (Cppi_HostDesc *)pCppiDesc;

    SYS_CACHE_INV (pHostDesc, SIZE_HOST_DESC, 0);    
    SYS_CACHE_INV ((void*)(pHostDesc->buffPtr), pHostDesc->buffLen, 0);
    
    /* Verify the application software info we received is same
     * as what we had sent earlier.
     */
    if (pHostDesc->softwareInfo0 != 0xaaaaaaaa)  
    {
        System_printf ("VerifyPacket: Found an entry in receive queue with swinfo0 = 0x%08x, expected 0x%08x\n", 
                        pHostDesc->softwareInfo0, 0xaaaaaaaa);
                   
        pHostDesc->buffLen = pHostDesc->origBufferLen;
        SYS_CACHE_WB (pHostDesc, SIZE_HOST_DESC, 0);
        Qmss_queuePush (gRxFreeQHnd, (void*)pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);
          
        return -1;
    }
        
    /* Verify the packet matches what we had sent */
    pDataBuffer = (uint8_t *) pHostDesc->buffPtr;
    for (i = 0; i < sizeof (pktMatch); i++)  
    {
        if (pktMatch[i] != pDataBuffer[i])  
        {
            System_printf ("VerifyPacket: Byte %d expected 0x%02x, found 0x%02x\n", i, pktMatch[i], pDataBuffer[i]);           

            /* Free the packet back to the Rx FDQ */
            pHostDesc->buffLen = pHostDesc->origBufferLen;
            SYS_CACHE_WB (pHostDesc, SIZE_HOST_DESC, 0);
            Qmss_queuePush (gRxFreeQHnd, (void*)pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);
            return -1;
        }
    }
        
    /* Increment Rx counter to indicate the number of successfully
     * received packets by the example app.
     */
    gRxCounter ++;

    /* Reset the buffer lenght and put the descriptor back on the free queue */      
    pHostDesc->buffLen = pHostDesc->origBufferLen;
    SYS_CACHE_WB (pHostDesc, SIZE_HOST_DESC, 0);
    Qmss_queuePush (gRxFreeQHnd, (void*)pHostDesc, pHostDesc->buffLen, SIZE_HOST_DESC, Qmss_Location_TAIL);

    /* Verify packet done. Return success. */
	return 0;
}
	

