/**
 *   @file  sample.c
 *
 *   @brief   
 *      This is the CPPI Low Level Driver example file.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  \par
*/

#include "fw_test.h"

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>

/* CPPI LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>

#include "cppi_test.h"

/************************ GLOBAL VARIABLES ********************/

/* Tx channel configuration */
Cppi_TxChInitCfg        txChCfg;
/* Rx channel configuration */
Cppi_RxChInitCfg        rxChCfg;
/* Memory region configuration information */
Qmss_MemRegInfo         memInfo;

/*************************** FUNCTIONS ************************/

/**
 *  @b Description
 *  @n  
 *      This is an example code that shows CPPI LLD API usage.
 *
 *      It performs the following
 *          - Initializes the Queue Manager low level driver.
 *          - Initializes the CPPI low level driver.
 *          - Opens the CPPI CPDMA in queue manager
 *          - Initializes descriptors and pushes to free queue
 *          - Opens Rx and Tx channel
 *          - Pushes packet on Tx channel. Diverts the packet to Tx channel.
 *          - Process the Rx packet
 *          - Closes Rx and Tx channel
 *          - Closes all open queues
 *          - Closes CPDMA instance
 *          - Deinitializes CPPI LLD
 *
 *  @retval
 *      Not Applicable.
 */
void sample_usage (Cppi_Handle cppiHnd)
{
    int32_t                   result;
    uint32_t                  numAllocated, i, destLen, runCount = 0;
    uint8_t                   *dataBuffPtr, *destDataPtr;
    uint8_t                   isAllocated;
    Cppi_ChHnd              rxChHnd, txChHnd;
    Qmss_QueueHnd           txQueHnd, rxQueHnd, freeQueHnd;
    Cppi_DescCfg            descCfg;
    Cppi_Desc               *hostDescPtr, *rxPkt;
    uint8_t                   srcData[4];
    uint32_t                  corenum ;


    memset ((void *) hostDesc, 0, SIZE_HOST_DESC * NUM_HOST_DESC);
    memset ((void *) monolithicDesc, 0, SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC);

    /* Setup memory region */
    memInfo.descBase = (uint32_t *)hostDesc;
    memInfo.descSize = SIZE_HOST_DESC;
    memInfo.descNum = NUM_HOST_DESC;
    memInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    memInfo.memRegion = (Qmss_MemRegion) 0;
    memInfo.startIndex = 0;
   
    result = Qmss_insertMemoryRegion (&memInfo);
    if (result == QMSS_MEMREGION_ALREADY_INITIALIZED)
    {
        printf ("Memory region is configured\n");
    }
    else if (result < QMSS_SOK)
    {
        printf ("Error %d : Inserting memory region %d error code : %d\n", memInfo.memRegion, result);
        return;
    }
    else
        printf ("Memory region %d inserted\n", result);


    /* Setup the descriptors */
    descCfg.memRegion = (Qmss_MemRegion) 0;
    descCfg.descNum = NUM_HOST_DESC/4;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_GENERAL_PURPOSE_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_HOST;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;
    /* Descriptor should be recycled back to freeQue allocated since destQueueNum is < 0 */
    descCfg.returnQueue.qMgr = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.returnQueue.qNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.returnPushPolicy = Qmss_Location_HEAD;
    descCfg.cfg.host.returnPolicy = Cppi_ReturnPolicy_RETURN_ENTIRE_PACKET;
    descCfg.cfg.host.psLocation = Cppi_PSLoc_PS_IN_DESC;

    /* Initialize the descriptors and push to free Queue */
    if ((freeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error: Initializing descriptor error code: %d \n", freeQueHnd);
		return;
	}
    else
        printf ("Number of descriptors requested : %d. Number of descriptors allocated : %d \n", 
                        descCfg.descNum, numAllocated);

    /* Set up Rx Channel parameters */
    rxChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    rxChCfg.rxEnable = Cppi_ChState_CHANNEL_ENABLE;
    
    /* Open Rx Channel */
    rxChHnd = (Cppi_ChHnd) Cppi_rxChannelOpen (cppiHnd, &rxChCfg, &isAllocated);
    if (rxChHnd == NULL)
    {
        printf ("Error: Opening Rx channel : %d\n", rxChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Rx channel : %d\n", Cppi_getChannelNumber (rxChHnd));


    /* Set up Tx Channel parameters */
    txChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    txChCfg.priority = 2;
    txChCfg.filterEPIB = 0;
    txChCfg.filterPS = 0;
    txChCfg.aifMonoMode = 0;
    txChCfg.txEnable = Cppi_ChState_CHANNEL_ENABLE;
    
    /* Open Tx Channel */
    txChHnd = (Cppi_ChHnd) Cppi_txChannelOpen (cppiHnd, &txChCfg, &isAllocated);
    if (txChHnd == NULL)
    {
        printf ("Error: Opening Tx channel : %d\n", txChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Tx channel : %d\n", Cppi_getChannelNumber (txChHnd));

    /* Opens transmit queue */
    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_LOW_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Queue Number\n");
		return;
	}
    else
        printf ("Queue Number : %d opened\n", txQueHnd);

    /* Opens receive queue */
    if ((rxQueHnd = Qmss_queueOpen (Qmss_QueueType_LOW_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Queue Number\n");
	}
    else
        printf ("Queue Number : %d opened\n", rxQueHnd);

    /* Set transmit queue threshold to high and when there is atleast one packet */
    /* Setting threshold on transmit queue is not required anymore. tx pending queue is not hooked to threshold. 
     * Qmss_setQueueThreshold (txQueHnd, 1, 1);
     */ 

    /* Fill in some data */
    for (i = 0; i < SIZE_DATA_BUFFER; i++) 
        dataBuff[i] = i;

    /* Get a free descriptor */
    while ((hostDescPtr = (Cppi_Desc *) Qmss_queuePop (freeQueHnd)) != NULL)
    {
        /* Add data buffer */
        Cppi_setData (Cppi_DescType_HOST, hostDescPtr, dataBuff, SIZE_DATA_BUFFER);

        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, hostDescPtr, dataBuff, SIZE_DATA_BUFFER);

        /* Set packet length */
        Cppi_setPacketLen (Cppi_DescType_HOST, hostDescPtr, SIZE_DATA_BUFFER);

        /* Fill in some data */
        srcData[0] = 0xAB;
        srcData[1] = 0xCD;
        srcData[2] = 0xEF;
        srcData[3] = 0xDC;

        /* Add PS data */
        Cppi_setPSData (Cppi_DescType_HOST, hostDescPtr, (uint8_t *) srcData, 4);

        /* Push descriptor to Tx queue */
    	Qmss_queuePushDescSize (txQueHnd, (Ptr) hostDescPtr, SIZE_HOST_DESC);

        result = Qmss_getQueueEntryCount (txQueHnd);
        printf ("Transmit Queue %d Entry Count : %d Tx descriptor 0x%p\n", txQueHnd, result, hostDescPtr);

        /* Here the packets are diverted to the destination queue. 
         * Can also configure flow to transfer packets to destination queue. 
         * */

        Qmss_queueDivert (txQueHnd, rxQueHnd, Qmss_Location_TAIL);

        result = Qmss_getQueueEntryCount (rxQueHnd);

        /* Wait for receive packet */
        while (Qmss_getQueueEntryCount (rxQueHnd) == 0);
        
        /* Get the rx packet */
        if ((rxPkt = (Cppi_Desc *) Qmss_queuePop (rxQueHnd)) == NULL)
        {
            return;
        }
        
    	/* The lower 4 bits of the descriptor address contain the size of the descriptor 
        that was specified during the queue push operation. Clear it */
    
        rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (rxPkt);

        printf ("Receive Queue %d Entry Count : %d Rx descriptor 0x%p\n", rxQueHnd, result, rxPkt);

        /* Get PS data */
        Cppi_getPSData (Cppi_DescType_HOST, Cppi_PSLoc_PS_IN_DESC, rxPkt, &destDataPtr, &destLen);

        /* Compare */
        for (i=0; i < destLen; i++)
        {
            if (srcData[i] != destDataPtr[i])
            {
                printf ("Error: In PS data Tx: %02x - Rx: %02x \n", srcData[i], destDataPtr[i]);
                break;
            }
        }

        /* Get data buffer */

        Cppi_getData (Cppi_DescType_HOST, rxPkt, &dataBuffPtr, &destLen);

        /* Compare */
        for (i=0; i < destLen; i++)
        {
            if (dataBuff[i] != dataBuffPtr[i])
            {
                printf ("Error: In data buffer Tx: %02x - Rx: %02x \n", dataBuff[i], dataBuffPtr[i]);
                break;
            }
        }
        runCount++;
        
    }
    if ((runCount == 0) && (runCount != NUM_HOST_DESC))
    {
        printf ("Getting host descriptor from Queue : %d \n", freeQueHnd);
    }
    else
        printf ("Received host descriptor from Queue %d Sucessfully\n", freeQueHnd);

    /* Close Tx channel */
    if ((result = Cppi_channelClose (txChHnd)) < 0)
        printf ("Error: Closing Tx channel error code: %d\n", result);
    else
        printf ("Tx Channel closed successfully. Ref count :%d\n", result);

    /* Close Rx channel */
    if ((result = Cppi_channelClose (rxChHnd)) < 0)
        printf ("Error: Closing Rx channel error code: %d\n", result);
    else
        printf ("Rx Channel closed successfully. Ref count :%d\n", result);
    
    /* Close the queues */
    if ((result = Qmss_queueClose (rxQueHnd)) < 0)
        printf ("Error: Closing Rx queue error code: %d\n", result);
    else
        printf ("Rx queue closed successfully. Ref count :%d\n", result);
    
    if ((result = Qmss_queueClose (txQueHnd)) < 0)
        printf ("Error: Closing tx queue error code: %d\n", result);
    else
        printf ("Tx queue closed successfully. Ref count :%d\n", result);

    if ((result = Qmss_queueClose (freeQueHnd)) < 0)
        printf ("Error: Closing free queue error code: %d\n", result);
    else
        printf ("Free queue closed successfully. Ref count :%d\n", result);

    /* Close CPPI CPDMA instance */
    if ((result = Cppi_close (cppiHnd)) != CPPI_SOK)
    {
        printf ("Closing CPPI CPDMA Ref count : %d\n", result);
        while (result < CPPI_SOK)
            result = Cppi_close (cppiHnd);
        printf ("CPPI CPDMA closed successfully\n");
    }
    else
        printf ("CPPI CPDMA closed successfully\n");

    /* Deinitialize CPPI LLD */
    if ((result = Cppi_exit ()) != CPPI_SOK)
    {
        printf ("Exiting CPPI Ref count : %d\n", result);
        while (result < CPPI_SOK)
            result = Cppi_exit ();
        printf ("CPPI exit successful\n");
    }
    else
        printf ("CPPI exit successful\n");

}

