/*
 *
 * Copyright (C) 2010-2012 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/




#ifndef _PAUTEST_H
#define _PAUTEST_H


#include <xdc/std.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Memory.h>
#include <xdc/runtime/Error.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/hal/Hwi.h>
#include <ti/sysbios/knl/task.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/gates/GateHwi.h> 
#include "pa_log.h"

#include <ti/drv/pa/pa.h>
#include <ti/drv/pa/pasahost.h>   
#include <ti/drv/qmss/qmss_drv.h>
#include <ti/drv/qmss/qmss_osal.h>
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>
#include <ti/drv/cppi/cppi_osal.h>
#include <ti/csl/csl_cacheAux.h>
#include <ti/csl/csl_xmcAux.h>

#include <string.h>

/* Due to a bug in the simulator all packet lengths must be rounded up to 4 byte multiples */
#define PA_SIM_BUG_4BYTES

typedef enum  {
	PA_TEST_FAILED  = -1,
	PA_TEST_NOT_RUN,
	PA_TEST_PASSED
} paTestStatus_t;


/* Define the test interface */
typedef struct paTest_s
{
	void (*testFunction)(UArg, UArg);   /* The code that runs the test */
	char *name;							/* The test name */
	paTestStatus_t testStatus;			/* Test status */
	
} paTest_t;

/* Number of cpdma channels is fixed by device */
#define TF_PA_NUM_RX_CPDMA_CHANNELS 24
#define TF_PA_NUM_TX_CPDMA_CHANNELS  9
#define TF_Q_FREE_DESC						900   /* Unassociated descriptor queue number */
#define TF_TEARDOWN_QUE_MGR  				901   /* Teardown queue number */
#define TF_TEARDOWN_QUE_NUM  				  0

#define TF_LINKED_BUF_Q1           			902     	/* First queue with attached buffers */
#define TF_LINKED_BUF_Q1_BUF_SIZE  			128
#define TF_LINKED_BUF_Q1_NBUFS       		 28
extern UChar memQ1[TF_LINKED_BUF_Q1_NBUFS][TF_LINKED_BUF_Q1_BUF_SIZE];

#define TF_LINKED_BUF_Q2		   			903		/* Second queue with attached buffers */
#define TF_LINKED_BUF_Q2_BUF_SIZE  			512
#define TF_LINKED_BUF_Q2_NBUFS				 32
extern UChar memQ2[TF_LINKED_BUF_Q2_NBUFS][TF_LINKED_BUF_Q2_BUF_SIZE];

#define TF_LINKED_BUF_Q3		   			904
#define TF_LINKED_BUF_Q3_BUF_SIZE  		   1600	
#define TF_LINKED_BUF_Q3_NBUFS      		  8
extern UChar memQ3[TF_LINKED_BUF_Q3_NBUFS][TF_LINKED_BUF_Q3_BUF_SIZE];

#define TF_DEF_RET_Q                        905     /* Default return of queues with linked descriptors */

#define TF_COMMON_CMD_REPL_Q				906		/* Common code PA command reply queue */


#define TF_FIRST_GEN_QUEUE		   			907		/* Queues available for general use */
#define TF_NUM_GEN_QUEUES		   			100

/* The following software info 0 command IDs are reserved by the common setup */
#define TF_COMMON_CMD_ID_ADD_MAC	0xff000000
#define TF_COMMON_CMD_ID_ADD_IP		0xff010000
#define TF_COMMON_CMD_ID_ADD_PORT   0xff020000
#define TF_COMMON_CMD_ID_CFG_CL4  	0xff030000
#define TF_COMMON_CMD_ID_ADD_CL4	0xff040000
#define TF_COMMON_CMD_ID_DEL_CL4	0xff050000
#define TF_COMMON_CMD_ID_DEL_PORT   0xff060000
#define TF_COMMON_CMD_ID_DEL_IP		0xff070000
#define TF_COMMON_CMD_ID_DEL_MAC    0xff080000
#define TF_COMMON_CMD_ID_CFG_CL3  	0xff090000
#define TF_COMMON_CMD_ID_ADD_CL3	0xff0A0000
#define TF_COMMON_CMD_ID_DEL_CL3	0xff0B0000
#define TF_COMMON_CMD_ID_ADD_CMDSET 0xff0C0000

/* Define the test framework */
typedef struct tFramework_s  {
	
	Pa_Handle   passHandle;	/* PA instance handle */
	
	GateHwi_Handle gateHwi;		/* HW interrupt disable handle */
	
	Cppi_Handle tfPaCppiHandle; /* PA CDMA handle */ 
	
	Cppi_ChHnd  tfPaTxChHnd[TF_PA_NUM_TX_CPDMA_CHANNELS];
	Cppi_ChHnd  tfPaRxChHnd[TF_PA_NUM_RX_CPDMA_CHANNELS];
	
	/* Queues */
	Int32 QPaTx[TF_PA_NUM_TX_CPDMA_CHANNELS];
	Int32 QfreeDesc;		      				/* Unassociated descriptor queue handle */
	Int32 QtDown;								/* Tear down queue handle */
    Int32 QDefRet;                              /* Default return queue for descriptors with linked buffers */
	Int32 QLinkedBuf1;							/* First Queue with descriptors and attached linked buffers */
	Int32 QLinkedBuf2;							/* Second Queue with descriptors and attached linked buffers */
	Int32 QLinkedBuf3;							/* Third Queue with descriptors and attached linked buffers */
	Int32 QCommonCmdRep;						/* Queue for command reply used by common code				*/
	Int32 QGen[TF_NUM_GEN_QUEUES];				/* General purpose queues */
	
	Cppi_FlowHnd tfPaFlowHnd0;					/* Flow handle for flow 0  */
	Int32		 tfFlowNum;						/* Physical flow number */
	
	Semaphore_Handle tfPaTableL2Sem;			/* Semaphore for PA internal table for L2 */
	Semaphore_Handle tfPaTableL3Sem;			/* Semaphore for PA internal table for L3 */
	
	Semaphore_Handle tfQmSem;					/* Semaphore for queue manager */
	
} tFramework_t;

extern tFramework_t tFramework;

#define TF_CACHE_LINESZ    128
#define TF_ROUND_UP(x, y)   ((x) + ((y)-1))/(y)*(y)

/* PA memory */
#define TF_PA_INST_SIZE	  128					/* Required size = 84 */
extern UInt8 memPaInst[TF_ROUND_UP(TF_PA_INST_SIZE, TF_CACHE_LINESZ)];

#define TF_MAX_NUM_L2_HANDLES	64
#define TF_L2_TABLE_SIZE	(TF_MAX_NUM_L2_HANDLES * 32)	/* Requires 32 bytes per entry */
extern UInt8 memL2Ram[TF_ROUND_UP(TF_L2_TABLE_SIZE, TF_CACHE_LINESZ)];

#define TF_MAX_NUM_L3_HANDLES	128
#define TF_L3_TABLE_SIZE	(TF_MAX_NUM_L3_HANDLES * 68)	/* Requires 68 bytes per entry */
extern UInt8 memL3Ram[TF_ROUND_UP(TF_L3_TABLE_SIZE, TF_CACHE_LINESZ)];

#define TF_USR_STATS_LNK_TABLE_SIZE   (pa_USR_STATS_MAX_COUNTERS * 4)  /* Require 4 bytes per satistics */ 
extern UInt8 memUsrStatsLnkTbl[TF_USR_STATS_LNK_TABLE_SIZE];
	
/* QM memory */	
#define TF_NUM_DESC		 128		/* 128 host descriptors managed by the QM */
#define TF_SIZE_DESC     128		/* Must be multiple of 16 */

/* Memory used for the linking RAM and descriptor RAM */
extern UInt64 memLinkRam[TF_NUM_DESC];
extern UInt8  memDescRam[TF_NUM_DESC * TF_SIZE_DESC];	


typedef UInt32  paStatsBmap_t; 

/* Stats numbers. Util functions are used to add these to the bit maps.
 * The values match the actual stats numbers */
#define TF_STATS_BM_C1_NUM_PACKETS		0
#define TF_STATS_BM_C1_NUM_IPV4			1
#define TF_STATS_BM_C1_NUM_IPV6			2
#define TF_STATS_BM_C1_NUM_CUSTOM		3
#define TF_STATS_BM_C1_NUM_SRIO		    4
#define TF_STATS_BM_C1_TABLE_MATCH   	6
#define TF_STATS_BM_C1_NO_TABLE_MATCH   7
#define TF_STATS_BM_C1_IP_FRAG          8
#define TF_STATS_BM_C1_VLAN_OVERFLOW    10
#define TF_STATS_BM_C1_NUM_MPLS			12
#define TF_STATS_BM_C1_PARSE_FAIL		13
#define TF_STATS_BM_C1_TX_IP_FRAG       15
#define TF_STATS_BM_C1_SILENT_DISCARD   17

#define TF_STATS_BM_C2_NUM_PACKETS		21
#define TF_STATS_BM_C2_NUM_UDP			23
#define TF_STATS_BM_C2_NUM_TCP			24
#define TF_STATS_BM_C2_NUM_CUSTOM		25

#define TF_STATS_BM_M_CMD_FAIL			30


/* Define additional PA Error Codes */
#define pa_NO_BUF                       1    /* Descriptor for PA command packet is not available */


/* Packet and associated structures */
typedef struct pktTestInfo_s  {
	UInt8 		     *pkt;
	pasahoLongInfo_t *info;
	Int			      pktLen;
	paStatsBmap_t     statsMap[3];  /* Bit map of which stats to increment. Some stats must be incremented 3 times */
	Int    		      idx;		    /* Used to increment the test tracking - tells the test to look for 
						             * a packet with this index */						             
						             
} pktTestInfo_t;					


/* Macros to form the packet info structure */
#define TF_FORM_PKT_INFO_WORD0(cmdId,recordLen,startOffset)  ((cmdId << 29) | (recordLen << 24) | startOffset)
#define TF_FORM_PKT_INFO_WORD1(endOffset,errIdx,pmatch,c2c,l1PdspId,l1Idx)  ((endOffset << 16) | (errIdx << 11) | (pmatch << 10) | \
																			 (c2c << 9) | (l1PdspId << 6) | (l1Idx))
#define TF_FORM_PKT_INFO_WORD2(l3Offset,l4Offset,l5Offset,ahEspOffset)		((l3Offset << 24) | (l4Offset << 16) | (l5Offset << 8) | ahEspOffset)
#define TF_FORM_PKT_INFO_WORD3(hdrBitmask,nextHdr,vCount,ipCount,greCount,frag,hdrBitmask2,ipmrt)  \
								((hdrBitmask << 21) | (nextHdr << 16) | (vCount << 14) | (ipCount << 8) | (greCount << 11) | \
								 (hdrBitmask2 << 4) | (frag << 3) | (ipmrt << 1))
#define TF_FORM_PKT_INFO_WORD4(pseudo)  (pseudo << 16)

#define TF_GET_UDP_OFFSET(x) ((((x)->word2) >> 16) & 0x00ff)
#define TF_GET_UDP_PSEUDO(x) (((x)->word4) >> 16)
#define TF_GET_IP_OFFSET(x) ((((x)->word2) >> 24) & 0x00ff)

/* Macros to form the PS Info words for the SRIO packet */
#define TF_FORM_SRIO_PSINFO_WORD0(srcId, destId)        (srcId << 16) | (destId)
#define TF_FORM_SRIO_PSINFO_WORD1_TYPE11(pri, cc, tt, letter, mbox)  ((cc) << 15) | ((pri) << 11) | ((tt) << 9) | ((letter) << 6) | (mbox)
#define TF_FORM_SRIO_PSINFO_WORD1_TYPE9(pri, cc, tt, streamId, cos)  ((cc) <<  8) | ((pri) << 11) | ((tt) << 10) | ( (streamId) << 16) | (cos)

#define PA_PKT_TYPE_SRIO_TYPE_9     30
#define PA_PKT_TYPE_SRIO_TYPE_11    31

/* A simple fifo */
typedef struct pauFifo_s  {
	Int		 inIdx;
	Int		 outIdx;
	Int		 size;
	UInt    *data;
} pauFifo_t;


extern paLog_t *paLogLevel;


/* Generic test setup
 * Test setup involves configuring PA to receive MAC, IP, TCP/UDP, custom.
 */
 
typedef enum   {
	PAU_TEST_SETUP_STATUS_CMD_NOT_SENT,		/* Command not yet sent to the PA */
	PAU_TEST_SETUP_STATUS_CMD_SENT,			/* Command sent, no reply received */
	PAU_TEST_SETUP_STATUS_CMD_REPLIED,		/* Received reply from PA, no errors */
	PAU_TEST_SETUP_STATUS_CMD_REP_ERR,		/* Received reply from PA with errors */
	PAU_TEST_SETUP_STATUS_CMD_NO_CMD		/* Command was intentionally invalid to test PA */
} pauTestSetupStatus_t;



/* MAC test setup */
typedef struct pauTestMacSetup_s  {
	
	paEthInfo_t  	ethInfo;		/* PA LLD Ethernet configuration */
	paRouteInfo_t   matchRoute;		/* Routing information for a match */
	paRouteInfo_t   nFail;			/* Routing information on subsequent failure */
	paReturn_t   	paret;			/* The expected PA return value  */
	paHandleL2L3_t  handle;			/* The returned PA handle */
	Bool			waitDone;		/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */
	
} pauTestMacSetup_t;

/* IP test setup. It is an error if both l2Link and L3Link are non-null */
typedef struct pauTestIpSetup_s  {
	
	paIpInfo_t  			 ipInfo;			/* PA LLD IP configuration */
	paRouteInfo_t   		 matchRoute;		/* Routing information for a match */
	paRouteInfo_t   		 nFail;				/* Routing information on subsequent failure */
	pauTestMacSetup_t  	    *l2Link;			/* Link to L2 handle, NULL if no L2 link */
	struct pauTestIpSetup_s *l3Link;			/* Link to L3 handle, NULL if no L3 Link. */
	paReturn_t      		 paret;				/* The expected PA return value */
	paHandleL2L3_t  		 handle;			/* The returned PA handle */
	Bool					 waitDone;			/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */
	
} pauTestIpSetup_t;

/* TCP/UDP test setup */
typedef struct pauTestL4Setup_s  {
	
	UInt16				destPort;		/* PA LLD destination port */
	paRouteInfo_t   	matchRoute;		/* Routing information for a match */
	pauTestIpSetup_t    *l3Link;		/* Link to L3 handle, NULL if no L3 Link */
	paReturn_t          paret;			/* The expected PA return value */
	paHandleL4_t        handle;			/* The returned PA handle */
	Bool			    waitDone;		/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */

} pauTestL4Setup_t;


/* Custom L3 (LUT1) test setup */
typedef struct pauTestCl3Config_s  {
	
	UInt16		custIndex;  					     	/* The custom Index */
    UInt16      offset;                                 /* offset to where the PA begins custom match */
    UInt16      nextHdr;                                /* The next header to be parsed */ 
    UInt16      nextHdrOffset;                          /* offset to where the PA begins after the custom match */
	UInt8		byteMasks[pa_NUM_BYTES_CUSTOM_LUT1];		/* The byte bit field masks */
	paReturn_t  paret;									/* The expected PA return value */
	Bool		waitDone;								/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */
	
} pauTestCl3Config_t;

/* Custom L3 table entries */
typedef struct pauTestCl3Setup_s  {
	
    UInt16            custIndex;                        /* Custom Index */
	UInt8			  match[pa_NUM_BYTES_CUSTOM_LUT1];	/* The table match values */
	paRouteInfo_t     matchRoute;						/* Routing information for a match */
	paRouteInfo_t     nFail;				            /* Routing information on subsequent failure */
	pauTestMacSetup_t  	    *l2Link;			        /* Link to L2 handle, NULL if no L2 link */
	struct pauTestIpSetup_s *l3Link;			        /* Link to L3 handle, NULL if no L3 Link. */
	paReturn_t        paret;							/* The expected PA return value */
	paHandleL2L3_t    handle;			                /* The returned PA handle */
	Bool			  waitDone;							/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */

} pauTestCl3Setup_t;

/* Custom L4 test setup */
typedef struct pauTestCl4Config_s  {
	
	Bool		handleLink;								/* Set to True if the previous link is included in the look up pattern */
	UInt16		custIndex;  					     	/* The custom Index */
	UInt16		byteOffsets[pa_NUM_BYTES_CUSTOM_LUT2];	/* The byte offsets */
	UInt8		byteMasks[pa_NUM_BYTES_CUSTOM_LUT2];		/* The byte bit field masks */
	paReturn_t  paret;									/* The expected PA return value */
	Bool		waitDone;								/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */
	
} pauTestCl4Config_t;

/* Custom L4 table entries */
typedef struct pauTestCl4Setup_s  {
	
    UInt16            custIndex;                        /* Custom Index */
	UInt8			  match[pa_NUM_BYTES_CUSTOM_LUT2];	/* The table match values */
	paRouteInfo_t     matchRoute;						/* Routing information for a match */
	pauTestIpSetup_t *l3Link;							/* Link to IP handle */
	paReturn_t        paret;							/* The expected PA return value */
	paHandleL4_t      handle;							/* The returned PA handle */
	Bool			  waitDone;							/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */

} pauTestCl4Setup_t;

/* Command set entries */
typedef struct pauTestCmdSetSetup_s  {
	
    UInt16            index;                            /* command set index */
    int               nCmds;                            /* number of commands */
    paCmdInfo_t      *cmdInfo;                          /* pointer to the command array */
	paReturn_t        paret;							/* The expected PA return value */
	Bool			  waitDone;							/* If True wait for the reply before proceeding */
	
	Int				bufferQ;		/* The queue to use to get the buffer for the command */
	pauTestSetupStatus_t status;    /* The command status. Used internally by the setup program */

} pauTestCmdSetSetup_t;


/* A complete test setup */
typedef struct pauTestSetup_s  {
	
	Int 				nMac;			/* Number of MAC setups */
	pauTestMacSetup_t  *macs;			/* Array of MAC setups */
	
	Int					nIps;			/* Number of IP setups */
	pauTestIpSetup_t   *ips;			/* Array of IP setups */
	
	Int					nL4s;			/* Number of L4 setups */
	pauTestL4Setup_t   *l4s;			/* Array of L4 setups */
	
	Int					nCl3Cfgs;		/* Number of custom L3 configurations */
	pauTestCl3Config_t *cl3cfgs;		/* Array of L3 configs */
	
	Int					nCl3s;			/* Number of custom L3 setups */
	pauTestCl3Setup_t  *cl3s;			/* Array of custom L3 setups */
	
	Int					nCl4Cfgs;		/* Number of custom L4 configurations */
	pauTestCl4Config_t *cl4cfgs;		/* Array of L4 configs */
	
	Int					nCl4s;			/* Number of custom L4 setups */
	pauTestCl4Setup_t  *cl4s;			/* Array of custom L4 setups */
    
	Int					nCmdSets;   	/* Number of command sets */
	pauTestCmdSetSetup_t  *cmdSet; 		/* Array of command sets */
    
} pauTestSetup_t;


typedef struct pauUsrStatsSetup_s  {
	int	nStats;                                   /* number of user-defined statistics to be configured */
	paUsrStatsCounterEntryConfig_t* cntEntryTbl;  /* PA IP configuration structure */
	paReturn_t  paret;							  /* The expected PA return value */
	
} pauUsrStatsSetup_t;


/******************************************************************************
 * DATA DEFINITION:  PA Unit Test Payload Type
 ******************************************************************************
 * DESCRIPTION:  Defines the payload types
 ******************************************************************************/
typedef enum pauPayloadType_s
{
    PAU_PAYLOAD_CONST,
    PAU_PAYLOAD_INC8,
    PAU_PAYLOAD_DEC8,
    PAU_PAYLOAD_RANDOM
}   pauPayloadType_e;                                     

/* Prototypes */
Int setupTestFramework (Void);
Int verifyTestFramework (Void);
Void utilCycleDelay (Int count);
UInt32 utilgAddr(UInt32 x);
UInt16 utilCompUdpChksums (pktTestInfo_t *pktinfo, Bool insert);
UInt16 utilCompIpChksums (pktTestInfo_t *pktinfo,  Bool insert);
UInt16 utilGetIpPsudoChkSum (UInt8 *ip, UInt16 payloadLen);
UInt16 utilUpdateIpChksums (UInt8  *ip);


void testGenPayload(pauPayloadType_e type, UInt8 initValue, UInt16 len, UInt8* buf);

Int testCommonRequestPaStats (char *fname, tFramework_t *tf, Bool reset, Int32 QSource, Int32 QRecycle,  paCmdReply_t *reply);
Int testCommonRequestUsrStats (char *fname, tFramework_t *tf, Bool reset, Int32 QSource, Int32 QRecycle,  paCmdReply_t *reply, paUsrStats_t  *usrStats);
Int testCommonCompareStats (char *fname, paSysStats_t *expected, paSysStats_t *actual);
Int testCommonCompareUsrStats (char *fname, paUsrStats_t *expected, paUsrStats_t *actual);
Int testCommonRecycleLBDesc (tFramework_t *tf, Cppi_HostDesc *hd);
Int testCommonComparePktInfo (Char *tfName, pasahoLongInfo_t *expected, pasahoLongInfo_t *actual);
Void testCommonIncStats (paStatsBmap_t *map,  paSysStats_t *stats);
Int testCommonSendPackets (tFramework_t *tf, char *tfName, paSysStats_t *stats, pktTestInfo_t *pktInfo, Int nPackets, Int dest);
Cppi_HostDesc *testCommonAddSrio (tFramework_t *tf, int index, paSrioInfo_t *srioInfo, uint16_t nextHdr, uint16_t nextHdrOffset,
                                 paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, paHandleL2L3_t *l2handle, Int32 Qrecycle, Int32 QCmdMem, 
 	                        	 paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonAddMac (tFramework_t *tf, Int index, paEthInfo_t *ethInfo, paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, 
 	                       		 paHandleL2L3_t *l2handle, Int32 Qrecycle, Int32 QCmdMem, 
 	                        	 paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonDelHandle (tFramework_t *tf, paHandleL2L3_t *paHdl, Int Qrecycle, Int QCmdMem, paCmdReply_t *cmdReply, 
									Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret); 	
Cppi_HostDesc *testCommonDelL4Handles (tFramework_t *tf, paHandleL4_t l4Handle, Int Qrecycle, Int QCmdMem, paCmdReply_t *reply, 
									   Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);									   
Cppi_HostDesc *testCommonAddIp (tFramework_t *tf, Int index, paIpInfo_t *ipInfo, paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, 
 	                       		 paHandleL2L3_t *l3handle, paHandleL2L3_t linkedL2Handle, Int32 Qrecycle, Int32 QCmdMem, 
 	                        	 paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonAddIp2 (tFramework_t *tf, int lutInst,Int index, paIpInfo_t *ipInfo, paRouteInfo_t *matchRoute, paRouteInfo_t *nfailRoute, 
 	                       		 paHandleL2L3_t *l3handle, paHandleL2L3_t linkedL2Handle, Int32 Qrecycle, Int32 QCmdMem, 
 	                        	 paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonAddPort (tFramework_t *tf, int portSize, UInt32 destPort, paRouteInfo_t *matchRoute, paHandleL4_t *l4handle, paHandleL2L3_t *l3handle,
                                Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonAddPort2 (tFramework_t *tf, int portSize, UInt32 destPort, Uint16 fReplace, UInt16 divertQ, paRouteInfo_t *matchRoute, paHandleL4_t *l4handle, paHandleL2L3_t *l3handle,
                                   Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonConfigMultiRoute (tFramework_t *tf, paMultiRouteModes_e mode, uint16_t index, uint16_t nRoute, paMultiRouteEntry_t *routeEntry, 
 	                       		           Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonConfigExceptionRoute (tFramework_t *tf, int nRoute, int *routeTypes, paRouteInfo_t *eRoutes, 
 	                       		               Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonConfigCmdSet (tFramework_t *tf, uint16_t index, int nCmd, paCmdInfo_t *cmdInfo, 
 	                       		       Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonConfigCrcEngine (tFramework_t *tf, uint16_t index, paCrcConfig_t *cfgInfo, 
 	                       		          Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonGlobalConfig (tFramework_t *tf, paCtrlInfo_t *cfgInfo, 
 	                       		       Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
Cppi_HostDesc *testCommonConfigUsrStats (tFramework_t *tf, paUsrStatsConfigInfo_t *cfgInfo, 
 	                       		         Int32 Qrecycle, Int32 QCmdMem, paCmdReply_t *repInfo, Int *cmdDest, UInt16 *cmdSize, paReturn_t *paret);
                                
UInt16 testCommonOnesCompAdd (UInt16 v1, UInt16 v2);
Int testCommonWaitCmdReply (tFramework_t *tf, paTest_t *pat, char *tname, Int Qcmd, UInt32 swinfo0, Int line);
Int testCommonSetupTest (tFramework_t *tf, paSysStats_t *stats, pauTestSetup_t *setup, char *tfName, char *fname, Int fline);
Int testCommonTeardownTest (tFramework_t *tf, paSysStats_t *stats, pauTestSetup_t *setup, char *tfName, char *fname, Int fline);
paTestStatus_t testCommonCheckStats (tFramework_t *tf, paTest_t *pat, char *tName, paSysStats_t *expected, Int32 qSource, Int32 qCmdRecycle, Int32 qReply, Bool clear);
paTestStatus_t testCommonCheckUsrStats (tFramework_t *tf, paTest_t *pat, char *tName, paUsrStats_t *expected, Int32 qSource, Int32 qCmdRecycle, Int32 qReply, Bool clear);
            
Int commonFifoPushElement (pauFifo_t *fifo, UInt elem);
Int commonFifoGetCount (pauFifo_t *fifo);
UInt commonFifoPopElement (pauFifo_t *fifo, Int *numInQBeforePop);                       
 	
Void mdebugHaltPdsp (Int pdspNum);
Void mdebugRunPdsp (Int pdspNum);

/* Tests */
Void paTestUnconfigured (UArg a0, UArg a1);
Void paTestL2Routing (UArg a0, UArg a1);
Void paTestSrioRouting (UArg a0, UArg a1);
Void saSimple (UArg a0, UArg a1);
Void paTestTxFmtRt (UArg a0, UArg a1);
Void paTestL3Routing (UArg a0, UArg a1);
Void paTestL4Routing (UArg a0, UArg a1);
Void paTestPatchRoute (UArg a0, UArg a1);
Void paTestCustom (UArg a0, UArg a1);
Void paTestMultiRouting (UArg a0, UArg a1);
Void paTestIPv4FragReassem (UArg a0, UArg a1);

#endif /* _PAUTEST_H */
