#include "ti/drv/rm/test/rm_test.h"
#include "ti/drv/qmss/qmss_drv.h"
#include "ti/drv/cppi/cppi_drv.h"
#include "ti/drv/cppi/cppi_desc.h"
#include "ti/drv/pa/pa.h"
/* Firmware images */
#include "ti/drv/qmss/qmss_firmware.h"
#include "ti/drv/pa/fw/classify1_bin.c"
#include "ti/drv/pa/fw/classify2_bin.c"
#include "ti/drv/pa/fw/pam_bin.c"

#include <ti/csl/cslr_device.h>

#define CONFIG_MAX_L2_HANDLES        10
#define CONFIG_MAX_L3_HANDLES        20

#define CONFIG_BUFSIZE_PA_INST      256
#define CONFIG_BUFSIZE_L2_TABLE     1000
#define CONFIG_BUFSIZE_L3_TABLE     4000

/* PA instance */
#pragma DATA_ALIGN(paInstBuf, 8)
Uint8 paInstBuf[CONFIG_BUFSIZE_PA_INST];
Pa_Handle paInst;

/* Memory used for PA handles */
#pragma DATA_ALIGN(memL2Ram, 8)
#pragma DATA_ALIGN(memL3Ram, 8)
Uint8 memL2Ram[CONFIG_BUFSIZE_L2_TABLE];
Uint8 memL3Ram[CONFIG_BUFSIZE_L3_TABLE];

/* Hardware Semaphore to synchronize access from
 * multiple applications (PASS applications and non-PASS applications)
 * across different cores to the PA library.
 */
#define     PA_HW_SEM           5 

Void rmTestAddCustomLUT2 (Pa_Handle iHandle, rm_test_expect_e expect)
{
    paReturn_t ret_val;

    ret_val = Pa_addCustomLUT2 (iHandle, pa_MAX_CUSTOM_TYPES_LUT2 - 1, NULL, 0, 0, 
            NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if (ret_val == pa_RESOURCE_USE_DENIED) {
        RM_PRINT_RESULT("Pa_addCustomLUT2", expect, rm_denied, ret_val);
    }
    else {
        RM_PRINT_RESULT("Pa_addCustomLUT2", expect, rm_granted, ret_val);
    }
}

Void rmTestAddSRIO (Pa_Handle iHandle, rm_test_expect_e expect)
{
    paReturn_t ret_val;

    ret_val = Pa_addSrio (iHandle, pa_LUT1_INDEX_NOT_SPECIFIED, NULL, 0, 0, 
            NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if (ret_val == pa_RESOURCE_USE_DENIED) {
        RM_PRINT_RESULT("Pa_addSrio", expect, rm_denied, ret_val);
    }
    else {
        RM_PRINT_RESULT("Pa_addSrio", expect, rm_granted, ret_val);
    }
}

Void rmTestAddMac (Pa_Handle iHandle, rm_test_expect_e expect)
{
    paReturn_t       ret_val;

    ret_val = Pa_addMac  ((Pa_Handle)iHandle,
            pa_LUT1_INDEX_NOT_SPECIFIED, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL);
    
    if (ret_val == pa_RESOURCE_USE_DENIED) {
        RM_PRINT_RESULT("Pa_addMac", expect, rm_denied, ret_val);
    }
    else {
        RM_PRINT_RESULT("Pa_addMac", expect, rm_granted, ret_val);
    }
}

Void rmTestAddPort (Pa_Handle iHandle, rm_test_expect_e expect)
{
    paReturn_t       ret_val;

    ret_val = Pa_addPort  ((Pa_Handle)iHandle,
            pa_LUT2_PORT_SIZE_16, NULL, NULL, NULL, NULL, NULL,
            NULL, NULL, NULL, NULL, NULL);
    
    if (ret_val == pa_RESOURCE_USE_DENIED) {
        RM_PRINT_RESULT("Pa_addPort", expect, rm_denied, ret_val);
    }
    else {
        RM_PRINT_RESULT("Pa_addPort", expect, rm_granted, ret_val);
    }
}

void rm_test_pa(Rm_Handle rm_handle, rm_test_expect_e expect)
{
    paSizeInfo_t  paSize;
    paConfig_t    paCfg;
    int           sizes[pa_N_BUFS];
    int           aligns[pa_N_BUFS];
    void*         bases[pa_N_BUFS];
    paStartCfg_t  paStartCfg;
    paReturn_t    ret_val;
    int i;

    System_printf ("~~~~~~~~~~~~~Core %d RM PA TEST START~~~~~~~~~~~~~~~~\n", DNUM);

    paStartCfg.rmHandle = rm_handle;

    paSize.nMaxL2 = CONFIG_MAX_L2_HANDLES;
    paSize.nMaxL3 = CONFIG_MAX_L3_HANDLES;
    paSize.nUsrStats = 0;

    ret_val = Pa_getBufferReq(&paSize, sizes, aligns);


    bases[0] = (void *)paInstBuf;
    bases[1] = (void *)memL2Ram;
    bases[2] = (void *)memL3Ram;
    bases[3] = 0;

    paCfg.initTable = TRUE;
    paCfg.initDefaultRoute = TRUE;
    paCfg.baseAddr = CSL_PA_SS_CFG_REGS;
    paCfg.sizeCfg = &paSize;

    ret_val = Pa_create (&paCfg, bases, &paInst);

    Pa_startCfg (paInst, &paStartCfg);

    Pa_resetControl (paInst, pa_STATE_RESET);

    /* PDPSs 0-2 use image c1 */
    for (i = 0; i < 3; i++)
    {
        ret_val = Pa_downloadImage (paInst, i, (void *)c1, c1Size);
        if (ret_val == pa_RESOURCE_INIT_DENIED) {
#ifdef _TCI6614_Atrenta_DSP1_H_          
            /* ARM downloads PA firmware, all cores should be denied */
            RM_PRINT_RESULT("Pa_downloadImage[0-2]", rm_denied, rm_denied, ret_val);
#else  /* All other devices without an ARM */
            RM_PRINT_RESULT("Pa_downloadImage[0-2]", expect, rm_denied, ret_val);
#endif
        } else {
#ifdef _TCI6614_Atrenta_DSP1_H_        
            /* ARM downloads PA firmware, all cores should be denied */
            RM_PRINT_RESULT("Pa_downloadImage[0-2]", rm_denied, rm_granted, ret_val);
#else  /* All other devices without an ARM */
            RM_PRINT_RESULT("Pa_downloadImage[0-2]", expect, rm_granted, ret_val);
#endif
        }
    }

    /* PDSP 3 uses image c2 */
    ret_val = Pa_downloadImage (paInst, 3, (void *)c2, c2Size);
    if (ret_val == pa_RESOURCE_INIT_DENIED) {
#ifdef _TCI6614_Atrenta_DSP1_H_          
        /* ARM downloads PA firmware, all cores should be denied */
        RM_PRINT_RESULT("Pa_downloadImage[3]", rm_denied, rm_denied, ret_val);
#else  /* All other devices without an ARM */
        RM_PRINT_RESULT("Pa_downloadImage[3]", expect, rm_denied, ret_val);
#endif
    } else {
#ifdef _TCI6614_Atrenta_DSP1_H_        
        /* ARM downloads PA firmware, all cores should be denied */
        RM_PRINT_RESULT("Pa_downloadImage[3]", rm_denied, rm_granted, ret_val);
#else  /* All other devices without an ARM */
        RM_PRINT_RESULT("Pa_downloadImage[3]", expect, rm_granted, ret_val);
#endif
    }

    /* PDSPs 4-5 use image m */
    for (i = 4; i < 6; i++)
    {
        ret_val = Pa_downloadImage (paInst, i, (void *)m, mSize);
        if (ret_val == pa_RESOURCE_INIT_DENIED) {
#ifdef _TCI6614_Atrenta_DSP1_H_              
            /* ARM downloads PA firmware, all cores should be denied */
            RM_PRINT_RESULT("Pa_downloadImage[4-6]", rm_denied, rm_denied, ret_val);
#else  /* All other devices without an ARM */
            RM_PRINT_RESULT("Pa_downloadImage[4-6]", expect, rm_denied, ret_val);
#endif
        } else {
#ifdef _TCI6614_Atrenta_DSP1_H_            
            /* ARM downloads PA firmware, all cores should be denied */
            RM_PRINT_RESULT("Pa_downloadImage[4-6]", rm_denied, rm_granted, ret_val);
#else  /* All other devices without an ARM */
            RM_PRINT_RESULT("Pa_downloadImage[4-6]", expect, rm_granted, ret_val);
#endif
        }
    }

#ifdef _TCI6614_Atrenta_DSP1_H_    
    /* MAC uses LUT1-0 which is always used by ARM.  AddMac should be denied for all cores */
    rmTestAddMac(paInst, rm_denied);
    /* SRIO uses LUT1-0 which is always used by ARM.  AddSrio should be denied for all cores */
    rmTestAddSRIO (paInst, rm_denied);
#else  /* All other devices without an ARM */
    rmTestAddMac(paInst, expect);

    rmTestAddSRIO (paInst, expect);
#endif

    /*rmTestAddIP(paInst, expect);*/

    rmTestAddPort(paInst, expect);

    /*rmTestAddCustomLUT1(paInst, expect);*/

    rmTestAddCustomLUT2(paInst, expect);
    
    System_printf ("~~~~~~~~~~~~~Core %d RM PA TEST DONE~~~~~~~~~~~~~~~~\n", DNUM);
}


UInt32      paMemProtNestedLevel= 0;

/**
 * @brief  This macro is used to alert the application that the PA is
 *         going to access table memory. The application must ensure
 *         cache coherency and semaphores for multi-core applications
 *
 *
 * <b> Prototype: </b>
 *  The following is the C prototype for the expected OSAL API.
 *
 *  @verbatim
        void Osal_paBeginMemAccess (void* addr, uint32_t sizeWords)
    @endverbatim
 *
 *  <b> Parameters </b>
 *  @n  The address of the table to be accessed
 *  @n  The number of bytes in the table
 *
 *  @note PA will make nested calls to this function for memory access
 *        protection of different memory tables. The multicore semaphore
 *        should be allocated only for the first call of a nested group 
 *        of calls. 
 */
 

void Osal_paBeginMemAccess (Ptr addr, UInt32 size)
{
    uint32_t    key;
    
    /* Disable Interrupts */
    key = Hwi_disable();

    /* Cleanup the prefetch buffer also. */
    CSL_XMC_invalidatePrefetchBuffer();
    
    SYS_CACHE_INV (addr, size, CACHE_FENCE_WAIT);
    
    asm   (" nop      4");
    asm   (" nop      4");
    asm   (" nop      4");
    asm   (" nop      4");

    /* Reenable Interrupts. */
    Hwi_restore(key);

}

/**
 * @brief  This macro is used to alert the application that the PA
 *         has completed access to table memory. This call will always
 *         be made following a call to Osal_paBeginMemAccess and have
 *         the same parameters
 *
 * <b> Prototype: </b>
 *  The following is the C prototype for the expected OSAL API.
 *
 *  @verbatim
        void Osal_paEndMemAccess (void* addr, uint32_t sizeWords)
    @endverbatim
 *
 *  <b> Parameters </b>
 *  @n The address of the table to be accessed
 *  @n The number of bytes in the table
 *
 *  @note PA will make nested calls to this function for memory access
 *        protection of different memory tables. The multicore semaphore
 *        should be freed when all previous memory access has completed,
 *        in other words, when the nested call level reaches 0.
 */

void Osal_paEndMemAccess (Ptr addr, UInt32 size)
{
    uint32_t    key;
    
    /* Disable Interrupts */
    key = Hwi_disable();
    
    SYS_CACHE_WB (addr, size, CACHE_FENCE_WAIT);

    asm   (" nop      4");
    asm   (" nop      4");
    asm   (" nop      4");
    asm   (" nop      4");

    /* Reenable Interrupts. */
    Hwi_restore(key);
    
}

/**
 *  @b Description
 *  @n  
 *      The function is used to enter a critical section.
 *      Function protects against 
 *
 *      access from multiple threads on single core
 *      and
 *      access from multiple cores 
 *
 *  @param[in]  key
 *      Key used to lock the critical section.
 *
 *  @retval
 *      Not Applicable
 */
Void Osal_paMtCsEnter (uint32_t *key)
{

    /* Get the hardware semaphore. 
     *
     * Acquire Multi core PA synchronization lock 
     */
     while ((CSL_semAcquireDirect (PA_HW_SEM)) == 0);
     *key = 0;
}

/**
 *  @b Description
 *  @n  
 *      The function is used to exit a critical section 
 *      protected using Osal_salldCsEnter() API.
 *
 *  @param[in]  key
 *      Key used to unlock the critical section.
 *
 *  @retval
 *      Not Applicable
 */
Void Osal_paMtCsExit (uint32_t key)
{
    /* Release the hardware semaphore */
    CSL_semReleaseSemaphore (PA_HW_SEM);

}
