/**
 *   @file  infrastructure_mode.c
 *
 *   @brief   
 *      This is the QMSS infrastructure mode example code in Linux User space. 
 *      Runs only in polling mode.
 *      Uses both host and monolithic descriptors for data transfer.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  \par
*/

#include "fw_test.h"
#include "qmss_test.h"

#include <string.h>

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>

/* CPPI LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>


/************************ GLOBAL VARIABLES ********************/
/* Memory region configuration information */
Qmss_MemRegInfo         monoMemInfo;
/* Memory region configuration information */
Qmss_MemRegInfo         hostMemInfo;
/* Tx channel configuration */
Cppi_TxChInitCfg        txChCfg;
/* Rx channel configuration */
Cppi_RxChInitCfg        rxChCfg;
/* Rx flow configuration */
Cppi_RxFlowCfg          rxFlowCfg;
/* Accumulator configuration */
#ifdef CONFIG_ACC
Qmss_AccCmdCfg          cfg;
/* List address for accumulator - twice the number of entries for Ping and Pong page */
#pragma DATA_ALIGN (hiPrioList, 16)
uint32_t                  hiPrioList[(NUM_PACKETS + 1) * 2];
#endif

/**********************************FUNCTIONS**********************************/

#ifdef CONFIG_ACC
void testQueueReclaim (Qmss_QueueHnd freeHostQueHnd)
{
    Qmss_Result             result;
    Qmss_QueueHnd           reclaimQueHnd, returnQueHnd;
    Cppi_Desc               *hostDescPtr, *rxPkt;
    Qmss_Queue              queInfo;
    uint8_t                   isAllocated;

    /* Opens reclamation queue. */
    if ((reclaimQueHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening reclamation Queue Number\n");
		return;
	}
    else
        printf ("Reclaimation Queue Number : %d opened\n", reclaimQueHnd);

    /* Opens return queue. */
    if ((returnQueHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening return Queue Number\n");
		return;
	}
    else
        printf ("Return Queue Number : %d opened\n", returnQueHnd);

   
    if ((result = Qmss_programReclaimQueue (Qmss_PdspId_PDSP1, reclaimQueHnd)) != QMSS_ACC_SOK)
	{
        printf ("Error: Configuring reclamation queue : %d error code : %d\n",
                        reclaimQueHnd, result);
		return;
	}
    else
        printf ("Configured reclamation queue : %d\n", 
                        reclaimQueHnd);

    if ((hostDescPtr = (Cppi_Desc *) Qmss_queuePop (freeHostQueHnd)) == NULL)
    {
        printf ("Error: Getting descriptor from Queue Number %d\n", freeHostQueHnd);
        return;
    }
    else
        printf ("Descriptor address 0x%p\n", hostDescPtr);

    queInfo = Qmss_getQueueNumber (returnQueHnd);
    
    Cppi_setReturnQueue (Cppi_DescType_HOST, hostDescPtr, queInfo);

    /* Push descriptor to Tx free queue */
    Qmss_queuePushDesc (reclaimQueHnd, (uint32_t *) hostDescPtr);
    while (Qmss_getQueueEntryCount (returnQueHnd) == 0);

    while ((rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (Qmss_queuePop (returnQueHnd))) != NULL) 
    {
        printf ("Reclaimed descriptor address 0x%p\n", rxPkt);
    }

    if ((result = Qmss_programReclaimQueue (Qmss_PdspId_PDSP1, 0)) != QMSS_ACC_SOK)
    {
        printf ("Error: Disabling reclamation queue : %d error code : %d\n",
                        reclaimQueHnd, result);
		return;
	}
    else
        printf ("Disabled reclamation queue : %d\n", 
                        reclaimQueHnd);
    if ((hostDescPtr = (Cppi_Desc *) Qmss_queuePop (freeHostQueHnd)) == NULL)
    {
        printf ("Error: Getting descriptor from Queue Number %d\n", freeHostQueHnd);
        return;
    }
    else
        printf ("Descriptor address 0x%p\n", hostDescPtr);

    Cppi_setReturnQueue (Cppi_DescType_HOST, hostDescPtr, queInfo);

    /* Push descriptor to Tx free queue */
    Qmss_queuePushDesc (reclaimQueHnd, (uint32_t *) hostDescPtr);
   
    /* Descriptor should not be reclaimed */ 
    if ((Qmss_getQueueEntryCount (returnQueHnd) != 0))
    {
        printf ("Error: Descriptor count on return queue %d is %d\n", 
                        returnQueHnd, Qmss_getQueueEntryCount (returnQueHnd));
        return;
    }

    printf ("\nQueue reclamation feature Passed\n");
    return;
}
#endif

/**
 *  @b Description
 *  @n  
 *      Data transfer using host descriptors 
 *
 *  @param[in]  cppiHnd
 *      Handle to access CPPI queue manager CPDMA
 *
 *  @retval
 *      None
 */
void hostModeTransfer (Cppi_Handle cppiHnd)
{
    Qmss_Result             result;
    uint32_t                  numAllocated, i, length, destLen;
    uint8_t                   isAllocated;
    uint8_t                   *dataBuffPtr;
    Cppi_ChHnd              rxChHnd, txChHnd;
    Qmss_QueueHnd           txQueHnd, rxQueHnd, freeQueHnd, txCmplQueHnd, txFreeQueHnd;
    Cppi_DescCfg            descCfg;
    Cppi_Desc               *hostDescPtr;
    Cppi_FlowHnd            rxFlowHnd;
    Qmss_Queue              queInfo;
    Cppi_Desc               *rxPkt;
    
#ifdef CONFIG_ACC
    uint32_t                  index, base, count, packetCount;
    Bool                    usePing = TRUE;
    volatile uint32_t         temp;
#endif

    printf ("\n*************** Data transfer host mode *****************\n\n");

    /* Setup the descriptors for transmit free queue */
    /* Memory region obtained is zero since there is only Qmss_insertMemoryRegion() call. 
     * else find the memory region using Qmss_getMemoryRegionCfg() */
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION0;
    descCfg.descNum = NUM_PACKETS;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_STARVATION_COUNTER_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_HOST;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;

    /* Descriptor should be recycled back to Queue Number 1000 */
    descCfg.returnQueue.qMgr = CPPI_COMPLETION_QUE_MGR;
    descCfg.returnQueue.qNum = CPPI_COMPLETION_QUE_NUM;

    /* Initialize the descriptors and push to free Queue */
    if ((txFreeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error: Initializing Tx descriptor error code: %d \n", txFreeQueHnd);
		return;
	}
    else
        printf ("Number of Tx descriptors requested : %d. Number of descriptors allocated : %d \n", 
            descCfg.descNum, numAllocated);

    /* Setup the descriptors for receive free queue */
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION0;
    descCfg.descNum = NUM_HOST_DESC - NUM_PACKETS;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_STARVATION_COUNTER_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_HOST;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;
    
    /* Descriptor should be recycled back to queue allocated above */
    descCfg.returnQueue.qMgr = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.returnQueue.qNum = QMSS_PARAM_NOT_SPECIFIED;
    
    /* Initialize the descriptors and push to free Queue */
    if ((freeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error: Initializing Rx descriptor error code: %d \n", freeQueHnd);
		return;
	}
    else
        printf ("Number of Rx descriptors requested : %d. Number of descriptors allocated : %d \n", 
            descCfg.descNum, numAllocated);

    /* Setup Rx descriptors with receive buffers */
    for (i = 0; i < numAllocated; i++)
    {
        Ptr dataPtr = rxDataBuff + i * SIZE_DATA_BUFFER;

        /* Get a descriptor */
        if ((hostDescPtr = (Cppi_Desc *) Qmss_queuePop (freeQueHnd)) == NULL)
        {
            printf ("Error: Getting descriptor from Queue Number\n", freeQueHnd);
            return;
        }

        /* Add data buffer */
        Cppi_setData (Cppi_DescType_HOST, hostDescPtr, (uint8_t *) dataPtr, SIZE_DATA_BUFFER);

        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, hostDescPtr, (uint8_t *) dataPtr, SIZE_DATA_BUFFER);

        /* Push descriptor to Rx free queue */
        Qmss_queuePushDesc (freeQueHnd, (uint32_t *) hostDescPtr);
    }
    
    /* Set up Rx Channel parameters */
    rxChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    rxChCfg.rxEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Rx Channel */
    rxChHnd = (Cppi_ChHnd) Cppi_rxChannelOpen (cppiHnd, &rxChCfg, &isAllocated);
    if (rxChHnd == NULL)
    {
        printf ("Error: Opening Rx channel : %d\n", rxChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Rx channel : %d\n", Cppi_getChannelNumber (rxChHnd));


    /* Set up Tx Channel parameters */
    txChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    txChCfg.priority = 0;
    txChCfg.filterEPIB = 0;
    txChCfg.filterPS = 0;
    txChCfg.aifMonoMode = 0;
    txChCfg.txEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Tx Channel */
    txChHnd = (Cppi_ChHnd) Cppi_txChannelOpen (cppiHnd, &txChCfg, &isAllocated);
    if (txChHnd == NULL)
    {
        printf ("Error: Opening Tx channel : %d\n", txChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Tx channel : %d\n", Cppi_getChannelNumber (txChHnd));

    /* Opens transmit queue. This is the infrastructure queue */
    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_INFRASTRUCTURE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Transmit Queue Number\n");
		return;
	}
    else
        printf ("Transmit Queue Number : %d opened\n", txQueHnd);

    /* Opens receive queue */
    if ((rxQueHnd = Qmss_queueOpen (Qmss_QueueType_HIGH_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Receive Queue Number\n");
        return;
	}
    else
        printf ("Receive Queue Number : %d opened\n", rxQueHnd);

    /* Opens transmit completion queue. */
    if ((txCmplQueHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, CPPI_COMPLETION_QUE_NUM, &isAllocated)) < 0)
	{
        printf ("Error: Opening Tx Completion Queue Number\n");
		return;
	}
    else
        printf ("Tx Completion Queue Number : %d opened\n", txCmplQueHnd);

    printf ("Free Queue Number : %d opened\n", freeQueHnd);
    printf ("Transmit Free Queue Number : %d opened\n", txFreeQueHnd);

#ifdef CONFIG_ACC

    if ((result = Qmss_configureAccTimer (Qmss_PdspId_PDSP1, 3500)) != QMSS_ACC_SOK)
	{
        printf ("Error: Changing accumulator tick to 20us error code : %d\n",
                        cfg.channel, cfg.queMgrIndex, result);
		return;
	}
    else
        printf ("Changed accumulator tick to 20us\n");


    /* program the high priority accumulator */
    memset ((void *) &hiPrioList, 0, sizeof (hiPrioList));
    cfg.channel = 0;
    cfg.command = Qmss_AccCmd_ENABLE_CHANNEL;
    cfg.queueEnMask = 0;
    cfg.listAddress = hiPrioList; /* Should be global if reading on another core */
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    cfg.queMgrIndex = queInfo.qNum;
    cfg.maxPageEntries = ACC_ENTRY_SIZE;
    cfg.timerLoadCount = 0;
    cfg.interruptPacingMode = Qmss_AccPacingMode_NONE;
    cfg.listEntrySize = Qmss_AccEntrySize_REG_D;
    cfg.listCountMode = Qmss_AccCountMode_ENTRY_COUNT;
    cfg.multiQueueMode = Qmss_AccQueueMode_SINGLE_QUEUE;
    
    if ((result = Qmss_programAccumulator (Qmss_PdspId_PDSP1, &cfg)) != QMSS_ACC_SOK)
	{
        printf ("Error: Programming high priority accumulator for channel : %d queue : %d error code : %d\n",
                        cfg.channel, cfg.queMgrIndex, result);
		return;
	}
    else
        printf ("high priority accumulator programmed for channel : %d queue : %d\n", 
                        cfg.channel, cfg.queMgrIndex);
#endif

    /* Set transmit queue threshold to high and when there is atleast one packet */
    /* Setting threshold on transmit queue is not required anymore. tx pending queue is not hooked to threshold. 
     * Qmss_setQueueThreshold (txQueHnd, 1, 1);
     */ 
            
    /* Setup Rx flow parameters */
    memset ((void *) &rxFlowCfg, 0, sizeof (Cppi_RxFlowCfg));

    /* Don't specify flow number and let CPPI allocate the next available one */
    rxFlowCfg.flowIdNum = CPPI_PARAM_NOT_SPECIFIED;
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    rxFlowCfg.rx_dest_qnum = queInfo.qNum;
    rxFlowCfg.rx_dest_qmgr = queInfo.qMgr;
    rxFlowCfg.rx_desc_type = Cppi_DescType_HOST; 
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (freeQueHnd);
    rxFlowCfg.rx_fdq0_sz0_qnum = queInfo.qNum;
    rxFlowCfg.rx_fdq0_sz0_qmgr = queInfo.qMgr;

    /* Configure Rx flow */
    rxFlowHnd = (Cppi_FlowHnd) Cppi_configureRxFlow (cppiHnd, &rxFlowCfg, &isAllocated);
    if (rxFlowHnd == NULL)
    {
        printf ("Error: Opening Rx flow : %d\n", rxFlowCfg.flowIdNum);
        return;
    }
    else 
        printf ("Opened Rx flow : %d\n", Cppi_getFlowId(rxFlowHnd));
    
    /* Enable transmit channel */
    if (Cppi_channelEnable (txChHnd) != CPPI_SOK)
        printf ("Error: Enabling Tx channel : %d\n", Cppi_getChannelNumber (txChHnd));
    else 
        printf ("Tx channel : %d enabled \n", Cppi_getChannelNumber (txChHnd));

    /* Enable receive channel */
    if (Cppi_channelEnable (rxChHnd) != CPPI_SOK)
        printf ("Error: Enabling Rx channel : %d\n", Cppi_getChannelNumber (rxChHnd));
    else 
        printf ("Rx channel : %d enabled \n", Cppi_getChannelNumber (rxChHnd));


    /* Fill in some data */
    for (i = 0; i < SIZE_DATA_BUFFER; i++) 
        txDataBuff[i] = i;

    printf ("\n--------------------Transmitting packets----------------------\n");
    /* Send out 8 packets */
    for (i = 0; i < NUM_PACKETS; i++)
    {
        /* Get a free descriptor */
        if ((hostDescPtr = (Cppi_Desc *) Qmss_queuePop (txFreeQueHnd)) == NULL)
        {
            printf ("Error: Getting descriptor from Queue Number\n", txFreeQueHnd);
            return;
        }

        /* Add data buffer */
        Cppi_setData (Cppi_DescType_HOST, hostDescPtr, (uint8_t *) txDataBuff, SIZE_DATA_BUFFER);

        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, hostDescPtr, (uint8_t *) txDataBuff, SIZE_DATA_BUFFER);

        /* Set packet length */
        Cppi_setPacketLen (Cppi_DescType_HOST, hostDescPtr, SIZE_DATA_BUFFER);

        printf ("Transmitting descriptor 0x%p\n", hostDescPtr);

        /* Push descriptor to Tx queue */
        Qmss_queuePushDescSize (txQueHnd, (uint32_t *) hostDescPtr, SIZE_HOST_DESC);
    }

    printf ("\n-------------------------Queue status-------------------------\n");
    result = Qmss_getQueueEntryCount (txQueHnd);
    printf ("Transmit Queue %d Entry Count : %d \n", txQueHnd, result);

    result = Qmss_getQueueEntryCount (txFreeQueHnd);
    printf ("Tx Free Queue %d Entry Count : %d \n", txFreeQueHnd, result);

    result = Qmss_getQueueEntryCount (freeQueHnd);
    printf ("Rx Free Queue %d Entry Count : %d \n", freeQueHnd, result);

    result = Qmss_getQueueEntryCount (rxQueHnd);
    printf ("Receive Queue %d Entry Count : %d \n", rxQueHnd, result);

    result = Qmss_getQueueEntryCount (txCmplQueHnd);
    printf ("Tx completion Queue %d Entry Count : %d \n", txCmplQueHnd, result);

    printf ("\n--------------------Receiving packets-------------------------\n");

#ifndef CONFIG_ACC
    while (Qmss_getQueueEntryCount (rxQueHnd) == 0);
    
    /* Get the rx packet */
    while ((rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (Qmss_queuePop (rxQueHnd))) != NULL)
    {
        length = Cppi_getPacketLen (Cppi_DescType_HOST, rxPkt);

        printf ("Received descriptor 0x%p of length : %d\n", rxPkt, length);

        /* Get data buffer */
        Cppi_getData (Cppi_DescType_HOST, rxPkt, &dataBuffPtr, &destLen);

        /* Compare */
        for (i = 0; i < destLen; i++)
        {
            if (txDataBuff[i] != dataBuffPtr[i])
                printf ("Error: In data buffer Tx: %02X - Rx: %02X \n", txDataBuff[i], dataBuffPtr[i]);
        }
        
        /* Recycle the descriptors */
        queInfo = Cppi_getReturnQueue (Cppi_DescType_HOST, rxPkt);
        
        /* Push descriptor back to Rx free queue */
        Qmss_queuePushDesc (Qmss_getQueueHandle(queInfo), (uint32_t *) rxPkt);
    }
#else
    {
        /* Burn some time for the accumulators to run. */
#if 1 
		temp = 0;
        for (i = 0; i < 50000; i++)
        {
          temp = i;
        }
        temp = 1;
#endif
    }

    /* Get the Rx packet */
    /* 8 packets were sent out. Check how many where received. Count should be the first entry since acc 
     * is programmed in entry count. Start with the ping side */
    packetCount = 0;
    while (1)
    {
        if (usePing == TRUE)
        {
            /* Use Ping side */
            base = 0;
            usePing = FALSE;
            printf ("Processing ping page\n");
        }
        else
        {
            /* Switch to pong side */
            base = ACC_ENTRY_SIZE; 
            usePing = TRUE;
            printf ("Processing pong page\n");
        }
        
        /* Using Entry count mode */
        count = hiPrioList[base];
        hiPrioList[base] = 0;
        for (index = base + 1; index <= base + count; index++)
        {
            rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (hiPrioList[index]);
            hiPrioList[index] = 0;

            /* Get packet length */
            length = Cppi_getPacketLen (Cppi_DescType_HOST, rxPkt);

            printf ("Received descriptor 0x%p of length : %d\n", rxPkt, length);

            /* Get data buffer */
            Cppi_getData (Cppi_DescType_HOST, rxPkt, &dataBuffPtr, &destLen);

            /* Compare */
            for (i = 0; i < destLen; i++)
            {
                if (txDataBuff[i] != dataBuffPtr[i])
                 printf ("Error: In data buffer Tx: %02X - Rx: %02X \n", txDataBuff[i], dataBuffPtr[i]);
            }
        
            /* Recycle the descriptors */
            queInfo = Cppi_getReturnQueue (Cppi_DescType_HOST, rxPkt);
       
            /* Push descriptor back to Rx free queue */
            Qmss_queuePushDesc (Qmss_getQueueHandle(queInfo), (uint32_t *) rxPkt);
            packetCount++;
        }
        Qmss_ackInterrupt (cfg.channel, 1);
        Qmss_setEoiVector (Qmss_IntdInterruptType_HIGH, cfg.channel);

        if (packetCount == NUM_PACKETS)
            break;
    }
#endif

    printf ("\n--------------------Deinitializing----------------------------\n");

    result = Qmss_getQueueEntryCount (txQueHnd);
    printf ("Transmit Queue %d Entry Count : %d \n", txQueHnd, result);

    result = Qmss_getQueueEntryCount (txFreeQueHnd);
    printf ("Tx Free Queue %d Entry Count : %d \n", txFreeQueHnd, result);

    result = Qmss_getQueueEntryCount (freeQueHnd);
    printf ("Rx Free Queue %d Entry Count : %d \n", freeQueHnd, result);

    result = Qmss_getQueueEntryCount (rxQueHnd);
    printf ("Receive Queue %d Entry Count : %d \n", rxQueHnd, result);

    result = Qmss_getQueueEntryCount (txCmplQueHnd);
    printf ("Tx completion Queue %d Entry Count : %d \n", txCmplQueHnd, result);

#ifdef CONFIG_ACC
    /* Disable accumulator */
    if ((result = Qmss_disableAccumulator (Qmss_PdspId_PDSP1, cfg.channel)) != QMSS_ACC_SOK)
	{
        printf ("Error: Disabling high priority accumulator for channel : %d queue : %d error code: %d\n",
                        cfg.channel, cfg.queMgrIndex, result);
		return;
	}
    else
        printf ("high priority accumulator disabled for channel : %d queue : %d\n", 
                        cfg.channel, cfg.queMgrIndex);
#endif

#ifdef CONFIG_ACC
    /* Test queue reclamation feature */
    testQueueReclaim (txCmplQueHnd);
#endif

    /* Flush the queues */
    Qmss_queueEmpty (txFreeQueHnd);
    Qmss_queueEmpty (freeQueHnd);
    Qmss_queueEmpty (txCmplQueHnd);

    /* Close Tx channel */
    if ((result = Cppi_channelClose (txChHnd)) != CPPI_SOK)
        printf ("Error: Closing Tx channel error code : %d\n", result);
    else
        printf ("Tx Channel closed successfully. Ref count : %d\n", result);

    /* Close Rx channel */
    if ((result = Cppi_channelClose (rxChHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx channel error code : %d\n", result);
    else
        printf ("Rx Channel closed successfully. Ref count : %d\n", result);

    /* Close Rx flow */
    if ((result = Cppi_closeRxFlow (rxFlowHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx flow error code : %d\n", result);
    else
        printf ("Rx flow closed successfully. Ref count : %d\n", result);
    
    /* Close the queues */
    if ((result = Qmss_queueClose (rxQueHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx queue error code : %d\n", result);
    else
        printf ("Rx queue closed successfully. Ref count : %d\n", result);
    
    if ((result = Qmss_queueClose (txQueHnd)) != CPPI_SOK)
        printf ("Error: Closing tx queue error code : %d\n", result);
    else
        printf ("Tx queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (freeQueHnd)) != CPPI_SOK)
        printf ("Error: Closing free queue error code : %d\n", result);
    else
        printf ("Free queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (txCmplQueHnd)) != CPPI_SOK)
        printf ("Error: Closing transmit completion queue error code : %d\n", result);
    else
        printf ("Transmit completion queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (txFreeQueHnd)) != CPPI_SOK)
        printf ("Error: Closing transmit freequeue error code : %d\n", result);
    else
        printf ("Transmit free queue closed successfully. Ref count : %d\n", result);
}

/**
 *  @b Description
 *  @n  
 *      Data transfer using monolithic descriptors 
 *
 *  @param[in]  cppiHnd
 *      Handle to access CPPI queue manager CPDMA
 *
 *  @retval
 *      None
 */

void monoModeTransfer (Cppi_Handle cppiHnd)
{
    Qmss_Result             result;
    uint32_t                  numAllocated, i, length, destLen;
    uint8_t                   isAllocated;
    uint8_t                   *dataBuffPtr;
    Cppi_ChHnd              rxChHnd, txChHnd;
    Qmss_QueueHnd           txQueHnd, rxQueHnd, freeQueHnd, txCmplQueHnd, txFreeQueHnd;
    Cppi_DescCfg            descCfg;
    Cppi_Desc               *monoDescPtr;
    Cppi_FlowHnd            rxFlowHnd;
    Qmss_Queue              queInfo;
    Cppi_Desc               *rxPkt;
#ifdef CONFIG_ACC
    uint32_t                  index, base, packetCount;
    Bool                    usePing = TRUE;
    volatile uint32_t         temp; 
#endif

    printf ("\n***************** Data transfer monolithic mode ***************\n\n");

    /* Setup the descriptors for transmit free queue */
    /* Memory region obtained is zero since there is only Qmss_insertMemoryRegion() call. 
     * else find the memory region using Qmss_getMemoryRegionCfg() */
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION1;
    descCfg.descNum = NUM_PACKETS;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_STARVATION_COUNTER_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_MONOLITHIC;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;

    descCfg.cfg.mono.dataOffset = MONOLITHIC_DESC_DATA_OFFSET;
    
    /* Descriptor should be recycled back to Queue Number 1000 */
    descCfg.returnQueue.qMgr = CPPI_COMPLETION_QUE_MGR;
    descCfg.returnQueue.qNum = CPPI_COMPLETION_QUE_NUM;

    
    /* Initialize the descriptors and push to free Queue */
    if ((txFreeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error: Initializing Tx descriptor error code: %d \n", txFreeQueHnd);
		return;
	}
    else
        printf ("Number of Tx descriptors requested : %d. Number of descriptors allocated : %d \n", 
            descCfg.descNum, numAllocated);

    /* Setup the descriptors for receive free queue */
    descCfg.memRegion = Qmss_MemRegion_MEMORY_REGION1;
    descCfg.descNum = NUM_MONOLITHIC_DESC - NUM_PACKETS;
    descCfg.destQueueNum = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.queueType = Qmss_QueueType_STARVATION_COUNTER_QUEUE;
    descCfg.initDesc = Cppi_InitDesc_INIT_DESCRIPTOR;
    descCfg.descType = Cppi_DescType_MONOLITHIC;
    descCfg.epibPresent = Cppi_EPIB_NO_EPIB_PRESENT;
    descCfg.cfg.mono.dataOffset = MONOLITHIC_DESC_DATA_OFFSET;
    
    /* Descriptor should be recycled back to queue allocated above */
    descCfg.returnQueue.qMgr = QMSS_PARAM_NOT_SPECIFIED;
    descCfg.returnQueue.qNum = QMSS_PARAM_NOT_SPECIFIED;
    
    /* Initialize the descriptors and push to free Queue */
    if ((freeQueHnd = Cppi_initDescriptor (&descCfg, &numAllocated)) < 0)
	{
        printf ("Error: Initializing Rx descriptor error code: %d \n", freeQueHnd);
		return;
	}
    else
        printf ("Number of Rx descriptors requested : %d. Number of descriptors allocated : %d \n", 
            descCfg.descNum, numAllocated);

    /* Set up Rx Channel parameters */
    rxChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    rxChCfg.rxEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Rx Channel */
    rxChHnd = (Cppi_ChHnd) Cppi_rxChannelOpen (cppiHnd, &rxChCfg, &isAllocated);
    if (rxChHnd == NULL)
    {
        printf ("Error: Opening Rx channel : %d\n", rxChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Rx channel : %d\n", Cppi_getChannelNumber (rxChHnd));


    /* Set up Tx Channel parameters */
    txChCfg.channelNum = CPPI_PARAM_NOT_SPECIFIED;
    txChCfg.priority = 0;
    txChCfg.filterEPIB = 0;
    txChCfg.filterPS = 0;
    txChCfg.aifMonoMode = 0;
    txChCfg.txEnable = Cppi_ChState_CHANNEL_DISABLE;
    
    /* Open Tx Channel */
    txChHnd = (Cppi_ChHnd) Cppi_txChannelOpen (cppiHnd, &txChCfg, &isAllocated);
    if (txChHnd == NULL)
    {
        printf ("Error: Opening Tx channel : %d\n", txChCfg.channelNum);
        return;
    }
    else 
        printf ("Opened Tx channel : %d\n", Cppi_getChannelNumber (txChHnd));

    /* Opens transmit queue. This is the infrastructure queue */
    if ((txQueHnd = Qmss_queueOpen (Qmss_QueueType_INFRASTRUCTURE_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Transmit Queue Number\n");
		return;
	}
    else
        printf ("Transmit Queue Number : %d opened\n", txQueHnd);

    /* Opens receive queue */
    if ((rxQueHnd = Qmss_queueOpen (Qmss_QueueType_HIGH_PRIORITY_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAllocated)) < 0)
	{
        printf ("Error: Opening Receive Queue Number\n");
        return;
	}
    else
        printf ("Receive Queue Number : %d opened\n", rxQueHnd);

    /* Opens transmit completion queue. */
    if ((txCmplQueHnd = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, CPPI_COMPLETION_QUE_NUM, &isAllocated)) < 0)
	{
        printf ("Error: Opening Tx Completion Queue Number\n");
		return;
	}
    else
        printf ("Tx Completion Queue Number : %d opened\n", txCmplQueHnd);

    printf ("Free Queue Number : %d opened\n", freeQueHnd);
    printf ("Transmit Free Queue Number : %d opened\n", txFreeQueHnd);

#ifdef CONFIG_ACC
    /* program the high priority accumulator */
    memset ((void *) &hiPrioList, 0, sizeof (hiPrioList));
    cfg.channel = 0;
    cfg.command = Qmss_AccCmd_ENABLE_CHANNEL;
    cfg.queueEnMask = 0;
    cfg.listAddress = hiPrioList; /* Should be global if reading on another core */
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    cfg.queMgrIndex = queInfo.qNum;
    cfg.maxPageEntries = ACC_ENTRY_SIZE;
    cfg.timerLoadCount = 0;
    cfg.interruptPacingMode = Qmss_AccPacingMode_NONE;
    cfg.listEntrySize = Qmss_AccEntrySize_REG_D;
    cfg.listCountMode = Qmss_AccCountMode_NULL_TERMINATE;
    cfg.multiQueueMode = Qmss_AccQueueMode_SINGLE_QUEUE;
    
    if ((result = Qmss_programAccumulator (Qmss_PdspId_PDSP1, &cfg)) != QMSS_ACC_SOK)
	{
        printf ("Error: Programming high priority accumulator for channel : %d queue : %d error code : %d\n",
                        cfg.channel, cfg.queMgrIndex, result);
		return;
	}
    else
        printf ("high priority accumulator programmed for channel : %d queue : %d\n", 
                        cfg.channel, cfg.queMgrIndex);
#endif

    /* Set transmit queue threshold to high and when there is atleast one packet */
    /* Setting threshold on transmit queue is not required anymore. tx pending queue is not hooked to threshold. 
     * Qmss_setQueueThreshold (txQueHnd, 1, 1);
     */ 
            
    /* Setup Rx flow parameters */
    memset ((void *) &rxFlowCfg, 0, sizeof (Cppi_RxFlowCfg));

    /* Don't specify flow number and let CPPI allocate the next available one */
    rxFlowCfg.flowIdNum = CPPI_PARAM_NOT_SPECIFIED;
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (rxQueHnd);
    rxFlowCfg.rx_dest_qnum = queInfo.qNum;
    rxFlowCfg.rx_dest_qmgr = queInfo.qMgr;
    rxFlowCfg.rx_sop_offset = MONOLITHIC_DESC_DATA_OFFSET;
    rxFlowCfg.rx_desc_type = Cppi_DescType_MONOLITHIC; 
    /* Get queue manager and queue number from handle */
    queInfo = Qmss_getQueueNumber (freeQueHnd);
    rxFlowCfg.rx_fdq0_sz0_qnum = queInfo.qNum;
    rxFlowCfg.rx_fdq0_sz0_qmgr = queInfo.qMgr;

    /* Configure Rx flow */
    rxFlowHnd = (Cppi_FlowHnd) Cppi_configureRxFlow (cppiHnd, &rxFlowCfg, &isAllocated);
    if (rxFlowHnd == NULL)
    {
        printf ("Error: Opening Rx flow : %d\n", rxFlowCfg.flowIdNum);
        return;
    }
    else 
        printf ("Opened Rx flow : %d\n", Cppi_getFlowId(rxFlowHnd));
    
    /* Enable transmit channel */
    if (Cppi_channelEnable (txChHnd) != CPPI_SOK)
        printf ("Error: Enabling Tx channel : %d\n", Cppi_getChannelNumber (txChHnd));
    else 
        printf ("Tx channel : %d enabled \n", Cppi_getChannelNumber (txChHnd));

    /* Enable receive channel */
    if (Cppi_channelEnable (rxChHnd) != CPPI_SOK)
        printf ("Error: Enabling Rx channel : %d\n", Cppi_getChannelNumber (rxChHnd));
    else 
        printf ("Rx channel : %d enabled \n", Cppi_getChannelNumber (rxChHnd));


    /* Fill in some data */
    for (i = 0; i < SIZE_DATA_BUFFER; i++) 
        txDataBuff[i] = i;

    printf ("\n--------------------Transmitting packets----------------------\n");
    /* Send out 8 packets */
    for (i = 0; i < NUM_PACKETS; i++)
    {
        /* Get a free descriptor */
        if ((monoDescPtr = (Cppi_Desc *) Qmss_queuePop (txFreeQueHnd)) == NULL)
        {
            printf ("Error: Getting descriptor from Queue Number\n", txFreeQueHnd);
            return;
        }

        /* Add data buffer */
        Cppi_setData (Cppi_DescType_MONOLITHIC, monoDescPtr, (uint8_t *) txDataBuff, SIZE_DATA_BUFFER);

        /* Set packet length */
        Cppi_setPacketLen (Cppi_DescType_MONOLITHIC, monoDescPtr, SIZE_DATA_BUFFER);

        printf ("Transmitting descriptor 0x%p\n", monoDescPtr);

        /* Push descriptor to Tx queue */
        Qmss_queuePushDescSize (txQueHnd, (uint32_t *) monoDescPtr, 16);
    }

    printf ("\n-------------------------Queue status-------------------------\n");
    result = Qmss_getQueueEntryCount (txQueHnd);
    printf ("Transmit Queue %d Entry Count : %d \n", txQueHnd, result);

    result = Qmss_getQueueEntryCount (txFreeQueHnd);
    printf ("Tx Free Queue %d Entry Count : %d \n", txFreeQueHnd, result);

    result = Qmss_getQueueEntryCount (freeQueHnd);
    printf ("Rx Free Queue %d Entry Count : %d \n", freeQueHnd, result);

    result = Qmss_getQueueEntryCount (rxQueHnd);
    printf ("Receive Queue %d Entry Count : %d \n", rxQueHnd, result);

    result = Qmss_getQueueEntryCount (txCmplQueHnd);
    printf ("Tx completion Queue %d Entry Count : %d \n", txCmplQueHnd, result);

    printf ("\n--------------------Receiving packets-------------------------\n");

#ifndef CONFIG_ACC
    while (Qmss_getQueueEntryCount (rxQueHnd) == 0);
    
    /* Get the rx packet */
    while ((rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (Qmss_queuePop (rxQueHnd))) != NULL)
    {
        length = Cppi_getPacketLen (Cppi_DescType_MONOLITHIC, rxPkt);

        printf ("Received descriptor 0x%p of length : %d\n", rxPkt, length);

        /* Get data buffer */
        Cppi_getData (Cppi_DescType_MONOLITHIC, rxPkt, &dataBuffPtr, &destLen);

        /* Compare */
        for (i = 0; i < destLen; i++)
        {
            if (txDataBuff[i] != dataBuffPtr[i])
                printf ("Error: In data buffer Tx: %02X - Rx: %02X \n", txDataBuff[i], dataBuffPtr[i]);
        }
        
        /* Recycle the descriptors */
        queInfo = Cppi_getReturnQueue (Cppi_DescType_MONOLITHIC, rxPkt);
        
        /* Push descriptor back to free queue */
        Qmss_queuePushDesc (Qmss_getQueueHandle(queInfo), (uint32_t *) rxPkt);
    }
#else
    {
        /* Burn some time for the accumulators to run. */
#if 1 
		temp = 0;
        for (i = 0; i < 50000; i++)
        {
          temp = i;
        }
        temp = 1;
#endif
    }

    /* Get the Rx packet */
    /* 8 packets were sent out. The list is NULL terminated. Start with the ping side */

    packetCount = 0;
    while (1)
    {
        if (usePing == TRUE)
        {
            /* Use ping side */
            base = 0;
            usePing = FALSE;
            printf ("Processing ping page\n");
        }
        else
        {
            /* Switch to pong side */
            base = ACC_ENTRY_SIZE;
            usePing = TRUE;
            printf ("Processing pong page\n");
        }

        /* Using Null terminate mode */
        index = base;
        while (hiPrioList[index] != NULL)
        {
            rxPkt = (Cppi_Desc *) QMSS_DESC_PTR (hiPrioList[index]);
            hiPrioList[index] = 0;

            /* Get packet length */
            length = Cppi_getPacketLen (Cppi_DescType_MONOLITHIC, rxPkt);

            printf ("Received descriptor 0x%p of length : %d\n", rxPkt, length);

            /* Get data buffer */
            Cppi_getData (Cppi_DescType_MONOLITHIC, rxPkt, &dataBuffPtr, &destLen);

            /* Compare */
            for (i = 0; i < destLen; i++)
            {
                if (txDataBuff[i] != dataBuffPtr[i])
                 printf ("Error: In data buffer Tx: %02X - Rx: %02X \n", txDataBuff[i], dataBuffPtr[i]);
            }
        
            /* Recycle the descriptors */
            queInfo = Cppi_getReturnQueue (Cppi_DescType_MONOLITHIC, rxPkt);
       
            /* Push descriptor back to free queue */
            Qmss_queuePushDesc (Qmss_getQueueHandle(queInfo), (uint32_t *) rxPkt);
            index++;
            packetCount++;
        }
        Qmss_ackInterrupt (cfg.channel, 1);
        Qmss_setEoiVector (Qmss_IntdInterruptType_HIGH, cfg.channel);

        if (packetCount == NUM_PACKETS)
            break;
    }
    Qmss_ackInterrupt (cfg.channel, 1);
    Qmss_setEoiVector (Qmss_IntdInterruptType_HIGH, cfg.channel);
#endif

    printf ("\n--------------------Deinitializing----------------------------\n");
#ifdef CONFIG_ACC
    /* Disable accumulator */
    if ((result = Qmss_disableAccumulator (Qmss_PdspId_PDSP1, cfg.channel)) != QMSS_ACC_SOK)
	{
        printf ("Error: Disabling high priority accumulator for channel : %d queue : %d error code: %d\n",
                        cfg.channel, cfg.queMgrIndex, result);
		return;
	}
    else
        printf ("high priority accumulator disabled for channel : %d queue : %d\n", 
                        cfg.channel, cfg.queMgrIndex);
#endif

    result = Qmss_getQueueEntryCount (txQueHnd);
    printf ("Transmit Queue %d Entry Count : %d \n", txQueHnd, result);

    result = Qmss_getQueueEntryCount (txFreeQueHnd);
    printf ("Tx Free Queue %d Entry Count : %d \n", txFreeQueHnd, result);

    result = Qmss_getQueueEntryCount (freeQueHnd);
    printf ("Rx Free Queue %d Entry Count : %d \n", freeQueHnd, result);

    result = Qmss_getQueueEntryCount (rxQueHnd);
    printf ("Receive Queue %d Entry Count : %d \n", rxQueHnd, result);

    result = Qmss_getQueueEntryCount (txCmplQueHnd);
    printf ("Tx completion Queue %d Entry Count : %d \n", txCmplQueHnd, result);

    /* Close Tx channel */
    if ((result = Cppi_channelClose (txChHnd)) != CPPI_SOK)
        printf ("Error: Closing Tx channel error code : %d\n", result);
    else
        printf ("Tx Channel closed successfully. Ref count : %d\n", result);

    /* Close Rx channel */
    if ((result = Cppi_channelClose (rxChHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx channel error code : %d\n", result);
    else
        printf ("Rx Channel closed successfully. Ref count : %d\n", result);

    /* Close Rx flow */
    if ((result = Cppi_closeRxFlow (rxFlowHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx flow error code : %d\n", result);
    else
        printf ("Rx flow closed successfully. Ref count : %d\n", result);
    
    /* Close the queues */
    if ((result = Qmss_queueClose (rxQueHnd)) != CPPI_SOK)
        printf ("Error: Closing Rx queue error code : %d\n", result);
    else
        printf ("Rx queue closed successfully. Ref count : %d\n", result);
    
    if ((result = Qmss_queueClose (txQueHnd)) != CPPI_SOK)
        printf ("Error: Closing tx queue error code : %d\n", result);
    else
        printf ("Tx queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (freeQueHnd)) != CPPI_SOK)
        printf ("Error: Closing free queue error code : %d\n", result);
    else
        printf ("Free queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (txCmplQueHnd)) != CPPI_SOK)
        printf ("Error: Closing transmit completion queue error code : %d\n", result);
    else
        printf ("Transmit completion queue closed successfully. Ref count : %d\n", result);

    if ((result = Qmss_queueClose (txFreeQueHnd)) != CPPI_SOK)
        printf ("Error: Closing transmit freequeue error code : %d\n", result);
    else
        printf ("Transmit free queue closed successfully. Ref count : %d\n", result);
}

/**
 *  @b Description
 *  @n  
 *      This is an QMSS infrastructure mode example. Works in polled mode
 *
 *      It performs the following
 *          - Initializes the Queue Manager low level driver.
 *          - Initializes the CPPI low level driver.
 *          - Opens the CPPI CPDMA in queue manager
 *          - Initializes descriptors and pushes to free queue
 *          - Programs accumulator
 *          - Pushes packets on Tx channel          
 *          - Processes the accumulated packets from Rx channel
 *          - Closes Rx and Tx channel
 *          - Closes all open queues
 *          - Closes CPDMA instance
 *          - Deinitializes CPPI LLD
 *  @retval
 *      Not Applicable.
 */
void example_main(Cppi_Handle cppiHnd)
{
    Qmss_Result             result;

    /* Setup memory region for host descriptors */
    memset ((void *) hostDesc, 0, SIZE_HOST_DESC * NUM_HOST_DESC);
    hostMemInfo.descBase = (uint32_t *) hostDesc;
    hostMemInfo.descSize = SIZE_HOST_DESC;
    hostMemInfo.descNum = NUM_HOST_DESC;
    hostMemInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    hostMemInfo.memRegion = Qmss_MemRegion_MEMORY_REGION_NOT_SPECIFIED;
    hostMemInfo.startIndex = 0;

    result = Qmss_insertMemoryRegion (&hostMemInfo);
    if (result < QMSS_SOK)
    {
        printf ("Error : Inserting memory region %d error code : %d\n", hostMemInfo.memRegion, result);
    }
    else
        printf ("Memory region %d inserted\n", result);


    /* Setup memory region for monolithic descriptors */
    memset ((void *) monolithicDesc, 0, SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC);
    monoMemInfo.descBase = (uint32_t *) monolithicDesc;
    monoMemInfo.descSize = SIZE_MONOLITHIC_DESC;
    monoMemInfo.descNum = NUM_MONOLITHIC_DESC;
    monoMemInfo.manageDescFlag = Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    monoMemInfo.memRegion = Qmss_MemRegion_MEMORY_REGION_NOT_SPECIFIED;
    monoMemInfo.startIndex = 0;

    result = Qmss_insertMemoryRegion (&monoMemInfo);
    if (result < QMSS_SOK)
    {
        printf ("Error : Inserting memory region %d error code : %d\n", monoMemInfo.memRegion, result);
    }
    else
        printf ("Memory region %d inserted\n", result);

    /* Transfer data using host descriptors */
    hostModeTransfer (cppiHnd);

    /* Transfer data using monolithic descriptors */
    monoModeTransfer (cppiHnd);

    /* Close CPPI CPDMA instance */
    if ((result = Cppi_close (cppiHnd)) != CPPI_SOK)
        printf ("Error : Closing CPPI CPDMA error code : %d\n", result);
    else 
        printf ("CPPI CPDMA closed successfully\n");

    /* Deinitialize CPPI LLD */
    if ((result = Cppi_exit ()) != CPPI_SOK)
        printf ("Error : Exiting CPPI error code : %d\n", result);
    else 
        printf ("CPPI exit successful\n");

    return;
}


