/**  
 *  (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include "fw_test.h"
#include "fw_mem_allocator.h"

#include <ti/drv/qmss/qmss_qm.h>
#include <ti/drv/qmss/device/qmss_device.c>
#include <ti/drv/cppi/device/cppi_device.c>

/* Test specific includes */
#include "qmss_test.h"

/* QMSS device specific configuration */
extern Qmss_GlobalConfigParams qmssGblCfgParams[];

/* CPPI device specific configuration */
extern Cppi_GlobalConfigParams  cppiGblCfgParams[];

/************************ GLOBAL VARIABLES ********************/
/* linking RAM */
uint64_t *linkingRAM0;

/* Descriptor pool [Size of descriptor * Number of descriptors] */
uint8_t *hostDesc;
uint8_t *monolithicDesc;
uint8_t *txDataBuff;
uint8_t *rxDataBuff;

/******************************************************************************
* Macro to convert to IP Register Virtual Address from a mapped base Virtual Address
* Input: virtBaseAddr: Virtual base address mapped using mmap for IP
*        phyBaseAddr: Physical base address for the IP
*        phyRegAddr:  Physical register address
******************************************************************************/
static inline void* FW_GET_REG_VADDR (void * virtBaseAddr, uint32_t phyBaseAddr, uint32_t phyRegAddr)
{
    return((void *)((uint8_t *)virtBaseAddr + (phyRegAddr - phyBaseAddr)));
}

/** ============================================================================
 *   @n@b initQmss
 *
 *   @b Description
 *   @n This API initializes the QMSS LLD on core 0 only.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t initQmss (void)
{
    int32_t                     result;
    Qmss_MemRegInfo             memCfg;
    Qmss_InitCfg                qmssInitConfig;
    Cppi_DescCfg                cppiDescCfg;
    uint32_t                    numAllocated;
    Qmss_GlobalConfigParams     fw_qmssGblCfgParams;

    /* Initialize QMSS */
    memset (&qmssInitConfig, 0, sizeof (Qmss_InitCfg));

    /* Set up QMSS configuration */

    /* Use internal linking RAM */
#ifdef INTERNAL_LINKING_RAM 
    qmssInitConfig.linkingRAM0Base  = 0;   
    qmssInitConfig.linkingRAM0Size  = 0;
    qmssInitConfig.linkingRAM1Base  = 0;
    qmssInitConfig.maxDescNum       = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
#else
    memset ((void *) &linkingRAM0, 0, ((NUM_HOST_DESC + NUM_MONOLITHIC_DESC) * sizeof(uint64_t)));
    qmssInitConfig.linkingRAM0Base = (uint32_t)linkingRAM0;
    qmssInitConfig.linkingRAM0Size = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
    qmssInitConfig.linkingRAM1Base = 0;
    qmssInitConfig.maxDescNum      = NUM_HOST_DESC + NUM_MONOLITHIC_DESC;
#endif    

    /* Bypass hardware initialization as it is done within Kernel */
    qmssInitConfig.qmssHwStatus     =   QMSS_HW_INIT_COMPLETE;

    fw_qmssGblCfgParams = qmssGblCfgParams[0];

    /* Convert address to Virtual address */ 
    fw_qmssGblCfgParams.qmConfigReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CONFIG_STARVATION_COUNTER_REGS);

    fw_qmssGblCfgParams.qmDescReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_DESCRIPTION_REGS);

    fw_qmssGblCfgParams.qmQueMgmtReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueMgmtProxyReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_PROXY_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueStatReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QUE_PEEK_REGS);
    
    fw_qmssGblCfgParams.qmQueIntdReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_INTD_REGS);

    fw_qmssGblCfgParams.qmPdspCmdReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM1_REGS);
  
    fw_qmssGblCfgParams.qmPdspCmdReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM2_REGS);
        
    fw_qmssGblCfgParams.qmPdspCtrlReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP1_REGS);

    fw_qmssGblCfgParams.qmPdspCtrlReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP2_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP1_RAM_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP2_RAM_REGS);

    fw_qmssGblCfgParams.qmStatusRAM = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_STATUS_RAM_REGS);

    fw_qmssGblCfgParams.qmLinkingRAMReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_LINKING_RAM_REGS);

    fw_qmssGblCfgParams.qmMcDMAReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_MCDMA_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER1_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER2_REGS);

    fw_qmssGblCfgParams.qmQueMgmtDataReg = (void *)((uint32_t)fw_qmssDataVaddr);

    fw_qmssGblCfgParams.qmQueMgmtProxyDataReg = 
        FW_GET_REG_VADDR(fw_qmssDataVaddr,QMSS_DATA_BASE_ADDR,QMSS_DATA_BASE_QUEUE_PROXY_ADDR);

    /* Initialize the Queue Manager */
    result = Qmss_init (&qmssInitConfig, &fw_qmssGblCfgParams);
    if (result != QMSS_SOK)
    {
        printf ("initQmss: Error initializing Queue Manager SubSystem, Error code : %d\n", result);
        return -1;
    }

    /* Start Queue manager on this core */
    result = Qmss_start ();
    if (result != QMSS_SOK)
    {
        printf ("initQmss: Error starting Queue Manager SubSystem, Error code : %d\n", result);
        return -1;
    }

    /* Queue Manager Initialization Done */
    return 0;
}

/** ============================================================================
 *   @n@b initCppi
 *
 *   @b Description
 *   @n This API initializes the CPPI LLD, opens the PASS CPDMA and opens up
 *      the Tx, Rx channels required for data transfers.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t initCppi (Cppi_Handle *cppiHnd)
{
    int32_t                     result, i;        
    Cppi_CpDmaInitCfg           cpdmaCfg;
    uint8_t                     isAllocated;        
    Cppi_TxChInitCfg            txChCfg;
    Cppi_RxChInitCfg            rxChInitCfg;
    Cppi_GlobalConfigParams     fw_cppiGblCfgParams[CPPI_MAX_CPDMA];

  for (i=0; i<CPPI_MAX_CPDMA; i++)
    fw_cppiGblCfgParams[i] = cppiGblCfgParams[i];


  /* Convert Physical address to Virtual address for LLD access */
  /* SRIO CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* PASS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxFlowRegs =
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* QMSS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);


    /* Initialize CPPI LLD */
    result = Cppi_init (fw_cppiGblCfgParams);
    if (result != CPPI_SOK)
    {
        printf ("initCppi: Error initializing CPPI LLD, Error code : %d\n", result);
        return -1;
    }

    /* Initialize QMSS CPDMA */
    memset (&cpdmaCfg, 0, sizeof (Cppi_CpDmaInitCfg));
    cpdmaCfg.dmaNum     = Cppi_CpDma_QMSS_CPDMA;

    if ((*cppiHnd = Cppi_open (&cpdmaCfg)) == NULL)
    {
        printf ("initCppi: Error initializing CPPI for QMSS CPDMA %d \n", cpdmaCfg.dmaNum);
        return -1;
    }

    return 0;
}    

/* The QM/CPDMA are setup */
int initQm (Cppi_Handle *cppiHnd)
{
  if (initQmss())  {
  	printf ("initQm: initQmss failed\n");
    return (-1);
  }

  if (initCppi (cppiHnd))  {
  	printf ("initQm: initCppi failed\n");
    return (-1);
  }
  
  return (0);
}

/* Initialize the test framework */
int setupTestFramework (Cppi_Handle *cppiHnd)
{

#ifndef INTERNAL_LINKING_RAM 
    /* Memory allocation for linking RAM */
    linkingRAM0 = (uint64_t *)fw_memAlloc(((NUM_HOST_DESC + NUM_MONOLITHIC_DESC) * sizeof(uint64_t)), CACHE_LINESZ);
    if (linkingRAM0 == NULL) {
		printf ("setupTestFramework: Failed to allocate memory for linkingRAM, exiting\n");
		return (-1);
    }
#endif

    /* Memory allocation for descriptors */
    hostDesc = (uint8_t *)fw_memAlloc((SIZE_HOST_DESC * NUM_HOST_DESC),
                                      CACHE_LINESZ);
    if (hostDesc == NULL) {
		printf ("setupTestFramework: Failed to allocate memory for Host \
                descriptors, exiting\n");
		return (-1);
    }

    /* Memory allocation for descriptors */
    monolithicDesc = (uint8_t *)fw_memAlloc((SIZE_MONOLITHIC_DESC * NUM_MONOLITHIC_DESC),
                                            CACHE_LINESZ);
    if (monolithicDesc == NULL) {
		printf ("setupTestFramework: Failed to allocate memory for Monolithic \
                descriptors, exiting\n");
		return (-1);
    }

    /* Memory allocation for TX data buffers */
    txDataBuff = (uint8_t *)fw_memAlloc((SIZE_DATA_BUFFER * NUM_TX_DATA_BUFFER),
                                      CACHE_LINESZ);
    if (txDataBuff == NULL) {
		printf ("setupTestFramework: Failed to allocate memory for TX data buffer, exiting\n");
		return (-1);
    }

    /* Memory allocation for RX data buffers */
    rxDataBuff = (uint8_t *)fw_memAlloc((SIZE_DATA_BUFFER * NUM_RX_DATA_BUFFER),
                                      CACHE_LINESZ);
    if (rxDataBuff == NULL) {
		printf ("setupTestFramework: Failed to allocate memory for RX data buffer, exiting\n");
		return (-1);
    }

	/* Setup QMMS and initialize CPPI */
	if (initQm(cppiHnd))  {
		printf ("setupTestFramework: initQm returned error, exiting\n");
		return (-1);
	}
	
	return (0);
}

