/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "pautest.h"

extern cregister volatile unsigned int DNUM;
extern cregister volatile unsigned int TSCL;
Void utilCycleDelay (Int count)
{
  UInt32 TSCLin = TSCL;

  if (count <= 0)
    return;

  while ((TSCL - TSCLin) < (UInt32)count);

}



UInt32 utilgAddr(UInt32 x)
{
	if ((x >= 0x800000) && (x < 0x900000))
	  x = (1 << 28) | (DNUM << 24) | x;
	  
  	return (x);
}

UInt16 utilOnesComplementAdd (UInt16 v1, UInt16 v2)
{
  UInt result;

  result = (UInt)v1 + (UInt)v2;
  result = (result >> 16) + (result & 0xffff);
  result = (result >> 16) + (result & 0xffff);

  return ((UInt16)result);

}

UInt16 utilOnesCompChkSum (UInt8 *p, UInt nwords)
{
  UInt16 chksum = 0;
  UInt16 v;
  UInt i;
  UInt j;

  for (i = j = 0; i < nwords; i++, j+=2)  {
    v = (p[j] << 8) | p[j+1];
    chksum = utilOnesComplementAdd (chksum, v);
  }

  return (chksum);

} /* onesCompChkSum */

/**************************************************************************************
 * FUNCTION PURPOSE: Compute ipv4 psudo checksum
 **************************************************************************************
 * DESCRIPTION: Compute ipv4 psudo checksum
 **************************************************************************************/
UInt16 utilGetIpv4PsudoChkSum (UInt8 *data, UInt16 payloadLen)
{
  UInt16 psudo_chksum;

  psudo_chksum = utilOnesCompChkSum (&data[12], 4);
  psudo_chksum = utilOnesComplementAdd(psudo_chksum, (UInt16) data[9]);
  psudo_chksum = utilOnesComplementAdd(psudo_chksum, payloadLen);
  
  return (psudo_chksum);

} /* utilGetIpv4PsudoChkSum */

/**************************************************************************************
 * FUNCTION PURPOSE: Compute ipv6 psudo checksum
 **************************************************************************************
 * DESCRIPTION: Compute ipv6 psudo checksum
 **************************************************************************************/
UInt16 utilGetIpv6PsudoChkSum (UInt8 *data, UInt16 payloadLen)
{
  UInt16 psudo_chksum;

  psudo_chksum = utilOnesCompChkSum (&data[8], 16);
  psudo_chksum = utilOnesComplementAdd(psudo_chksum, (UInt16) data[6]);
  psudo_chksum = utilOnesComplementAdd(psudo_chksum, payloadLen);
  
  return (psudo_chksum);

} /* utilGetIpv6PsudoChkSum */

/**************************************************************************************
 * FUNCTION PURPOSE: Compute ip psudo checksum
 **************************************************************************************
 * DESCRIPTION: Compute ip psudo checksum
 **************************************************************************************/
UInt16 utilGetIpPsudoChkSum (UInt8 *ip, UInt16 payloadLen)
{
  UInt16 psudo_chksum;
  
  if ((ip[0] & 0xF0) == 0x40)
  {
      psudo_chksum = utilGetIpv4PsudoChkSum(ip, payloadLen);
  }
  else
  {
      psudo_chksum = utilGetIpv6PsudoChkSum(ip, payloadLen);
  }
  
  return (psudo_chksum);

} /* utilGetIpPsudoChkSum */

/* Compute UDP checksums */
UInt16 utilCompUdpChksums (pktTestInfo_t *pktinfo, Bool insert)
{
	UInt8  *udp;
    UInt8  *ip;
	UInt16 	udpOffset;
    UInt16  ipOffset;
	UInt16  pseudo;
	UInt16  sum;
	Int     pktLen;
	
	udpOffset = TF_GET_UDP_OFFSET(pktinfo->info);
	ipOffset  = TF_GET_IP_OFFSET(pktinfo->info);
	
	udp   = &(pktinfo->pkt[udpOffset]);
	ip    = &(pktinfo->pkt[ipOffset]);
    
	pktLen = ((((Int)udp[4] << 8) | (Int)udp[5]) + 1) & ~1; 
    
    if ((ip[0] & 0xF0) == 0x40)
    {
        pseudo = utilGetIpv4PsudoChkSum(ip, pktLen);
    }
    else
    {
        pseudo = utilGetIpv6PsudoChkSum(ip, pktLen);
    }
	
	if (insert == TRUE)  {	
		/* replace the checksum with the pseudo header checksum */
		udp[6] = pseudo >> 8;
		udp[7] = pseudo & 0xff;
	}
	
	sum = utilOnesCompChkSum (udp, pktLen>>1);
    
    if (pktLen & 1)
    {
        sum = utilOnesComplementAdd(sum, udp[pktLen-1] << 8);
    }
    
    sum = ~sum;
	
    if(sum == 0)
       sum = 0xFFFF;
    
	if (insert == TRUE)  {
		udp[6] = sum >> 8;
		udp[7] = sum & 0x00ff;
	}
	
	return (sum);
}

/* Compute and Insert IP checksums */
UInt16 utilCompIpChksums (pktTestInfo_t *pktinfo,  Bool insert)
{
	UInt8  *ip;
	UInt16 	offset;
	UInt16  sum;
	
	offset = TF_GET_IP_OFFSET(pktinfo->info);
	
	ip   = &(pktinfo->pkt[offset]);
	
	/* reset the checksum field to zero */
	if (insert == TRUE)  {	
	    ip[10] = 0;
	    ip[11] = 0;
	}
	sum = ~utilOnesCompChkSum (ip, 10);
	
	if (insert == TRUE)  {	
	    ip[10] = sum >> 8;
	    ip[11] = sum & 0x00ff;
	}
	
	return (sum);
}

UInt16 utilUpdateIpChksums (UInt8  *ip)
{
	UInt16  sum;
	
	/* reset the checksum field to zero */
	ip[10] = 0;
	ip[11] = 0;
    
	sum = ~utilOnesCompChkSum (ip, 10);
	
	ip[10] = sum >> 8;
	ip[11] = sum & 0x00ff;
	
	return (sum);
}


/*******************************************************************************
 *  Function: Generate payload  
 *******************************************************************************
 *  DESCRIPTION:  Fill the data buffer with the specified payload data  
 *
 *  Note: It is up to the caller to allocate buffer
 ******************************************************************************/
void testGenPayload(pauPayloadType_e type, UInt8 initValue, UInt16 len, UInt8* buf)
{
    UInt8 data = initValue;
    int i;
    
    switch (type)
    {
        case PAU_PAYLOAD_CONST:
            memset(buf, data, len);
            break;
            
        case PAU_PAYLOAD_INC8: 
            for(i = 0; i < len; i++) buf[i] = data++;       
            break;
            
        case PAU_PAYLOAD_DEC8: 
            for(i = 0; i < len; i++) buf[i] = data--;       
            break;
            
         case PAU_PAYLOAD_RANDOM:
            for(i = 0; i < len; i++) buf[i] = rand() & 0xFF;       
            break;
            
         default:
            System_printf("testGenPayload: invalid paylaod type (%d)\n", type);
            System_flush();
            break;   
    }
}


	
		
		
		


