/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "../pautest.h"

/* Add/Delete IP and PDSP routing test
 * This test test the LLD Pa_addIp, Pa_delHandle and Pa_configCrcEngine functions, as well as the
 * PDSP firmware for routing IP packets. This test has the following sub-tests
 *   - Test the LLD for the ability to determine if a new entry invalidates a previous entry
 *   - Test the LLD to enter increasingly strict match requirements (dst, src, etype, spi, flow index, tos, gre protocol, sctp port)
 *   - Test the LLD when entering more entries then configured for
 *   - Test the LLD for global configuration 
 *   - Test the firmware for the ability to take a burst of Pa_AddIp commands
 *   - Test the firmware for the routing of matches and next step fail routes
 *   - Test the firmware for the ability to detect IP header errors
 *   - Test the firmware for the ability to handle nested IP headers
 *   - Test the firmware for the ability to break out of too many nested IP headers
 *   - Test the firmware for the ability to take a burst of data packets
 *   - Test the firmware for the ability to verify the SCTP CRC32-C
 * 	 - Test the firmware for correct header offset calculation
 *   - Test the firmware for PPPoE and IP header error processing
 *   - Test the LLD/firmware for the ability to delete entries
 */

 static char *tfName = "paTestL3Routing";
 
 #define T4_NUM_PACKET_ITERATIONS	2  		/* Number of times the packet stream is passed through */
 
 #define T4_EXPTPKT_FIFO_SIZE		20		/* Must hold the number of packets in transit in PA at one time */
 
 /* General purpose queue usage */
 #define Q_CMD_RECYCLE		  0		/* Command descriptors/buffers recycled here after sent to PA */
 #define Q_CMD_REPLY  		  1		/* Replies from PA routed here */
 #define Q_MATCH		  	  2		/* Packets from PA which match a lookup criteria */
 #define Q_NFAIL		      3		/* Packets from PA which matches a mac lookup, but failed an L3 lookup */
 #define Q_PARSE_ERR		  4		/* Packets which resulted in a parse error */
 #define Q_PPPoE_FAIL         5     /* PPPoE Parsing Error */
 #define Q_PPPoE_CTRL         6     /* PPPoE Control Packets */
 #define Q_IP_FAIL            7     /* IP Header Error */
 #define Q_IP_FRAG            8     /* IP Fragments */
 
/* Exception Route  packet index */
#define  T4_PPPoE_FAIL_PKT_INDEX        20
#define  T4_PPPoE_CTRL_PKT_INDEX        21
#define  T4_IP_FAIL_PKT_INDEX           22
#define  T4_IP_FRAG_PKT_INDEX           23
#define  T4_DROP_PKT_INDEX              0xffff

/* The number of PA L2 and L3 handles maintained by this test */
#define T4_NUM_LOCAL_L2_HANDLES   			2
#define T4_NUM_LOCAL_L3_OUTER_IP_HANDLES	64
#define T4_NUM_LOCAL_L3_INNER1_IP_HANDLES	62
#define T4_NUM_LOCAL_L3_OUTER_AND_INNER1_IP_HANDLE  (T4_NUM_LOCAL_L3_OUTER_IP_HANDLES + T4_NUM_LOCAL_L3_INNER1_IP_HANDLES)
#define T4_NUM_LOCAL_L3_INNER2_IP_HANDLES	 2		/* Tripple nested IP Headers */
#define T4_NUM_LOCAL_L3_HANDLES				(T4_NUM_LOCAL_L3_OUTER_IP_HANDLES+T4_NUM_LOCAL_L3_INNER1_IP_HANDLES+T4_NUM_LOCAL_L3_INNER2_IP_HANDLES+2)
#define T4_NUM_LOCAL_L4_HANDLES	  			8

typedef struct t4IpPaSetup_s  {
	Int        seqId;		/* Sequential enumeration to identify */
	Int		   handleIdx;	/* Local handle index. Specifies which local handle corresponds to this entry */
    Int        lutInst;     /* Specify which LUT1 (0-2) should be used */           
	Int		   lHandleIdx;  /* Linked handle (to previous L2 or L3 layer) */
	Int		   routeIdx;	/* Which match route index to use, 0 or 1 */
	paIpInfo_t ipInfo;		/* PA IP configuration structure */
	paReturn_t ret;			/* Expected return code from pa_addIp command */
	Bool	   acked;		/* Set to TRUE when the reply to the command is received */
	
} t4IpPaSetup_t;

#include "test4pkts.h"

/* The total number of buffers with linked descriptors */
#define TOTAL_BUFS   (TF_LINKED_BUF_Q1_NBUFS + TF_LINKED_BUF_Q2_NBUFS + TF_LINKED_BUF_Q3_NBUFS)
 
 /* Commands to the PA are verified through the value in swinfo0.
  * The 16 ms bits are used as verification, the 16 lbs are for local handle id */
#define T4_CMD_SWINFO0_ADD_MAC_ID  		0x11100000  /* Identifies add mac command */
#define T4_CMD_SWINFO0_DEL_MAC_ID  		0x11110000  /* Identifies del mac command */
#define T4_CMD_SWINFO0_ADD_IP_ID		0x22200000  /* Identifies the add IP command */
#define T4_CMD_SWINFO0_DEL_IP_ID		0x22210000  /* Identifies the del IP command */
#define T4_CMD_SWINFO0_ADD_PORT_ID		0x22220000  /* Identifies the add port command */
#define T4_CMD_SWINFO0_DEL_PORT_ID		0x22230000  /* Identifies the del port command */
#define T4_CMD_SWINFO0_STATS_REQ_ID		0x33330000	/* Identifies the req stats command */
#define T4_CMD_SWINFO0_CRC_CFG_ID		0x44400000  /* Identifies the CRC config command */
#define T4_CMD_SWINFO0_GLOBAL_CFG_ID   	0x44410000  /* Identifies the Global config command */ 
#define T4_CMD_SWINFO0_EROUTE_CFG_ID   	0x44420000  /* Identifies the EROUTE config command */
#define T4_CMD_SWINFO0_USR_STATS_CFG_ID 0x44430000  /* Identifies the user Stats config command */
#define T4_CMD_SWINFO0_PKT_ID			0x55550000  /* Identifies the packet as a data packet */

#define T4_CMD_SWINFO0_TYPE_MASK		0xffff0000  /* Mask for the command type */
#define T4_CMD_SWINFO0_ID_MASK			0x0000ffff  /* Mask for the local ID */


 /* L3 handles are managed. This structure is used to track the handle and
  * the activation state state of the handle */
enum  {
	T4_HANDLE_UNCONFIGURED = 0,
	T4_HANDLE_PENDING_ACK,
	T4_HANDLE_ACTIVE,
	T4_HANDLE_DISABLED
};

 typedef struct t4Handles_s  {
 
  	paHandleL2L3_t  paHandle;     /* The L3 handle returned by the PA LLD */

 	UInt			state;		  /* T4_HANDLE_UNCONFIGURED = handle not configured
 								   * T4_HANDLE_PENDING_ACK = handle configured and sent to pa
 								   * T4_HANDLE_ACTIVE = handle creation acknowledged by pa
 								   * T4_HANDLE_DISABLED = handle was created then released */
 	
 } t4Handles_t;
 
 typedef struct t4HandlesL4_s  {
 	
 	paHandleL4_t   paHandle;
 	
 	UInt 		   state;

 } t4HandlesL4_t;
 
 /* A grouping of run time created grouped together to make cleanup easier on
  * error exit */
 typedef struct t4TestEncap_s  {
 	tFramework_t  *tf;
 	paTest_t      *pat;
 	
 	/* The +1 is a place holder handle used to pass known invalid configuration into the PA LLD */
 	t4Handles_t    l2Handles[T4_NUM_LOCAL_L2_HANDLES+1];		/* MAC handles */
 	t4Handles_t    l3Handles[T4_NUM_LOCAL_L3_HANDLES+1];		/* IP handles  */
 	t4HandlesL4_t  l4Handles[T4_NUM_LOCAL_L4_HANDLES+1];		/* UDP/TCP handles */
 	
 } t4TestEncap_t;
 
static paSysStats_t paTestL4ExpectedStats;    /* Expected stats results */
 
static paCmdReply_t cmdReply = {  pa_DEST_HOST,				/* Dest */
 							      0,						/* Reply ID (returned in swinfo0) */
 							   	  0,						/* Queue */
 							      0 };						/* Flow ID */
 							      
static paRouteInfo_t matchRoute[3] = {  {  	pa_DEST_HOST,		        /* Dest */
 								       		0,					        /* Flow ID */
 								       		0,					        /* queue */
 								       	   -1,					        /* Multi route */
 								       		0,					        /* sw Info 0 */
                                            0,                          /* sw Info 1 */       
                                            0,                          /* customType : not used  */         
                                            0,                          /* customIndex: not used  */     
                                            0,                          /* pkyType: for SRIO only */    
                                            NULL},                      /* No commands            */
 								       		
 								    {  		pa_DEST_CONTINUE_PARSE_LUT2,/* Dest */
 								       		0,							/* Flow ID */
 								       		0,							/* queue */
 								           -1,							/* Multi route */
 								        	0,							/* sw Info 0 */
                                            0,                          /* sw Info 1 */       
                                            0,                          /* customType : pa_CUSTOM_TYPE_NONE  */         
                                            0,                          /* customIndex: not used  */     
                                            0,                          /* pkyType: for SRIO only */    
                                            NULL},                      /* No commands            */
 								       		
 								       		
 								  	{		pa_DEST_CONTINUE_PARSE_LUT1,/* Dest */
 								  			0,							/* Flow ID */
 								  			0,							/* queue */
 								  		   -1,							/* Multi route */
 								  		    0,							/* sw Info 0 */
                                            0,                          /* sw Info 1 */       
                                            0,                          /* customType : pa_CUSTOM_TYPE_NONE  */         
                                            0,                          /* customIndex: not used  */     
                                            0,                          /* pkyType: for SRIO only */    
                                            NULL}                       /* No commands            */
                                    };        
 							      
static paRouteInfo_t   nfailRoute = {  pa_DEST_DISCARD,		/* Dest */
 										0,					/* Flow ID */
 										0,					/* queue */
 										-1,					/* Multi route */
 										0,					/* sw Info 0 */
                                        0,                  /* sw Info 1 */       
                                        0,                  /* customType : not used  */         
                                        0,                  /* customIndex: not used  */     
                                        0,                  /* pkyType: for SRIO only */    
                                        NULL};              /* No commands            */
 									 							 
                                                                 
static paRouteInfo_t   nfailRoute1 = {  pa_DEST_HOST,		/* Dest */
 										0,					/* Flow ID */
 										0,					/* queue */
 										-1,					/* Multi route */
 										0,					/* sw Info 0 */
                                        0,                  /* sw Info 1 */       
                                        0,                  /* customType : not used  */         
                                        0,                  /* customIndex: not used  */     
                                        0,                  /* pkyType: for SRIO only */    
                                        NULL};              /* No commands            */
                                                                 
                                                                 
/* CRC Configuration of CRC-32C for SCTP */
static paCrcConfig_t   t4CrcCfg = {
                                    pa_CRC_CONFIG_INVERSE_RESULT,       /* ctrlBitfield */
                                    pa_CRC_SIZE_32,
                                    0x1EDC6F41,                         /* polynomial */
                                    0xFFFFFFFF                          /* initValue */
                                  }; 
                                  
/* Global configurations */                                  
#define T4_NUM_64B_USR_STATS            64
#define T4_USR_STATS_L2_PADDING_ERR     0xc0
#define T4_USR_STATS_L2_TX_PADDING      0xc1

                                        
static paUsrStatsConfig_t   t4UsrStatsCfg =
       {
            pa_USR_STATS_MAX_COUNTERS - T4_NUM_64B_USR_STATS,   /* Number of user stats (448)*/
            T4_NUM_64B_USR_STATS                                /* Number of 64-bit user stats */
       };   
                                  
                                  
static paPacketControlConfig_t  t4PktCtrlCfg = 
        {
            pa_PKT_CTRL_HDR_VERIFY_PPPoE   |                              /* ctrlBitMap */
            pa_PKT_CTRL_HDR_VERIFY_IP      |
            pa_PKT_CTRL_IP_FRAGS_TO_EROUTE |
            pa_PKT_CTRL_MAC_PADDING_CHK,
            T4_USR_STATS_L2_PADDING_ERR,                                  /* rxPaddingErrStatsIndex */
            T4_USR_STATS_L2_TX_PADDING                                    /* txPaddingStatsIndex */
        };    
                                                                                              
static  paSysConfig_t  t4GlobalCfg = 
        {
            NULL,                   /* pProtoLimit */
            NULL,                   /* pOutIpReassmConfig */
            NULL,                   /* pInIpReassmConfig */
            NULL,                   /* pCmdSetConfig */
            &t4UsrStatsCfg,         /* pUsrStatsConfig */
            NULL,                   /* pQueueDivertConfig */
            &t4PktCtrlCfg           /* pPktControl */
        };
        
#define T4_NUM_EXCEPTION_ROUTES    4
 static int t4ErouteTypes[] = {
    pa_EROUTE_PPPoE_FAIL,
    pa_EROUTE_PPPoE_CTRL,
    pa_EROUTE_IP_FRAG,
    pa_EROUTE_IP_FAIL
 };
 
 static paRouteInfo_t t4Eroutes[] = {
 
    /* PPPoE Fail */
 	{  pa_DEST_HOST,		/* Dest */
 	   0,					/* Flow ID */
 	   Q_PPPoE_FAIL + TF_FIRST_GEN_QUEUE, /* queue */
 	   -1,					/* Multi route */
 	   T4_PPPoE_FAIL_PKT_INDEX + T4_CMD_SWINFO0_PKT_ID, /* sw Info 0 */
       0,                   /* sw Info 1 */       
       0,                   /* customType : not used */         
       0,                   /* customIndex: not used */     
       0,                   /* pkyType: for SRIO only */    
       NULL                 /* No commands */
    },
    
    /* PPPoE Control Packet */
 	{  pa_DEST_HOST,		/* Dest */
 	   0,					/* Flow ID */
 	   Q_PPPoE_CTRL + TF_FIRST_GEN_QUEUE, /* queue */
 	   -1,					/* Multi route */
 	   T4_PPPoE_CTRL_PKT_INDEX + T4_CMD_SWINFO0_PKT_ID, /* sw Info 0 */
       0,                   /* sw Info 1 */       
       0,                   /* customType : not used */         
       0,                   /* customIndex: not used */     
       0,                   /* pkyType: for SRIO only */    
       NULL                 /* No commands */
    },
    
    /* IP Frag */
 	{  pa_DEST_HOST,		/* Dest */
 	   0,					/* Flow ID */
 	   Q_IP_FRAG + TF_FIRST_GEN_QUEUE, /* queue */
 	   -1,					/* Multi route */
 	   T4_IP_FRAG_PKT_INDEX + T4_CMD_SWINFO0_PKT_ID, /* sw Info 0 */
       0,                   /* sw Info 1 */       
       0,                   /* customType : not used */         
       0,                   /* customIndex: not used */     
       0,                   /* pkyType: for SRIO only */    
       NULL                 /* No commands */
    },
    
    /* IP Fail */
 	{  pa_DEST_HOST,		/* Dest */
 	   0,					/* Flow ID */
 	   Q_IP_FAIL + TF_FIRST_GEN_QUEUE, /* queue */
 	   -1,					/* Multi route */
 	   T4_IP_FAIL_PKT_INDEX + T4_CMD_SWINFO0_PKT_ID, /* sw Info 0 */
       0,                   /* sw Info 1 */       
       0,                   /* customType : not used */         
       0,                   /* customIndex: not used */     
       0,                   /* pkyType: for SRIO only */    
       NULL                 /* No commands */
    }
}; 

/*
 * User-definded Statitics Map
 *
 * Group 1: counter 0xc0,  packet Counter (Rx Padding Error) no link
 *          counter 0xc1,  packet Counter (Tx Padding) no link
 */
 
#pragma DATA_SECTION (t4UsrStatsGroup1, ".testPkts")

static paUsrStatsCounterEntryConfig_t t4UsrStatsGroup1[ ] =
{
    {T4_USR_STATS_L2_PADDING_ERR, pa_USR_STATS_LNK_END, pa_USR_STATS_TYPE_PACKET},
    {T4_USR_STATS_L2_TX_PADDING,  pa_USR_STATS_LNK_END, pa_USR_STATS_TYPE_PACKET}
}; 

#pragma DATA_SECTION(t4UsrStatsSetup, ".testPkts")
static pauUsrStatsSetup_t  t4UsrStatsSetup[] = {
    /* entry 0 */
    {
        sizeof(t4UsrStatsGroup1)/sizeof(paUsrStatsCounterEntryConfig_t),    /* number of entries */
        t4UsrStatsGroup1,                                                   /* counter Info table */
        pa_OK                                                               /* Expected return value */
    }
};

static paUsrStats_t  paTestExpectedUsrStats;
 
 void t4Cleanup (t4TestEncap_t *tencap, paTestStatus_t testStatus)
{ 		
 	Int 	       i;
 	Int  	       cmdDest;
 	UInt16	       cmdSize;
 	paReturn_t     paret;
 	Cppi_HostDesc *hd;
 	paTestStatus_t newStatus;
 	
 	/* Delete active L4 handles */
 	for (i = 0; i < T4_NUM_LOCAL_L4_HANDLES; i++)  {
		
		cmdReply.replyId = T4_CMD_SWINFO0_DEL_PORT_ID + i;  /* T2_CMD_SWINFO0_ADD_ID identifies command, 16 LS bits identify the local handle number */
 		cmdReply.queue   = tencap->tf->QGen[Q_CMD_REPLY];
		
		if ((tencap->l4Handles[i].state == T4_HANDLE_PENDING_ACK) || (tencap->l4Handles[i].state == T4_HANDLE_ACTIVE))  {
			hd = testCommonDelL4Handles (tencap->tf, tencap->l4Handles[i].paHandle, tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf1,
					&cmdReply, &cmdDest, &cmdSize, &paret);
					
			if (paret != pa_OK)  {
				System_printf ("%s (%s:%d): PA LLD returned error code %d\n", tfName, __FILE__, __LINE__, paret);
				continue;
			}
		
			Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
		
			/* Wait for the response */
			if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, cmdReply.replyId, __LINE__)) {
				System_printf ("%s (%s:%d): testCommonWaitCmdReply failed\n", tfName, __FILE__, __LINE__);
				testStatus = PA_TEST_FAILED;
 			}
 			
 			/* Recycle the command packet as well */
 			hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 			if (hd == NULL)  {
 				System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 				continue;
 			}
 			testCommonRecycleLBDesc (tencap->tf, hd);
 		}
 	}
 	
 	/* Delete active L3 Handles */
 	for (i = 0; i < T4_NUM_LOCAL_L3_HANDLES; i++)  {
 		cmdReply.replyId = T4_CMD_SWINFO0_DEL_IP_ID + i;  /* T2_CMD_SWINFO0_ADD_ID identifies command, 16 LS bits identify the local handle number */
 		cmdReply.queue   = tencap->tf->QGen[Q_CMD_REPLY];
		
		if ((tencap->l3Handles[i].state == T4_HANDLE_PENDING_ACK) || (tencap->l3Handles[i].state == T4_HANDLE_ACTIVE))  {
			hd = testCommonDelHandle (tencap->tf, &tencap->l3Handles[i].paHandle, tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf1,
					&cmdReply, &cmdDest, &cmdSize, &paret);
					
			if (paret != pa_OK)  {
				System_printf ("%s (%s:%d): PA LLD returned error code %d\n", tfName, __FILE__, __LINE__, paret);
				continue;
			}
		
			Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
			paTestL4ExpectedStats.classify1.nPackets += 1;
		
			/* Wait for the response */
			if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, cmdReply.replyId, __LINE__)) {
				System_printf ("%s (%s:%d): testCommonWaitCmdReply failed\n", tfName, __FILE__, __LINE__);
				testStatus = PA_TEST_FAILED;
 			}
 			/* Recycle the command packet as well */
 			hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 			if (hd == NULL)  {
 				System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 				continue;
 			}
 			testCommonRecycleLBDesc (tencap->tf, hd);
 			
 			
 		}
 	}
 	
 	
 	/* Delete active L2 Handles */
 	for (i = 0; i < T4_NUM_LOCAL_L2_HANDLES; i++)  {
 		cmdReply.replyId = T4_CMD_SWINFO0_DEL_MAC_ID + i;  /* T2_CMD_SWINFO0_ADD_ID identifies command, 16 LS bits identify the local handle number */
 		cmdReply.queue   = tencap->tf->QGen[Q_CMD_REPLY];
		
		if ((tencap->l2Handles[i].state == T4_HANDLE_PENDING_ACK) || (tencap->l2Handles[i].state == T4_HANDLE_ACTIVE))  {
			hd = testCommonDelHandle (tencap->tf, &tencap->l2Handles[i].paHandle, tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf1,
					&cmdReply, &cmdDest, &cmdSize, &paret);
					
			if (paret != pa_OK)  {
				System_printf ("%s (%s:%d): PA LLD returned error code %d\n", tfName, __FILE__, __LINE__, paret);
				continue;
			}
		
			Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
			paTestL4ExpectedStats.classify1.nPackets += 1;
		
			/* Wait for the response */
			if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, cmdReply.replyId, __LINE__)) {
				System_printf ("%s (%s:%d): testCommonWaitCmdReply failed\n", tfName, __FILE__, __LINE__);
				testStatus = PA_TEST_FAILED;
 			}
 			
 			/* Recycle the command packet as well */
 			hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 			if (hd == NULL)  {
 				System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 				continue;
 			}
 			testCommonRecycleLBDesc (tencap->tf, hd);
 			
 			
 		}
 	}
 	
 	/* Pop any descriptors off of the return queues and restore them to the linked buffer queues or the free Q */
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_CMD_RECYCLE]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 		testCommonRecycleLBDesc (tencap->tf, hd);
 	}
 	
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_CMD_REPLY]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_REPLY])) & ~15);
 		testCommonRecycleLBDesc (tencap->tf, hd);
 	}
 	
 	 while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_MATCH]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_MATCH])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
 	
 	 while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_NFAIL]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_NFAIL])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
 	
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_PARSE_ERR]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_PARSE_ERR])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
    
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_PPPoE_FAIL]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_PPPoE_FAIL])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
    
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_PPPoE_CTRL]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_PPPoE_CTRL])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
    
 	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_IP_FAIL]) > 0)  {
 		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_IP_FAIL])) & ~15);
 		Qmss_queuePush (tencap->tf->QfreeDesc, (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
 	}
    
 	newStatus = testCommonCheckStats (tencap->tf, tencap->pat, tfName, &paTestL4ExpectedStats, tencap->tf->QLinkedBuf1, 
	                       tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QGen[Q_CMD_REPLY], TRUE);
	if (newStatus == PA_TEST_FAILED)
		testStatus = PA_TEST_FAILED;
 	 					
	/* Return result */                
    tencap->pat->testStatus = testStatus;
    
    /* Return */
    Task_exit();
}            

	
 
 /* Check for pa lld return errors. Exit the test on any error condition */
 static void t4HandleError (t4TestEncap_t *tencap, paReturn_t paret, Cppi_HostDesc *hd, Int line)
 {
 	
 	if (paret == pa_OK)
 		return;
 		
 	System_printf ("%s (%s:%d): PA LLD returned error code %d\n", tfName, __FILE__, line, paret);
 	
 	if ((hd != NULL) && testCommonRecycleLBDesc (tencap->tf, hd))
			System_printf ("%s: (%s:%d): testCommonRecycleLBDesc failed to return a buffer/descriptor\n", tfName, __FILE__, __LINE__);
	
	t4Cleanup (tencap, PA_TEST_FAILED);  /* No return */
	
 }
 
 
 /* Look for replies to add Ip commands and verify the results */
void t4L3CmdRep (t4TestEncap_t *tencap, t4IpPaSetup_t *ipSetup)
{
	Cppi_HostDesc  *hd;
	UInt32         *swinfo;
	paReturn_t      paret;
	UInt32			swinfoType;
	UInt32			swinfoIdx;
	paEntryHandle_t	reth;
    Int			    htype;
    Int             cmdDest;
	
	while (Qmss_getQueueEntryCount(tencap->tf->QGen[Q_CMD_REPLY]) > 0)  {
		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_REPLY])) & ~15);
		
		    Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (UInt8 **)&swinfo);
		    
		    swinfoType = swinfo[0] & T4_CMD_SWINFO0_TYPE_MASK;
		    swinfoIdx  = swinfo[0] & T4_CMD_SWINFO0_ID_MASK;
            
            if (swinfoType != T4_CMD_SWINFO0_ADD_IP_ID)  {
                System_printf ("%s (%s:%d): found packet in command reply queue without add IP swinfo type (found 0x%08x)\n", tfName, __FILE__, __LINE__, swinfo[0]);
                testCommonRecycleLBDesc (tencap->tf, hd);
                continue;
            }
            
            if (swinfoIdx >= sizeof(t4OuterIpInfo)/sizeof(t4IpPaSetup_t))  {
            	System_printf ("%s (%s:%d): found packet in command reply queue with invalid index (found 0x%08x)\n", tfName, __FILE__, __LINE__, swinfoIdx);
            	testCommonRecycleLBDesc (tencap->tf, hd);
                continue;
            }
            
            ipSetup[swinfoIdx].acked = TRUE;       
            paret = Pa_forwardResult (tencap->tf->passHandle, (Void *)hd->buffPtr, &reth, &htype, &cmdDest);
            
            if (paret != ipSetup[swinfoIdx].ret)  {
                System_printf ("%s (%s:%d): Pa_forwardResult returned %d, expected %d\n", tfName, __FILE__, __LINE__, paret, ipSetup[swinfoIdx].ret);
                testCommonRecycleLBDesc (tencap->tf, hd); 
                continue;
            }
            
            testCommonRecycleLBDesc (tencap->tf, hd); 
            
            tencap->l3Handles[ipSetup[swinfoIdx].handleIdx].state = T4_HANDLE_ACTIVE;
            
	}
    
    /* Recycle the command packet */
	while (Qmss_getQueueEntryCount(tencap->tf->QGen[Q_CMD_RECYCLE]) > 0)  {
    		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 			if (hd == NULL)  {
 				System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 				continue;
 			}
 			testCommonRecycleLBDesc (tencap->tf, hd);
    }
}

static paTestStatus_t t4ExceptionRoutes (t4TestEncap_t *tencap)
{
	Cppi_HostDesc  *hd;
	paReturn_t      paret;
	Int  			cmdDest;
	UInt16			cmdSize;
    
    /* Issue the exception route command */
    cmdReply.replyId  = T4_CMD_SWINFO0_EROUTE_CFG_ID;
	cmdReply.queue    = tencap->tf->QGen[Q_CMD_REPLY];
 	hd = testCommonConfigExceptionRoute (tencap->tf, T4_NUM_EXCEPTION_ROUTES, t4ErouteTypes, t4Eroutes,  
 	                                     tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf3, 
 	                                     &cmdReply, &cmdDest, &cmdSize, &paret);
                                         
 	t4HandleError (tencap, paret, hd, __LINE__); /* Will not return on error */   
                                         
    /* Send command */
 	Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
    
	if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_EROUTE_CFG_ID, __LINE__)) {
		System_printf ("%s (%s:%d): testCommonConfigExceptionRoute failed\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
 	}
    
 	/* Recycle the command packet as well */
 	hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 	if (hd == NULL)  {
 		System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 		return(PA_TEST_FAILED);
 	}
 	testCommonRecycleLBDesc (tencap->tf, hd);

	return (PA_TEST_PASSED);
}

static paTestStatus_t t4GlobalConfiguration (t4TestEncap_t *tencap)
{
	Cppi_HostDesc  *hd;
	paReturn_t      paret;
	Int  			cmdDest;
	UInt16			cmdSize;
    paCtrlInfo_t    ctrlInfo;
    
    /* Issue the command set command */
    ctrlInfo.code = pa_CONTROL_SYS_CONFIG;
    ctrlInfo.params.sysCfg = t4GlobalCfg;
    cmdReply.replyId  = T4_CMD_SWINFO0_GLOBAL_CFG_ID;
	cmdReply.queue    = tencap->tf->QGen[Q_CMD_REPLY];
 	hd = testCommonGlobalConfig (tencap->tf, &ctrlInfo,  
 	                             tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf3, 
 	                             &cmdReply, &cmdDest, &cmdSize, &paret);
                                 
 	t4HandleError (tencap, paret, hd, __LINE__); /* Will not return on error */   
                                         
    if (hd == NULL)  {
   			
   	  System_printf ("%s: (%s:%d): Failure in GlobalConfig command\n", tfName, __FILE__, __LINE__);
   	  return (PA_TEST_FAILED);
    }								 
    
    /* Send command */
 	Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
    
	if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_GLOBAL_CFG_ID, __LINE__)) {
		System_printf ("%s (%s:%d): testCommonGlobalConfig failed\n", tfName, __FILE__, __LINE__);
		return (PA_TEST_FAILED);
 	}
    
 	/* Recycle the command packet as well */
 	hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 	if (hd == NULL)  {
 		System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 		return(PA_TEST_FAILED);
 	}
 	testCommonRecycleLBDesc (tencap->tf, hd);
    
	return (PA_TEST_PASSED);
}

/* Simple User-Statistics routine: It does not process link and counter type */
static void t4UpdateUsrStats(paUsrStats_t* pStats, uint16_t cntIndex)
{
    if (cntIndex < T4_NUM_64B_USR_STATS)
    {
        pStats->count64[cntIndex]++ ;
    } 
    else
    {
        pStats->count32[cntIndex - T4_NUM_64B_USR_STATS]++;
    }
}


static paTestStatus_t t4ConfigUsrStats (t4TestEncap_t *tencap, int numEntries, pauUsrStatsSetup_t *usrStatsSetup, int clear)
{
	Int 			i;
	Cppi_HostDesc  *hd;
	paReturn_t      paret;
	Int  			cmdDest;
	UInt16			cmdSize;
    paUsrStatsCounterConfig_t   cntCfg;
    paUsrStatsConfigInfo_t      statsCfgInfo;
	
    statsCfgInfo.pCntCfg = &cntCfg;
    if (!clear)
    {
	    for (i = 0; i < numEntries; i++)  {
		    cmdReply.replyId = T4_CMD_SWINFO0_USR_STATS_CFG_ID + i;
		    cmdReply.queue   = tencap->tf->QGen[Q_CMD_REPLY];
        
            memset(&cntCfg, 0, sizeof(cntCfg));
            cntCfg.numCnt = usrStatsSetup[i].nStats;
            cntCfg.cntInfo = usrStatsSetup[i].cntEntryTbl;
		
		    hd = testCommonConfigUsrStats (tencap->tf, &statsCfgInfo, tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf3, 
 	        	                           &cmdReply, &cmdDest, &cmdSize, &paret);
                                           
            if (paret != pa_OK)
            {
                if (paret != usrStatsSetup[i].paret)
                {
			        System_printf ("%s: (%s:%d): configUsrStats command (%d): unexpected err = %d (expected err = %d)\n", tfName, 
                                __FILE__, __LINE__, i, paret, usrStatsSetup[i].paret);
		            return (PA_TEST_FAILED);
			    
                }
                continue;
            }
            else if (hd == NULL)  {
					
			    System_printf ("%s: (%s:%d): Failure in common configUsrStats command, entry number %d\n", tfName, __FILE__, __LINE__, i);
		        return (PA_TEST_FAILED);
		    }	        				 
                                           
		    Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
                                           
	        if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_USR_STATS_CFG_ID + i, __LINE__)) {
		        System_printf ("%s (%s:%d): testCommonConfigUsrStats\n", tfName, __FILE__, __LINE__);
		        return (PA_TEST_FAILED);
 	        }
            
 	        /* Recycle the command packet as well */
 	        hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 	        if (hd == NULL)  {
 		        System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 		        return(PA_TEST_FAILED);
 	        }
 	        testCommonRecycleLBDesc (tencap->tf, hd);
	    }
    }
    else
    {
        numEntries = 1;
	    cmdReply.replyId = T4_CMD_SWINFO0_USR_STATS_CFG_ID;
	    cmdReply.queue    = tencap->tf->QGen[Q_CMD_REPLY];
    
        memset(&cntCfg, 0, sizeof(cntCfg));
        cntCfg.ctrlBitfield = pa_USR_STATS_CONFIG_RESET;
        cntCfg.numCnt = 0;
        cntCfg.cntInfo = NULL;
	
	    hd = testCommonConfigUsrStats (tencap->tf, &statsCfgInfo, tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf3, 
 		                               &cmdReply, &cmdDest, &cmdSize, &paret);
						 
        if (paret != pa_OK)
        {
	        System_printf ("%s: (%s:%d): configUsrStats command (%d): unexpected err = %d\n", tfName, 
                        __FILE__, __LINE__, 0, paret);
	        return (PA_TEST_FAILED);
        }
        else if (hd == NULL)  {
			
	        System_printf ("%s: (%s:%d): Failure in common configUsrStats (clear) command\n", tfName, __FILE__, __LINE__);
	        return (PA_TEST_FAILED);
	    }	        				 
						 
	    Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
        
	    if (testCommonWaitCmdReply (tencap->tf, tencap->pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_USR_STATS_CFG_ID, __LINE__)) {
		    System_printf ("%s (%s:%d): testCommonConfigUsrStats\n", tfName, __FILE__, __LINE__);
		    return (PA_TEST_FAILED);
 	    }
        
 	    /* Recycle the command packet as well */
 	    hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_CMD_RECYCLE])) & ~15);
 	    if (hd == NULL)  {
 		    System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 		    return(PA_TEST_FAILED);
 	    }
 	    testCommonRecycleLBDesc (tencap->tf, hd);
    }
	
	return (PA_TEST_PASSED);
}


/*
 * Utility function to derive the LUT1 index from the local index
 * It is used to verify the Pa_addIP() with application specific index
 * The index should be consistent with the system allocated LUT1 index
 */
static Int t4GetLUT1Index(Int handleIndex)
{
    if (handleIndex >= T4_NUM_LOCAL_L3_HANDLES)
    {
        return (pa_LUT1_INDEX_NOT_SPECIFIED);
    }
    
    handleIndex = handleIndex % 64;
    
    /*
     * Special case for repeated entries
     *
     */
    if((handleIndex == 7) || (handleIndex == 18))
        return (pa_LUT1_INDEX_NOT_SPECIFIED);  
    
    if (handleIndex & 1)
    {
        return (pa_LUT1_INDEX_NOT_SPECIFIED);
    }
    else
    {
        return (63 - handleIndex);
    }
} 


static paTestStatus_t t4OpenIp (t4TestEncap_t  *tencap, t4IpPaSetup_t *ipSetup, int n, t4Handles_t *linkedHandles, char *id)
{
	Int 			i, j, m;
	Cppi_HostDesc  *hd;
	paReturn_t      paret;
	Int  			cmdDest;
	UInt16			cmdSize;
    paRouteInfo_t  *pNfailRoute;
	
	for (i = 0; i < n; i++)  {
		cmdReply.replyId = T4_CMD_SWINFO0_ADD_IP_ID + ipSetup[i].seqId;
		cmdReply.queue    = tencap->tf->QGen[Q_CMD_REPLY];
		
        pNfailRoute = (ipSetup[i].routeIdx == 1)?&nfailRoute1:&nfailRoute;
		hd = testCommonAddIp2 (tencap->tf, ipSetup[i].lutInst, t4GetLUT1Index(ipSetup[i].handleIdx), &ipSetup[i].ipInfo, &matchRoute[ipSetup[i].routeIdx], pNfailRoute,
							   &tencap->l3Handles[ipSetup[i].handleIdx].paHandle, 
							   linkedHandles[ipSetup[i].lHandleIdx].paHandle,
							   tencap->tf->QGen[Q_CMD_RECYCLE], tencap->tf->QLinkedBuf1,
							   &cmdReply, &cmdDest, &cmdSize, &paret);
								 
		if (hd == NULL)  {
			
			/* It's not a failure if the return code from pa was expected */
			if (paret == ipSetup[i].ret)  {
				ipSetup[i].acked = TRUE;
				continue;
			}
			
			System_printf ("%s: (%s:%d): Failure in common addIp command, %s entry number %d\n", tfName, __FILE__, __LINE__, id, ipSetup[i].seqId);
			t4HandleError (tencap, paret, hd, __LINE__);
		}	
        else if (paret == pa_DUP_ENTRY) {
            /* The ack will be handle for the first entry */
            ipSetup[i].acked = TRUE;
        }							 
								 
		Qmss_queuePush (tencap->tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd, cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 		tencap->l3Handles[ipSetup[i].handleIdx].state = T4_HANDLE_PENDING_ACK;
 		paTestL4ExpectedStats.classify1.nPackets += 1;
 		
 		t4L3CmdRep (tencap, ipSetup);
	}
	
	/* All the packets should have been acked */
	for (i = 0; i < 100; i++)  {
		t4L3CmdRep (tencap, ipSetup);
		
		for (j = m = 0; j < n; j++)  {
			if (ipSetup[j].acked == TRUE)
				m += 1;
		}
		
		if (m == n)
			break;
		else
			utilCycleDelay (1000);
	}
			
	if (i == 100)  {
		System_printf ("%s: (%s:%d): Command %d (out of %d) addIp commands were acked (%s)\n", tfName, __FILE__, __LINE__, m, n, id);
		return (PA_TEST_FAILED);
	}
	
	return (PA_TEST_PASSED);
	
}
		
static Cppi_HostDesc  *t4FormPktDataDescr (t4TestEncap_t *tencap, Int idx)
{	
	Cppi_HostDesc  *hd;
	Qmss_Queue      q;
	
    hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QfreeDesc)) & ~15);
    if (hd == NULL)  {
        System_printf ("%s (%s:%d): Failed to pop a descriptor off the free descriptor Q (%d)\n", tfName, __FILE__, __LINE__, tencap->tf->QfreeDesc);
        return (NULL);
    }
        
    /* Recycle the free descriptors right back to the free descriptor queue */
    q.qMgr = 0;
    q.qNum = tencap->tf->QfreeDesc;

    /* Setup the return for the descriptor */
    Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
    
  	/* Make sure there is no control info.  */
  	Cppi_setPSLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, 0);

    /* Attach the data and set the length */
    Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (UInt8 *)utilgAddr((UInt32)t4PktInfo[idx].pkt), t4PktInfo[idx].pktLen);
    Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, t4PktInfo[idx].pktLen);
        
    return (hd);
}
	

/* Look for packets in the receive queue, verify the match */
Int t4RxPkts (t4TestEncap_t *tencap, pauFifo_t *fifo)
{
	Cppi_HostDesc    *hd;
	UInt			  idx;
	Int				  n;
	pasahoLongInfo_t *pinfo;
	UInt32		      infoLen;
	
	/* Look for packets in the receive queue */
	while (Qmss_getQueueEntryCount (tencap->tf->QGen[Q_MATCH]) > 0)  {
	
		hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tencap->tf->QGen[Q_MATCH])) & ~15);
		if (hd == NULL)  {
			System_printf ("%s (%s:%d): Failed to pop a descriptor off the receive packet queue\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tencap->tf, hd);
			return (-1);
		}
	
		/* The packets must arrive in order */
		idx = commonFifoPopElement (fifo, &n); 
		if (n == 0)  {
			System_printf ("%s (%s:%d): Error - found an empty receive packet tracking index fifo\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tencap->tf, hd);
			return (-1);
		}
					  
		/* Verify the parse information is correct */
		if (Cppi_getPSData (Cppi_DescType_HOST, Cppi_PSLoc_PS_IN_DESC, (Cppi_Desc *)hd, (UInt8 **)&pinfo, &infoLen) != CPPI_SOK)  {
			System_printf ("%s (%s:%d): Error getting control info from received data packet\n", tfName, __FILE__, __LINE__);
			testCommonRecycleLBDesc (tencap->tf, hd);
			return (-1);
		}
			
		if (testCommonComparePktInfo (tfName, t4PktInfo[idx].info, pinfo))  {
			testCommonRecycleLBDesc (tencap->tf, hd);
			return (-1);
		}
		
		testCommonRecycleLBDesc (tencap->tf, hd);
	}

	return (0);
	
}

static Cppi_HostDesc *t4FormExpPacket (tFramework_t *tf, paTest_t *pat, Int pktIdx, Int* fDrop)
{
	Cppi_HostDesc *hd;
	Qmss_Queue     q;
	
	/* Pop a descriptor off of the free queue list, format the return queue, and attach the packet */
	hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tf->QfreeDesc)) & ~15);
	if (hd == NULL)  {
		System_printf ("%s (%s:%d): Failed to pop a descriptor off the free descriptor Q (%d)\n", tfName, __FILE__, __LINE__, tf->QfreeDesc);
		return (NULL);
	}
	
	/* Setup the return for the descriptor */
  	q.qMgr = 0;
  	q.qNum = tf->QfreeDesc;
  	Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
  	
  	/* Make sure there is no control info.  */
  	Cppi_setPSLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, 0);
  	
  	/* Attach the data and set the length */
  	Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, (UInt8 *)utilgAddr((UInt32)(t4ePktInfo[pktIdx].pkt)), t4ePktInfo[pktIdx].pktLen);
  	Cppi_setPacketLen (Cppi_DescType_HOST, (Cppi_Desc *)hd, t4ePktInfo[pktIdx].pktLen);
    
    if(t4ePktInfo[pktIdx].idx ==  T4_DROP_PKT_INDEX)
    {
        /*
         * Only rx padding error packet is dropped at this moment
         * future enhancements may be necessary.
         */
        t4UpdateUsrStats(&paTestExpectedUsrStats, T4_USR_STATS_L2_PADDING_ERR);
        *fDrop = TRUE;
    }
    else
    {
        *fDrop = FALSE;
    }
  	
  	return (hd);
}

static Cppi_HostDesc *t4GetExpPkt (tFramework_t *tf)
{
	Cppi_HostDesc *hd = NULL;
    int index;
	
    for (index = Q_PPPoE_FAIL; index <= Q_IP_FRAG; index++)
    {
    
	    if (Qmss_getQueueEntryCount(tf->QGen[index]) > 0)  {
		    hd = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (tf->QGen[index])) & ~15);
		    if (hd == NULL)  {
			    System_printf ("%s (%s:%d): Failed to pop a received packet from General Queue(%d)\n", tfName, __FILE__, __LINE__, index);
			    return (NULL);
		    }
        
            return (hd);
	    }	
    
    }
    
   return (NULL); 
	
}

/* Search the exception packet queue for received exceptionm packets. Remain in 
 * this function until all buffers are restored to their respective queues */
static Int t4ReceiveExpPkts (tFramework_t *tf, paTest_t *pat, int expCount)
{
	Cppi_HostDesc    *hd;
	UInt32		     *swInfo;
	pktTestInfo_t    *tinfo;
	pasahoLongInfo_t *pinfo;
    UInt8            *pkt;
	UInt32		      infoLen;
	Int               i, j;
	UInt		      chan;
    Int               count = 0;
    
    if (!expCount)
    {
        /* Wait for all packets to be processed by PASS even if all packets are expected to be dropped */
        utilCycleDelay (10000);
    }
	
	for (i = 0; i < 100; i++)  {
		
        utilCycleDelay (1000);
		while ((hd = t4GetExpPkt (tf)) != NULL)  {
			
			/* Verify swInfo0 for packet match and packet ID number */
			Cppi_getSoftwareInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, (UInt8 **)&swInfo);
			
			if ((*swInfo & T4_CMD_SWINFO0_TYPE_MASK) != T4_CMD_SWINFO0_PKT_ID)  {
				System_printf ("%s (%s:%d): Found a packet in the receive packet queue but with incorrect swinfo0 Id: 0x%08x\n",
							   tfName, __FILE__, __LINE__, *swInfo);
			 	testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			
			chan = *swInfo & T4_CMD_SWINFO0_ID_MASK;
            count++;
              
            pkt = (UInt8 *)hd->buffPtr;  
			  
			/* locate the associated test information based on the channel value */
			for (j = 0, tinfo = NULL; j < sizeof(t4ePktInfo) / sizeof(pktTestInfo_t); j++)  {
				if (t4ePktInfo[j].idx == chan)  {
					tinfo = &t4ePktInfo[j + pkt[6]];
					break;
				}
			}
			
			if (tinfo == NULL)  {
				System_printf ("%s (%s:%d): Found a packet in the receive packet queue for channel %d, but found no matching packet info\n",
								tfName, __FILE__, __LINE__, chan);
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			  
			/* Verify the parse information is correct */
			if (Cppi_getPSData (Cppi_DescType_HOST, Cppi_PSLoc_PS_IN_DESC, (Cppi_Desc *)hd, (UInt8 **)&pinfo, &infoLen) != CPPI_SOK)  {
				System_printf ("%s (%s:%d): Error getting control info from received data packet\n", tfName, __FILE__, __LINE__);
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			
			if (testCommonComparePktInfo (tfName, tinfo->info, pinfo))  {
				testCommonRecycleLBDesc (tf, hd);
				return (-1);
			}
			
			/* Return the descriptor/buffer */
			testCommonRecycleLBDesc (tf, hd);								
			
		}
        
        if(count >= expCount)
            break;
		
	}
	
	if (i == 100)  {
		System_printf ("%s (%s:%d): Exception Packet Processing Error - unable to receive all packets (rx = %d, exp = %d)\n", tfName, __FILE__, __LINE__, count, expCount);
        System_flush();
		return (-1);
	}
	
	return (0);

}

Void paTestL3Routing (UArg a0, UArg a1)
{
	t4TestEncap_t  t4Encap;	
 	Cppi_HostDesc  *hd[10];
 	paReturn_t      paret;
 	Int				i, j, n;
 	Int  			cmdDest;
 	UInt16			cmdSize;
 	paTestStatus_t  testStatus = PA_TEST_PASSED;
 	paTestStatus_t  newStatus;
 	
 	UInt			fifoData[T4_EXPTPKT_FIFO_SIZE];
 	pauFifo_t       fifo =  { 0, 0, T4_EXPTPKT_FIFO_SIZE, NULL };
    Int             fDrop, dropCnt;
    int             numUsrStatsEntries;
 
 	volatile Int mdebugWait = 1;
 							   
	/* Initialize the test state */
    numUsrStatsEntries = sizeof(t4UsrStatsSetup)/sizeof(pauUsrStatsSetup_t);
	fifo.data = fifoData;
 	memset (&t4Encap, 0, sizeof(t4Encap));
 	t4Encap.tf  = (tFramework_t *)a0;
 	t4Encap.pat = (paTest_t *)a1;
 	for (i = 0; i < T4_NUM_LOCAL_L4_HANDLES; i++)
 		t4Encap.l4Handles[i].state = T4_HANDLE_UNCONFIGURED; 		
 	for (i = 0; i < T4_NUM_LOCAL_L3_HANDLES; i++)
 		t4Encap.l3Handles[i].state = T4_HANDLE_UNCONFIGURED;	
 	for (i = 0; i < T4_NUM_LOCAL_L2_HANDLES; i++)
 		t4Encap.l2Handles[i].state = T4_HANDLE_UNCONFIGURED;				   
 		
 		
    /* Runtime initial values */
    matchRoute[0].queue = (UInt16) t4Encap.tf->QGen[Q_MATCH];
    nfailRoute.queue    = (UInt16) t4Encap.tf->QGen[Q_NFAIL];
    nfailRoute1.queue   = (UInt16) t4Encap.tf->QGen[Q_MATCH];
    cmdReply.queue      = (UInt16) t4Encap.tf->QGen[Q_CMD_REPLY];
    
    /* Zero out the expected stats. The stats will be updated as packets are sent into PA */
    memset (&paTestL4ExpectedStats, 0, sizeof(paTestL4ExpectedStats));
    memset (&paTestExpectedUsrStats, 0, sizeof(paTestExpectedUsrStats));
    
    /* 
     * Configure the CRC engine for SCTP CRC-32C checksum 
     * The CRC-engine connected to PDSP2 should be configured since the SCTP is within the
     * inner IP paylaod which is parased and lookup at PDSP2
     */
 	cmdReply.replyId = T4_CMD_SWINFO0_CRC_CFG_ID;  
 	cmdReply.queue = t4Encap.tf->QGen[Q_CMD_REPLY];
    
    hd[0] = testCommonConfigCrcEngine(t4Encap.tf, 2, &t4CrcCfg,  t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QLinkedBuf3,
                                      &cmdReply, &cmdDest, &cmdSize, &paret);
 	t4HandleError (&t4Encap, paret, hd[i], __LINE__); /* Will not return on error */   
    
 	Qmss_queuePush (t4Encap.tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[0], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 	paTestL4ExpectedStats.classify1.nPackets += 1;
                            
	if (testCommonWaitCmdReply (t4Encap.tf, t4Encap.pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_CRC_CFG_ID, __LINE__)) {
		System_printf ("%s (%s:%d): testCommonConfigCrcEngine failed\n", tfName, __FILE__, __LINE__);
		newStatus = PA_TEST_FAILED;
 	}
    
 	if (newStatus == PA_TEST_FAILED)  
 		t4Cleanup (&t4Encap, newStatus);  /* No return */
    
 	/* Recycle the command packet as well */
 	hd[0] = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (t4Encap.tf->QGen[Q_CMD_RECYCLE])) & ~15);
 	if (hd[0] == NULL)  {
 		System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 		newStatus = PA_TEST_FAILED;
 	}
    
 	if (newStatus == PA_TEST_FAILED)  
 		t4Cleanup (&t4Encap, newStatus);  /* No return */
    
 	testCommonRecycleLBDesc (t4Encap.tf, hd[0]);
    
    /* Global Configuration */
	newStatus = t4GlobalConfiguration (&t4Encap);
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
    
    /* Exception Route Configurations */
	newStatus = t4ExceptionRoutes (&t4Encap);
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
        
    /* Usr Stats configuration */
    newStatus = t4ConfigUsrStats (&t4Encap, numUsrStatsEntries, t4UsrStatsSetup, FALSE);
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
        
 								  
 	/* Add two mac entries into the table. All the test packets will match one of these */
 	for (i = 0; i < T4_NUM_LOCAL_L2_HANDLES; i++)  {
 		cmdReply.replyId = T4_CMD_SWINFO0_ADD_MAC_ID + i;  /* T2_CMD_SWINFO0_ADD_ID identifies command, 16 LS bits identify the local handle number */
 		cmdReply.queue = t4Encap.tf->QGen[Q_CMD_REPLY];
 	
 		hd[i] = testCommonAddMac (t4Encap.tf, pa_LUT1_INDEX_NOT_SPECIFIED, (paEthInfo_t *)&t4EthInfo[i], &matchRoute[2], &nfailRoute,
 	    	                    &t4Encap.l2Handles[i].paHandle, t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QLinkedBuf1, 
 	        	                &cmdReply, &cmdDest, &cmdSize, &paret);
 	                        
 	    t4HandleError (&t4Encap, paret, hd[i], __LINE__); /* Will not return on error */                    

 	}
 	
 	
 	/* Send the commands to PA. Each will result in 1 packet in classify1 */
 	for (i = 0; i < T4_NUM_LOCAL_L2_HANDLES; i++)  {
 		Qmss_queuePush (t4Encap.tf->QPaTx[cmdDest - pa_CMD_TX_DEST_0], (Ptr)hd[i], cmdSize, TF_SIZE_DESC, Qmss_Location_TAIL);
 		t4Encap.l2Handles[i].state = T4_HANDLE_PENDING_ACK;
 		paTestL4ExpectedStats.classify1.nPackets += 1;
 	}
 	
 	/* Wait for the responses. They will be received in the order in which they were sent */
 	newStatus = testStatus;
 	for (i = 0; i < T4_NUM_LOCAL_L2_HANDLES; i++)  {

		if (testCommonWaitCmdReply (t4Encap.tf, t4Encap.pat, tfName, cmdReply.queue, T4_CMD_SWINFO0_ADD_MAC_ID + i, __LINE__)) {
			System_printf ("%s (%s:%d): testCommonWaitCmdReply failed\n", tfName, __FILE__, __LINE__);
			newStatus = PA_TEST_FAILED;
            break;
 		}
 		
 		/* Recycle the command packet as well */
 		hd[0] = (Cppi_HostDesc *)(((UInt32)Qmss_queuePop (t4Encap.tf->QGen[Q_CMD_RECYCLE])) & ~15);
 		if (hd[0] == NULL)  {
 			System_printf ("%s (%s:%d): Did not find an expected command in the recycle queue\n", tfName, __FILE__, __LINE__);
 			newStatus = PA_TEST_FAILED;
            break;
 		}
 		testCommonRecycleLBDesc (t4Encap.tf, hd[0]);
 	}
 	
 	if (newStatus == PA_TEST_FAILED)  
 		t4Cleanup (&t4Encap, newStatus);  /* No return */
 	

	/* Burst in the next set of command packets. These packets are used as for the outer header in nested
	 * packets. It will include several configurations that are invalid. These are detected and 
	 * verified by the swinfo0 ID */
	n = sizeof (t4OuterIpInfo) / sizeof (t4IpPaSetup_t);
	newStatus = t4OpenIp (&t4Encap, t4OuterIpInfo, n, t4Encap.l2Handles, "Outer IP headers");
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
		
	/* Repeat for Inner IP packets (which are linked to outer IP packets) */
	n = sizeof (t4InnerIpInfo) / sizeof (t4IpPaSetup_t);
	newStatus = t4OpenIp (&t4Encap, t4InnerIpInfo, n, t4Encap.l3Handles, "Inner IP headers");
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
		
	/* One more time for tripple layered IP headers */
	n = sizeof (t4InnerInnerIpInfo) / sizeof (t4IpPaSetup_t);
	newStatus = t4OpenIp (&t4Encap, t4InnerInnerIpInfo, n, t4Encap.l3Handles, "Inner IP headers");
	if (newStatus == PA_TEST_FAILED)
		t4Cleanup (&t4Encap, newStatus);  /* No return */
				
	
	/* Verify and clear the stats */
	newStatus =  testCommonCheckStats (t4Encap.tf, t4Encap.pat, tfName, &paTestL4ExpectedStats, t4Encap.tf->QLinkedBuf1, 
	                       t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QGen[Q_CMD_REPLY], TRUE);
	                       
	if (newStatus == PA_TEST_FAILED)  {
		System_printf ("%s (%s:%d): testCommonCheckStats Failed\n", tfName, __FILE__, __LINE__);
		t4Cleanup (&t4Encap, newStatus);  /* No return */
	}
    
	/* Fire in each of the data packets multiple times */
	n = sizeof (t4PktInfo) / sizeof (pktTestInfo_t);
	for (j = 0; j < T4_NUM_PACKET_ITERATIONS; j++)  {
		for (i = 0; i < n; i++)  {
			hd[0] = t4FormPktDataDescr (&t4Encap, i);
			if (hd[0] == NULL)  {
				System_printf ("%s (%s:%d): Failed to get free descriptor\n", tfName, __FILE__, __LINE__);
				t4Cleanup (&t4Encap, PA_TEST_FAILED);
			}
          
			Qmss_queuePush (t4Encap.tf->QPaTx[0], (Ptr)hd[0], t4PktInfo[i].pktLen, TF_SIZE_DESC, Qmss_Location_TAIL);
			testCommonIncStats (t4PktInfo[i].statsMap, &paTestL4ExpectedStats);	
			
			if ((t4PktInfo[i].idx & T4_PACKET_DEST_MASK) == T4_PACKET_L3_MATCH_VALID)  {
				if (commonFifoPushElement (&fifo, (UInt)i) < 0)  {
					System_printf ("%s (%s:%d): Test failed - fifo is full\n", tfName, __FILE__, __LINE__);
					t4Cleanup (&t4Encap, PA_TEST_FAILED);
				}
			}
		}
            
        /* Dealy to allow packets go through the PASS */    
        utilCycleDelay (4000);
			
		if (t4RxPkts (&t4Encap, &fifo))  
			t4Cleanup (&t4Encap, PA_TEST_FAILED);
	}
	
	/* Give some delay for all packets to pass through the system */
	utilCycleDelay (4000);
	if (t4RxPkts (&t4Encap, &fifo))
		t4Cleanup (&t4Encap, PA_TEST_FAILED);
		
	/* The packet index fifo must be empty */
	n = commonFifoGetCount (&fifo);
	if (n != 0)  {
		System_printf ("%s (%s:%d): Packet reception complete but packet index fifo not empty\n", tfName, __FILE__, __LINE__);
		t4Cleanup (&t4Encap, PA_TEST_FAILED);
	}
	
	/* Verify and clear the stats */
	newStatus =  testCommonCheckStats (t4Encap.tf, t4Encap.pat, tfName, &paTestL4ExpectedStats, t4Encap.tf->QLinkedBuf1, 
	                       t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QGen[Q_CMD_REPLY], TRUE);
	if (newStatus == PA_TEST_FAILED)
    {
		System_printf ("%s (%s:%d): testCommonCheckStats Failed\n", tfName, __FILE__, __LINE__);
	    t4Cleanup (&t4Encap, newStatus);
    }   
    
    /* Exception packet testing */
	/* Run packets through the system. the complete set of packets is run through multiple times. */
	n = sizeof (t4ePktInfo) / sizeof (pktTestInfo_t);
	for (j = 0; j < T4_NUM_PACKET_ITERATIONS; j++)   {
        dropCnt = 0;
		for (i = 0; i < n; i++ )  {
		
			hd[0] = t4FormExpPacket (t4Encap.tf, t4Encap.pat, i, &fDrop);
			
			if (hd[0] == NULL)  {
			    System_printf ("%s (%s:%d): T4 Exception packet sent:  run out of buffers\n", tfName, __FILE__, __LINE__);
                System_flush();  
                t4Cleanup (&t4Encap, PA_TEST_FAILED);  /* no return */
                break;
			}
			
			/* Increment any expected stats */
			testCommonIncStats (t4ePktInfo[i].statsMap, &paTestL4ExpectedStats);
            
			Qmss_queuePush (t4Encap.tf->QPaTx[0], (Ptr)hd[0], hd[0]->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
            
            if(fDrop)dropCnt++;
		}
			
		if (t4ReceiveExpPkts (t4Encap.tf, t4Encap.pat, n - dropCnt))
        {
            /* Error Handling */
			System_printf ("%s (%s:%d): t4ReceiveExpPktstimeout\n", tfName, __FILE__, __LINE__);
            System_flush();  
            t4Cleanup (&t4Encap, PA_TEST_FAILED);  /* no return */
			break;
        }    
	}
    
	/* Verify and clear the Usr stats */
	newStatus =  testCommonCheckUsrStats (t4Encap.tf, t4Encap.pat, tfName, &paTestExpectedUsrStats, t4Encap.tf->QLinkedBuf1, 
	                       t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QGen[Q_CMD_REPLY], TRUE);
	if (newStatus == PA_TEST_FAILED)
    {
		System_printf ("%s (%s:%d): testCommonCheckUsrStats Failed\n", tfName, __FILE__, __LINE__);
		t4Cleanup (&t4Encap, newStatus);
    }
    
	/* Verify and clear the stats */
	newStatus =  testCommonCheckStats (t4Encap.tf, t4Encap.pat, tfName, &paTestL4ExpectedStats, t4Encap.tf->QLinkedBuf1, 
	                       t4Encap.tf->QGen[Q_CMD_RECYCLE], t4Encap.tf->QGen[Q_CMD_REPLY], TRUE);
	if (newStatus == PA_TEST_FAILED)
    {
		System_printf ("%s (%s:%d): testCommonCheckStats Failed\n", tfName, __FILE__, __LINE__);
	    t4Cleanup (&t4Encap, newStatus);
    }   
    
    /* Clear all Usr Stats Link */
    newStatus = t4ConfigUsrStats (&t4Encap, 0, NULL, TRUE);
    
	/* No return from cleanup */
	t4Cleanup (&t4Encap, newStatus);
	
}


 
 
 
 
