/**  
 * @file reassemLib.h
 *
 * @brief 
 *  Holds all the constants, structures and API definitions required by the 
 *  IP Reassembly sample code.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2011, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#ifndef _reassemLib_H_
#define _reassemLib_H_

#ifdef __cplusplus
extern "C" {
#endif

/* C Standard library Include */
#include <stdint.h>
#include <string.h>

/* CPPI LLD include */
#include <ti/drv/cppi/cppi_drv.h>
#include <ti/drv/cppi/cppi_desc.h>

/* QMSS LLD include */
#include <ti/drv/qmss/qmss_drv.h>

/* PA LLD include */
#include <ti/drv/pa/pa.h>
#include <ti/drv/pa/pasahost.h>

/* SA LLD Include Files. */
#ifdef _INCLUDE_SA
#include <ti/drv/sa/salld.h>
#endif

/** 
 * @brief IP Reassembly configuration structure
 */
typedef struct paIPReassemblyConfig_s
{
    uint16_t     numReassemblyContexts;      /**< maxmium number of active reassembly contexts supported */
    uint16_t     descSize;                   /**< CPPI descriptor size */
    uint32_t     timeout;                    /**< Reassembly context timeout in milliseconds */
    
} paIPReassemblyConfig_t;


/** 
 * @brief IP Reassembly Statistics
 */
typedef struct paIPReassemblyStats_s
{
    uint32_t    fragmentsRxed;          /**< number of fragments received */
    uint32_t    reassembledPkts;        /**< number of reassembled packet generated */
    uint32_t    reassemblyTimeout;      /**< number of the timeout event occurs */
    uint32_t    fragmentsDecrypted;     /**< number of decrypted (inner IP) fragments received */
    uint32_t    decryptionFailures;     /**< number of decrypted fragments with errors */
    
} paIPReassemblyStats_t;


/**********************************************************************
 ************************** Exported Functions ************************
 **********************************************************************/
/* IP Reassembly Exported Functions. */
extern int paEx_reassemLibInit(paIPReassemblyConfig_t *pConfig);
extern int paEx_reassemLibProc(Cppi_HostDesc *pHostDesc, Qmss_QueueHnd destQ);
extern int paEx_reassemLibDelete(void);
extern void paEx_reassemLibTimerTick(uint32_t timeElapsed);
extern void paEx_reassemLibQueryStats(paIPReassemblyStats_t *pStats, int doClear);


#ifdef __cplusplus
}
#endif

#endif
