/*
 *
 * Copyright (C) 2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *  File Name: pcie.c
 *
 *  Processing/configuration functions for the PCIe driver.
 *
 */

#include "pcie.h"
#include "pcieloc.h"
#include <ti/csl/cslr_device.h>
#include <ti/csl/cslr_pcie_cfg_space_endpoint.h>
#include <ti/csl/cslr_pcie_cfg_space_rootcomplex.h>
#include <ti/csl/cslr_pciess_app.h>
#include <ti/csl/csl_bootcfgAux.h>

#include <string.h>

/* Global Variable which describes the PCIE LLD Version Information */
const char PCIELLDVersionStr[] = pcie_LLD_VERSION_STR ":" __DATE__  ":" __TIME__;

/*****************************************************************************
 **********  External APIs **********************
 ****************************************************************************/

/*********************************************************************
 * FUNCTION PURPOSE: Sets PCIe mode to RC or EP
 *********************************************************************/
pcieRet_e Pcie_setMode
(
  pcieMode_e mode  /**< [in] PCIE Mode */
)
{
  CSL_BootCfgUnlockKicker();

  CSL_BootCfgSetPCIEMode (mode);  

  CSL_BootCfgLockKicker();

  return pcie_RET_OK;
} /* Pcie_setMode */

/*********************************************************************
 * FUNCTION PURPOSE: Returns the PCIe Internal Address Range for the 
 *                   Memory Space. This range is used for accessing memory.
 *********************************************************************/
pcieRet_e Pcie_getMemSpaceRange 
(
  Pcie_Handle  handle,     /**< [in]  The PCIE LLD instance identifier */
  void         **base,     /**< [out] Memory Space base address */
  uint32_t      *size      /**< [out] Memory Space total size */
)
{
  pcieRet_e retVal = pcie_RET_OK;
  pcie_check_handle(handle);

  if (base) *base = (void *)CSL_PCIE_REGS;
  if (size) *size = 0x10000000; /* 256 MB */

  return retVal;
} /* Pcie_getMemSpaceRange */

/*********************************************************************
 * FUNCTION PURPOSE: Reads any register
 ********************************************************************/
pcieRet_e Pcie_readRegs 
(
  Pcie_Handle      handle,   /**< [in] The PCIE LLD instance identifier */
  pcieLocation_e   location, /**< [in] Local or remote peripheral */
  pcieRegisters_t *readRegs  /**< [in/out] List of registers to read */
)
{
  /* Base Address for the Application Registers */
  CSL_Pciess_appRegs *baseAppRegs = (CSL_Pciess_appRegs *)handle; 

  /* Base Address for the Config Space
     These registers can be Local/Remote and Type0(EP)/Type1(RC) */
  CSL_Pcie_cfg_space_rootcomplexRegs  *baseCfgRcRegs = NULL;
  CSL_Pcie_cfg_space_endpointRegs     *baseCfgEpRegs = NULL;  
  
  pcieRet_e retVal = pcie_RET_OK;
  int i;

  pcie_check_handle(handle);

  /* Get base address for Local or Remote config space */
  if (location == pcie_LOCATION_LOCAL) 
  {
    pcie_get_loc_cfg_base(handle, baseCfgEpRegs, baseCfgRcRegs) 
  }
  else
  {
    pcie_get_rem_cfg_base(handle, baseCfgEpRegs, baseCfgRcRegs) 
  }

  /*****************************************************************************************
  *Application Registers
  *****************************************************************************************/
  if (readRegs->pid) {
    pcie_check_result(retVal, pcie_read_pid_reg (baseAppRegs, readRegs->pid));
  }
  if (readRegs->cmdStatus) {
    pcie_check_result(retVal, pcie_read_cmdStatus_reg (baseAppRegs, readRegs->cmdStatus));
  }
  if (readRegs->cfgTrans) {
    pcie_check_result(retVal, pcie_read_cfgTrans_reg (baseAppRegs, readRegs->cfgTrans));
  }
  if (readRegs->ioBase) {
    pcie_check_result(retVal, pcie_read_ioBase_reg (baseAppRegs, readRegs->ioBase));
  }
  if (readRegs->tlpCfg) {
    pcie_check_result(retVal, pcie_read_tlpCfg_reg (baseAppRegs, readRegs->tlpCfg));
  }
  if (readRegs->rstCmd) {
    pcie_check_result(retVal, pcie_read_rstCmd_reg (baseAppRegs, readRegs->rstCmd));
  }
  if (readRegs->pmCmd) {
    pcie_check_result(retVal, pcie_read_pmCmd_reg (baseAppRegs, readRegs->pmCmd));
  }
  if (readRegs->pmCfg) {
    pcie_check_result(retVal, pcie_read_pmCfg_reg (baseAppRegs, readRegs->pmCfg));
  }
  if (readRegs->actStatus) {
    pcie_check_result(retVal, pcie_read_actStatus_reg (baseAppRegs, readRegs->actStatus));
  }
  if (readRegs->obSize) {
    pcie_check_result(retVal, pcie_read_obSize_reg (baseAppRegs, readRegs->obSize));
  }
  if (readRegs->diagCtrl) {
    pcie_check_result(retVal, pcie_read_diagCtrl_reg (baseAppRegs, readRegs->diagCtrl));
  }
  if (readRegs->endian) {
    pcie_check_result(retVal, pcie_read_endian_reg (baseAppRegs, readRegs->endian));
  }
  if (readRegs->priority) {
    pcie_check_result(retVal, pcie_read_priority_reg (baseAppRegs, readRegs->priority));
  }
  if (readRegs->irqEOI) {
    pcie_check_result(retVal, pcie_read_irqEOI_reg (baseAppRegs, readRegs->irqEOI));
  }
  if (readRegs->msiIrq) {
    pcie_check_result(retVal, pcie_read_msiIrq_reg (baseAppRegs, readRegs->msiIrq));
  }
  if (readRegs->epIrqSet) {
    pcie_check_result(retVal, pcie_read_epIrqSet_reg (baseAppRegs, readRegs->epIrqSet));
  }
  if (readRegs->epIrqClr) {
    pcie_check_result(retVal, pcie_read_epIrqClr_reg (baseAppRegs, readRegs->epIrqClr));
  }
  if (readRegs->epIrqStatus) {
    pcie_check_result(retVal, pcie_read_epIrqStatus_reg (baseAppRegs, readRegs->epIrqStatus));
  }
  for (i = 0; i < 4; i++) {
    if (readRegs->genPurpose[i]) {
      pcie_check_result(retVal, pcie_read_genPurpose_reg (baseAppRegs, readRegs->genPurpose[i], i));
    }
  }
  for (i = 0; i < 8; i++) {
    if (readRegs->msiIrqStatusRaw[i]) {
      pcie_check_result(retVal, pcie_read_msiIrqStatusRaw_reg (baseAppRegs, readRegs->msiIrqStatusRaw[i], i));
    }
    if (readRegs->msiIrqStatus[i]) {
      pcie_check_result(retVal, pcie_read_msiIrqStatus_reg (baseAppRegs, readRegs->msiIrqStatus[i], i));
    }
    if (readRegs->msiIrqEnableSet[i]) {
      pcie_check_result(retVal, pcie_read_msiIrqEnableSet_reg (baseAppRegs, readRegs->msiIrqEnableSet[i], i));
    }
    if (readRegs->msiIrqEnableClr[i]) {
      pcie_check_result(retVal, pcie_read_msiIrqEnableClr_reg (baseAppRegs, readRegs->msiIrqEnableClr[i], i));
    }
  }
  for (i = 0; i < 4; i++) {
    if (readRegs->legacyIrqStatusRaw[i]) {
      pcie_check_result(retVal, pcie_read_legacyIrqStatusRaw_reg (baseAppRegs, readRegs->legacyIrqStatusRaw[i], i));
    }
    if (readRegs->legacyIrqStatus[i]) {
      pcie_check_result(retVal, pcie_read_legacyIrqStatus_reg (baseAppRegs, readRegs->legacyIrqStatus[i], i));
    }
    if (readRegs->legacyIrqEnableSet[i]) {
      pcie_check_result(retVal, pcie_read_legacyIrqEnableSet_reg (baseAppRegs, readRegs->legacyIrqEnableSet[i], i));
    }
    if (readRegs->legacyIrqEnableClr[i]) {
      pcie_check_result(retVal, pcie_read_legacyIrqEnableClr_reg (baseAppRegs, readRegs->legacyIrqEnableClr[i], i));
    }
  }
  if (readRegs->errIrqStatusRaw) {
    pcie_check_result(retVal, pcie_read_errIrqStatusRaw_reg (baseAppRegs, readRegs->errIrqStatusRaw));
  }
  if (readRegs->errIrqStatus) {
    pcie_check_result(retVal, pcie_read_errIrqStatus_reg (baseAppRegs, readRegs->errIrqStatus));
  }
  if (readRegs->errIrqEnableSet) {
    pcie_check_result(retVal, pcie_read_errIrqEnableSet_reg (baseAppRegs, readRegs->errIrqEnableSet));
  }
  if (readRegs->errIrqEnableClr) {
    pcie_check_result(retVal, pcie_read_errIrqEnableClr_reg (baseAppRegs, readRegs->errIrqEnableClr));
  }

  if (readRegs->pmRstIrqStatusRaw) {
    pcie_check_result(retVal, pcie_read_pmRstIrqStatusRaw_reg (baseAppRegs, readRegs->pmRstIrqStatusRaw));
  }
  if (readRegs->pmRstIrqStatus) {
    pcie_check_result(retVal, pcie_read_pmRstIrqStatus_reg (baseAppRegs, readRegs->pmRstIrqStatus));
  }
  if (readRegs->pmRstIrqEnableSet) {
    pcie_check_result(retVal, pcie_read_pmRstIrqEnableSet_reg (baseAppRegs, readRegs->pmRstIrqEnableSet));
  }
  if (readRegs->pmRstIrqEnableClr) {
    pcie_check_result(retVal, pcie_read_pmRstIrqEnableClr_reg (baseAppRegs, readRegs->pmRstIrqEnableClr));
  }

  for (i = 0; i < 8; i ++) {
    if (readRegs->obOffsetLo[i]) {
      pcie_check_result(retVal, pcie_read_obOffsetLo_reg (baseAppRegs, readRegs->obOffsetLo[i], i));
    }
    if (readRegs->obOffsetHi[i]) {
      pcie_check_result(retVal, pcie_read_obOffsetHi_reg (baseAppRegs, readRegs->obOffsetHi[i], i));
    }
  }

  for (i = 0; i < 4; i ++) {
    if (readRegs->ibBar[i]) {
      pcie_check_result(retVal, pcie_read_ibBar_reg (baseAppRegs,  readRegs->ibBar[i], i));
    }
    if (readRegs->ibStartLo[i]) {
      pcie_check_result(retVal, pcie_read_ibStartLo_reg (baseAppRegs,  readRegs->ibStartLo[i], i));
    }
    if (readRegs->ibStartHi[i]) {
      pcie_check_result(retVal, pcie_read_ibStartHi_reg (baseAppRegs,  readRegs->ibStartHi[i], i));
    }
    if (readRegs->ibOffset[i]) {
      pcie_check_result(retVal, pcie_read_ibOffset_reg (baseAppRegs,  readRegs->ibOffset[i], i));
    }
  }

  if (readRegs->pcsCfg0) {
    pcie_check_result(retVal, pcie_read_pcsCfg0_reg (baseAppRegs, readRegs->pcsCfg0));
  }
  if (readRegs->pcsCfg1) {
    pcie_check_result(retVal, pcie_read_pcsCfg1_reg (baseAppRegs, readRegs->pcsCfg1));
  }
  if (readRegs->pcsStatus) {
    pcie_check_result(retVal, pcie_read_pcsStatus_reg (baseAppRegs, readRegs->pcsStatus));
  }

  if (readRegs->serdesCfg0) {
    pcie_check_result(retVal, pcie_read_serdesCfg0_reg (baseAppRegs, readRegs->serdesCfg0));
  }
  if (readRegs->serdesCfg1) {
    pcie_check_result(retVal, pcie_read_serdesCfg1_reg (baseAppRegs, readRegs->serdesCfg1));
  }

  /*****************************************************************************************
  *Configuration Registers
  *****************************************************************************************/

  /*Type 0, Type1 Common Registers*/

  if (readRegs->vndDevId) {
    pcie_check_result(retVal, pcie_read_vndDevId_reg (baseCfgEpRegs, readRegs->vndDevId));
  }
  if (readRegs->statusCmd) {
    pcie_check_result(retVal, pcie_read_statusCmd_reg (baseCfgEpRegs, readRegs->statusCmd));
  }
  if (readRegs->revId) {
    pcie_check_result(retVal, pcie_read_revId_reg (baseCfgEpRegs, readRegs->revId));
  }

  /*Type 0 Registers*/
  if (readRegs->bist) {
    pcie_check_result(retVal, pcie_read_bist_reg (baseCfgEpRegs, readRegs->bist));
  }
  if (readRegs->type0BarIdx) {
    pcie_check_result(retVal, pcie_read_type0Bar_reg (baseCfgEpRegs, &(readRegs->type0BarIdx->reg), 
                                                                       readRegs->type0BarIdx->idx));
  }
  if (readRegs->type0Bar32bitIdx) {
    pcie_check_result(retVal, pcie_read_type0Bar32bit_reg (baseCfgEpRegs, &(readRegs->type0Bar32bitIdx->reg),
                                                                            readRegs->type0Bar32bitIdx->idx));
  }
  if (readRegs->subId) {
    pcie_check_result(retVal, pcie_read_subId_reg (baseCfgEpRegs, readRegs->subId));
  }
  if (readRegs->expRom) {
    pcie_check_result(retVal, pcie_read_expRom_reg (baseCfgEpRegs, readRegs->expRom));
  }
  if (readRegs->capPtr) {
    pcie_check_result(retVal, pcie_read_capPtr_reg (baseCfgEpRegs, readRegs->capPtr));
  }
  if (readRegs->intPin) {
    pcie_check_result(retVal, pcie_read_intPin_reg (baseCfgEpRegs, readRegs->intPin));
  }

  /*Type 1 Registers*/
  if (readRegs->type1BistHeader) {
    pcie_check_result(retVal, pcie_read_type1BistHeader_reg (baseCfgRcRegs, readRegs->type1BistHeader));
  }
  if (readRegs->type1BarIdx) {
    pcie_check_result(retVal, pcie_read_type1Bar_reg (baseCfgRcRegs, &(readRegs->type1BarIdx->reg), 
                                                                       readRegs->type1BarIdx->idx));
  }
  if (readRegs->type1Bar32bitIdx) {
    pcie_check_result(retVal, pcie_read_type1Bar32bit_reg (baseCfgRcRegs, &(readRegs->type1Bar32bitIdx->reg),
                                                                            readRegs->type1Bar32bitIdx->idx));
  }
  if (readRegs->type1BusNum) {
    pcie_check_result(retVal, pcie_read_type1BusNum_reg (baseCfgRcRegs, readRegs->type1BusNum));
  }
  if (readRegs->type1SecStat) {
    pcie_check_result(retVal, pcie_read_type1SecStat_reg (baseCfgRcRegs, readRegs->type1SecStat));
  }
  if (readRegs->type1Memspace) {
    pcie_check_result(retVal, pcie_read_type1Memspace_reg (baseCfgRcRegs, readRegs->type1Memspace));
  }
  if (readRegs->prefMem) {
    pcie_check_result(retVal, pcie_read_prefMem_reg (baseCfgRcRegs, readRegs->prefMem));
  }
  if (readRegs->prefBaseUpper) {
    pcie_check_result(retVal, pcie_read_prefBaseUpper_reg (baseCfgRcRegs, readRegs->prefBaseUpper));
  }
  if (readRegs->prefLimitUpper) {
    pcie_check_result(retVal, pcie_read_prefLimitUpper_reg (baseCfgRcRegs, readRegs->prefLimitUpper));
  }
  if (readRegs->type1IOSpace) {
    pcie_check_result(retVal, pcie_read_type1IOSpace_reg (baseCfgRcRegs, readRegs->type1IOSpace));
  }
  if (readRegs->type1CapPtr) {
    pcie_check_result(retVal, pcie_read_type1CapPtr_reg (baseCfgRcRegs, readRegs->type1CapPtr));
  }
  if (readRegs->type1ExpnsnRom) {
    pcie_check_result(retVal, pcie_read_type1ExpnsnRom_reg (baseCfgRcRegs, readRegs->type1ExpnsnRom));
  }
  if (readRegs->type1BridgeInt) {
    pcie_check_result(retVal, pcie_read_type1BridgeInt_reg (baseCfgRcRegs, readRegs->type1BridgeInt));
  }

  /* Power Management Capabilities Registers */
  if (readRegs->pmCap) {
    pcie_check_result(retVal, pcie_read_pmCap_reg (baseCfgEpRegs, readRegs->pmCap));
  }
  if (readRegs->pmCapCtlStat) {
    pcie_check_result(retVal, pcie_read_pmCapCtlStat_reg (baseCfgEpRegs, readRegs->pmCapCtlStat));
  }

  /*MSI Registers*/
  if (readRegs->msiCap) {
    pcie_check_result(retVal, pcie_read_msiCap_reg (baseCfgEpRegs, readRegs->msiCap));
  }
  if (readRegs->msiLo32) {
    pcie_check_result(retVal, pcie_read_msiLo32_reg (baseCfgEpRegs, readRegs->msiLo32));
  }
  if (readRegs->msiUp32) {
    pcie_check_result(retVal, pcie_read_msiUp32_reg (baseCfgEpRegs, readRegs->msiUp32));
  }
  if (readRegs->msiData) {
    pcie_check_result(retVal, pcie_read_msiData_reg (baseCfgEpRegs, readRegs->msiData));
  }

  /*Capabilities Registers*/
  if (readRegs->pciesCap) {
    pcie_check_result(retVal, pcie_read_pciesCap_reg (baseCfgEpRegs, readRegs->pciesCap));
  }
  if (readRegs->deviceCap) {
    pcie_check_result(retVal, pcie_read_deviceCap_reg (baseCfgEpRegs, readRegs->deviceCap));
  }
  if (readRegs->devStatCtrl) {
    pcie_check_result(retVal, pcie_read_devStatCtrl_reg (baseCfgEpRegs, readRegs->devStatCtrl));
  }
  if (readRegs->linkCap) {
    pcie_check_result(retVal, pcie_read_linkCap_reg (baseCfgEpRegs, readRegs->linkCap));
  }
  if (readRegs->linkStatCtrl) {
    pcie_check_result(retVal, pcie_read_linkStatCtrl_reg (baseCfgEpRegs, readRegs->linkStatCtrl));
  }
  if (readRegs->slotCap) {
    pcie_check_result(retVal, pcie_read_slotCap_reg (baseCfgRcRegs, readRegs->slotCap));
  }
  if (readRegs->slotStatCtrl) {
    pcie_check_result(retVal, pcie_read_slotStatCtrl_reg (baseCfgRcRegs, readRegs->slotStatCtrl));
  }
  if (readRegs->rootCtrlCap) {
    pcie_check_result(retVal, pcie_read_rootCtrlCap_reg (baseCfgRcRegs, readRegs->rootCtrlCap));
  }
  if (readRegs->rootStatus) {
    pcie_check_result(retVal, pcie_read_rootStatus_reg (baseCfgRcRegs, readRegs->rootStatus));
  }
  if (readRegs->devCap2) {
    pcie_check_result(retVal, pcie_read_devCap2_reg (baseCfgEpRegs, readRegs->devCap2));
  }
  if (readRegs->devStatCtrl2) {
    pcie_check_result(retVal, pcie_read_devStatCtrl2_reg (baseCfgEpRegs, readRegs->devStatCtrl2));
  }
  if (readRegs->linkCtrl2) {
    pcie_check_result(retVal, pcie_read_linkCtrl2_reg (baseCfgEpRegs, readRegs->linkCtrl2));
  }


  /*Capabilities Extended Registers*/
  if (readRegs->extCap) {
    pcie_check_result(retVal, pcie_read_extCap_reg (baseCfgEpRegs, readRegs->extCap));
  }
  if (readRegs->uncErr) {
    pcie_check_result(retVal, pcie_read_uncErr_reg (baseCfgEpRegs, readRegs->uncErr));
  }
  if (readRegs->uncErrMask) {
    pcie_check_result(retVal, pcie_read_uncErrMask_reg (baseCfgEpRegs, readRegs->uncErrMask));
  }
  if (readRegs->uncErrSvrty) {
    pcie_check_result(retVal, pcie_read_uncErrSvrty_reg (baseCfgEpRegs, readRegs->uncErrSvrty));
  }
  if (readRegs->corErr) {
    pcie_check_result(retVal, pcie_read_corErr_reg (baseCfgEpRegs, readRegs->corErr));
  }
  if (readRegs->corErrMask) {
    pcie_check_result(retVal, pcie_read_corErrMask_reg (baseCfgEpRegs, readRegs->corErrMask));
  }
  if (readRegs->accr) {
    pcie_check_result(retVal, pcie_read_accr_reg (baseCfgEpRegs, readRegs->accr));
  }
  for (i = 0; i < 4; i ++) {
    if (readRegs->hdrLog[i]) {
      pcie_check_result(retVal, pcie_read_hdrLog_reg (baseCfgEpRegs, readRegs->hdrLog[i], i));
    }
  }
  if (readRegs->rootErrCmd) {
    pcie_check_result(retVal, pcie_read_rootErrCmd_reg (baseCfgEpRegs, readRegs->rootErrCmd));
  }
  if (readRegs->rootErrSt) {
    pcie_check_result(retVal, pcie_read_rootErrSt_reg (baseCfgEpRegs, readRegs->rootErrSt));
  }
  if (readRegs->errSrcID) {
    pcie_check_result(retVal, pcie_read_errSrcID_reg (baseCfgEpRegs, readRegs->errSrcID));
  }

  /*Port Logic Registers*/
  if (readRegs->plAckTimer) {
    pcie_check_result(retVal, pcie_read_plAckTimer_reg (baseCfgEpRegs, readRegs->plAckTimer));
  }
  if (readRegs->plOMsg) {
    pcie_check_result(retVal, pcie_read_plOMsg_reg (baseCfgEpRegs, readRegs->plOMsg));
  }
  if (readRegs->plForceLink) {
    pcie_check_result(retVal, pcie_read_plForceLink_reg (baseCfgEpRegs, readRegs->plForceLink));
  }
  if (readRegs->ackFreq) {
    pcie_check_result(retVal, pcie_read_ackFreq_reg (baseCfgEpRegs, readRegs->ackFreq));
  }
  if (readRegs->lnkCtrl) {
    pcie_check_result(retVal, pcie_read_lnkCtrl_reg (baseCfgEpRegs, readRegs->lnkCtrl));
  }
  if (readRegs->laneSkew) {
    pcie_check_result(retVal, pcie_read_laneSkew_reg (baseCfgEpRegs, readRegs->laneSkew));
  }
  if (readRegs->symNum) {
    pcie_check_result(retVal, pcie_read_symNum_reg (baseCfgEpRegs, readRegs->symNum));
  }
  if (readRegs->symTimerFltMask) {
    pcie_check_result(retVal, pcie_read_symTimerFltMask_reg (baseCfgEpRegs, readRegs->symTimerFltMask));
  }
  if (readRegs->fltMask2) {
    pcie_check_result(retVal, pcie_read_fltMask2_reg (baseCfgEpRegs, readRegs->fltMask2));
  }
  if (readRegs->debug0) {
    pcie_check_result(retVal, pcie_read_debug0_reg (baseCfgEpRegs, readRegs->debug0));
  }
  if (readRegs->debug1) {
    pcie_check_result(retVal, pcie_read_debug1_reg (baseCfgEpRegs, readRegs->debug1));
  }
  if (readRegs->gen2) {
    pcie_check_result(retVal, pcie_read_gen2_reg (baseCfgEpRegs, readRegs->gen2));
  }

  return retVal;
} /* Pcie_readRegs */


/*********************************************************************
 * FUNCTION PURPOSE: Writes any register
 ********************************************************************/
pcieRet_e Pcie_writeRegs 
(
  Pcie_Handle      handle,   /**< [in] The PCIE LLD instance identifier */
  pcieLocation_e   location, /**< [in] Local or remote peripheral */
  pcieRegisters_t *writeRegs /**< [in] List of registers to write */
)
{
  /* Base Address for the Application Registers */
  CSL_Pciess_appRegs *baseAppRegs = (CSL_Pciess_appRegs *)handle; 

  /* Base Address for the Config Space
     These registers can be Local/Remote and Type0(EP)/Type1(RC) */
  CSL_Pcie_cfg_space_rootcomplexRegs  *baseCfgRcRegs = NULL;
  CSL_Pcie_cfg_space_endpointRegs     *baseCfgEpRegs = NULL;  
  
  pcieRet_e retVal = pcie_RET_OK;
  int i;

  pcie_check_handle(handle);

  /* Get base address for Local/Remote config space */
  if (location == pcie_LOCATION_LOCAL) 
  {
    pcie_get_loc_cfg_base(handle, baseCfgEpRegs, baseCfgRcRegs) 
  }
  else
  {
    pcie_get_rem_cfg_base(handle, baseCfgEpRegs, baseCfgRcRegs) 
  }

  /*****************************************************************************************
  *Application Registers
  *****************************************************************************************/
  if (writeRegs->cmdStatus) {
    pcie_check_result(retVal, pcie_write_cmdStatus_reg (baseAppRegs, writeRegs->cmdStatus));
  }
  if (writeRegs->cfgTrans) {
    pcie_check_result(retVal, pcie_write_cfgTrans_reg (baseAppRegs, writeRegs->cfgTrans));
  }
  if (writeRegs->ioBase) {
    pcie_check_result(retVal, pcie_write_ioBase_reg (baseAppRegs, writeRegs->ioBase));
  }
  if (writeRegs->tlpCfg) {
    pcie_check_result(retVal, pcie_write_tlpCfg_reg (baseAppRegs, writeRegs->tlpCfg));
  }
  if (writeRegs->rstCmd) {
    pcie_check_result(retVal, pcie_write_rstCmd_reg (baseAppRegs, writeRegs->rstCmd));
  }
  if (writeRegs->pmCmd) {
    pcie_check_result(retVal, pcie_write_pmCmd_reg (baseAppRegs, writeRegs->pmCmd));
  }
  if (writeRegs->pmCfg) {
    pcie_check_result(retVal, pcie_write_pmCfg_reg (baseAppRegs, writeRegs->pmCfg));
  }
  if (writeRegs->obSize) {
    pcie_check_result(retVal, pcie_write_obSize_reg (baseAppRegs, writeRegs->obSize));
  }
  if (writeRegs->diagCtrl) {
    pcie_check_result(retVal, pcie_write_diagCtrl_reg (baseAppRegs, writeRegs->diagCtrl));
  }
  if (writeRegs->endian) {
    pcie_check_result(retVal, pcie_write_endian_reg (baseAppRegs, writeRegs->endian));
  }
  if (writeRegs->priority) {
    pcie_check_result(retVal, pcie_write_priority_reg (baseAppRegs, writeRegs->priority));
  }
  if (writeRegs->irqEOI) {
    pcie_check_result(retVal, pcie_write_irqEOI_reg (baseAppRegs, writeRegs->irqEOI));
  }
  if (writeRegs->msiIrq) {
    pcie_check_result(retVal, pcie_write_msiIrq_reg (baseAppRegs, writeRegs->msiIrq));
  }
  if (writeRegs->epIrqSet) {
    pcie_check_result(retVal, pcie_write_epIrqSet_reg (baseAppRegs, writeRegs->epIrqSet));
  }
  if (writeRegs->epIrqClr) {
    pcie_check_result(retVal, pcie_write_epIrqClr_reg (baseAppRegs, writeRegs->epIrqClr));
  }
  if (writeRegs->epIrqStatus) {
    pcie_check_result(retVal, pcie_write_epIrqStatus_reg (baseAppRegs, writeRegs->epIrqStatus));
  }
  for (i = 0; i < 4; i++) {
    if (writeRegs->genPurpose[i]) {
      pcie_check_result(retVal, pcie_write_genPurpose_reg (baseAppRegs, writeRegs->genPurpose[i], i));
    }
  }
  for (i = 0; i < 8; i++) {
    if (writeRegs->msiIrqStatusRaw[i]) {
      pcie_check_result(retVal, pcie_write_msiIrqStatusRaw_reg (baseAppRegs, writeRegs->msiIrqStatusRaw[i], i));
    }
    if (writeRegs->msiIrqStatus[i]) {
      pcie_check_result(retVal, pcie_write_msiIrqStatus_reg (baseAppRegs, writeRegs->msiIrqStatus[i], i));
    }
    if (writeRegs->msiIrqEnableSet[i]) {
      pcie_check_result(retVal, pcie_write_msiIrqEnableSet_reg (baseAppRegs, writeRegs->msiIrqEnableSet[i], i));
    }
    if (writeRegs->msiIrqEnableClr[i]) {
      pcie_check_result(retVal, pcie_write_msiIrqEnableClr_reg (baseAppRegs, writeRegs->msiIrqEnableClr[i], i));
    }
  }
  for (i = 0; i < 4; i++) {
    if (writeRegs->legacyIrqStatusRaw[i]) {
      pcie_check_result(retVal, pcie_write_legacyIrqStatusRaw_reg (baseAppRegs, writeRegs->legacyIrqStatusRaw[i], i));
    }
    if (writeRegs->legacyIrqStatus[i]) {
      pcie_check_result(retVal, pcie_write_legacyIrqStatus_reg (baseAppRegs, writeRegs->legacyIrqStatus[i], i));
    }
    if (writeRegs->legacyIrqEnableSet[i]) {
      pcie_check_result(retVal, pcie_write_legacyIrqEnableSet_reg (baseAppRegs, writeRegs->legacyIrqEnableSet[i], i));
    }
    if (writeRegs->legacyIrqEnableClr[i]) {
      pcie_check_result(retVal, pcie_write_legacyIrqEnableClr_reg (baseAppRegs, writeRegs->legacyIrqEnableClr[i], i));
    }
  }
  if (writeRegs->errIrqStatusRaw) {
    pcie_check_result(retVal, pcie_write_errIrqStatusRaw_reg (baseAppRegs, writeRegs->errIrqStatusRaw));
  }
  if (writeRegs->errIrqStatus) {
    pcie_check_result(retVal, pcie_write_errIrqStatus_reg (baseAppRegs, writeRegs->errIrqStatus));
  }
  if (writeRegs->errIrqEnableSet) {
    pcie_check_result(retVal, pcie_write_errIrqEnableSet_reg (baseAppRegs, writeRegs->errIrqEnableSet));
  }
  if (writeRegs->errIrqEnableClr) {
    pcie_check_result(retVal, pcie_write_errIrqEnableClr_reg (baseAppRegs, writeRegs->errIrqEnableClr));
  }

  if (writeRegs->pmRstIrqStatusRaw) {
    pcie_check_result(retVal, pcie_write_pmRstIrqStatusRaw_reg (baseAppRegs, writeRegs->pmRstIrqStatusRaw));
  }
  if (writeRegs->pmRstIrqStatus) {
    pcie_check_result(retVal, pcie_write_pmRstIrqStatus_reg (baseAppRegs, writeRegs->pmRstIrqStatus));
  }
  if (writeRegs->pmRstIrqEnableSet) {
    pcie_check_result(retVal, pcie_write_pmRstIrqEnableSet_reg (baseAppRegs, writeRegs->pmRstIrqEnableSet));
  }
  if (writeRegs->pmRstIrqEnableClr) {
    pcie_check_result(retVal, pcie_write_pmRstIrqEnableClr_reg (baseAppRegs, writeRegs->pmRstIrqEnableClr));
  }

  for (i = 0; i < 8; i ++) {
    if (writeRegs->obOffsetLo[i]) {
      pcie_check_result(retVal, pcie_write_obOffsetLo_reg (baseAppRegs, writeRegs->obOffsetLo[i], i));
    }
    if (writeRegs->obOffsetHi[i]) {
      pcie_check_result(retVal, pcie_write_obOffsetHi_reg (baseAppRegs, writeRegs->obOffsetHi[i], i));
    }
  }

  for (i = 0; i < 4; i ++) {
    if (writeRegs->ibBar[i]) {
      pcie_check_result(retVal, pcie_write_ibBar_reg (baseAppRegs,  writeRegs->ibBar[i], i));
    }
    if (writeRegs->ibStartLo[i]) {
      pcie_check_result(retVal, pcie_write_ibStartLo_reg (baseAppRegs,  writeRegs->ibStartLo[i], i));
    }
    if (writeRegs->ibStartHi[i]) {
      pcie_check_result(retVal, pcie_write_ibStartHi_reg (baseAppRegs,  writeRegs->ibStartHi[i], i));
    }
    if (writeRegs->ibOffset[i]) {
      pcie_check_result(retVal, pcie_write_ibOffset_reg (baseAppRegs,  writeRegs->ibOffset[i], i));
    }
  }

  if (writeRegs->pcsCfg0) {
    pcie_check_result(retVal, pcie_write_pcsCfg0_reg (baseAppRegs, writeRegs->pcsCfg0));
  }
  if (writeRegs->pcsCfg1) {
    pcie_check_result(retVal, pcie_write_pcsCfg1_reg (baseAppRegs, writeRegs->pcsCfg1));
  }

  if (writeRegs->serdesCfg0) {
    pcie_check_result(retVal, pcie_write_serdesCfg0_reg (baseAppRegs, writeRegs->serdesCfg0));
  }
  if (writeRegs->serdesCfg1) {
    pcie_check_result(retVal, pcie_write_serdesCfg1_reg (baseAppRegs, writeRegs->serdesCfg1));
  }

  /*****************************************************************************************
  *Configuration Registers
  *****************************************************************************************/

  /*Type 0, Type1 Common Registers*/

  if (writeRegs->vndDevId) {
    pcie_check_result(retVal, pcie_write_vndDevId_reg (baseCfgEpRegs, writeRegs->vndDevId));
  }
  if (writeRegs->statusCmd) {
    pcie_check_result(retVal, pcie_write_statusCmd_reg (baseCfgEpRegs, writeRegs->statusCmd));
  }
  if (writeRegs->revId) {
    pcie_check_result(retVal, pcie_write_revId_reg (baseCfgEpRegs, writeRegs->revId));
  }

  /*Type 0 Registers*/
  if (writeRegs->bist) {
    pcie_check_result(retVal, pcie_write_bist_reg (baseCfgEpRegs, writeRegs->bist));
  }
  if (writeRegs->type0BarIdx) {
    pcie_check_result(retVal, pcie_write_type0Bar_reg (baseCfgEpRegs, &(writeRegs->type0BarIdx->reg), 
                                                                       writeRegs->type0BarIdx->idx));
  }
  if (writeRegs->type0Bar32bitIdx) {
    pcie_check_result(retVal, pcie_write_type0Bar32bit_reg (baseCfgEpRegs, &(writeRegs->type0Bar32bitIdx->reg),
                                                                            writeRegs->type0Bar32bitIdx->idx));
  }
  if (writeRegs->subId) {
    pcie_check_result(retVal, pcie_write_subId_reg (baseCfgEpRegs, writeRegs->subId));
  }
  if (writeRegs->expRom) {
    pcie_check_result(retVal, pcie_write_expRom_reg (baseCfgEpRegs, writeRegs->expRom));
  }
  if (writeRegs->capPtr) {
    pcie_check_result(retVal, pcie_write_capPtr_reg (baseCfgEpRegs, writeRegs->capPtr));
  }
  if (writeRegs->intPin) {
    pcie_check_result(retVal, pcie_write_intPin_reg (baseCfgEpRegs, writeRegs->intPin));
  }

  /*Type 1 Registers*/
  if (writeRegs->type1BistHeader) {
    pcie_check_result(retVal, pcie_write_type1BistHeader_reg (baseCfgRcRegs, writeRegs->type1BistHeader));
  }
  if (writeRegs->type1BarIdx) {
    pcie_check_result(retVal, pcie_write_type1Bar_reg (baseCfgRcRegs, &(writeRegs->type1BarIdx->reg), 
                                                                       writeRegs->type1BarIdx->idx));
  }
  if (writeRegs->type1Bar32bitIdx) {
    pcie_check_result(retVal, pcie_write_type1Bar32bit_reg (baseCfgRcRegs, &(writeRegs->type1Bar32bitIdx->reg),
                                                                            writeRegs->type1Bar32bitIdx->idx));
  }
  if (writeRegs->type1BusNum) {
    pcie_check_result(retVal, pcie_write_type1BusNum_reg (baseCfgRcRegs, writeRegs->type1BusNum));
  }
  if (writeRegs->type1SecStat) {
    pcie_check_result(retVal, pcie_write_type1SecStat_reg (baseCfgRcRegs, writeRegs->type1SecStat));
  }
  if (writeRegs->type1Memspace) {
    pcie_check_result(retVal, pcie_write_type1Memspace_reg (baseCfgRcRegs, writeRegs->type1Memspace));
  }
  if (writeRegs->prefMem) {
    pcie_check_result(retVal, pcie_write_prefMem_reg (baseCfgRcRegs, writeRegs->prefMem));
  }
  if (writeRegs->prefBaseUpper) {
    pcie_check_result(retVal, pcie_write_prefBaseUpper_reg (baseCfgRcRegs, writeRegs->prefBaseUpper));
  }
  if (writeRegs->prefLimitUpper) {
    pcie_check_result(retVal, pcie_write_prefLimitUpper_reg (baseCfgRcRegs, writeRegs->prefLimitUpper));
  }
  if (writeRegs->type1IOSpace) {
    pcie_check_result(retVal, pcie_write_type1IOSpace_reg (baseCfgRcRegs, writeRegs->type1IOSpace));
  }
  if (writeRegs->type1CapPtr) {
    pcie_check_result(retVal, pcie_write_type1CapPtr_reg (baseCfgRcRegs, writeRegs->type1CapPtr));
  }
  if (writeRegs->type1ExpnsnRom) {
    pcie_check_result(retVal, pcie_write_type1ExpnsnRom_reg (baseCfgRcRegs, writeRegs->type1ExpnsnRom));
  }
  if (writeRegs->type1BridgeInt) {
    pcie_check_result(retVal, pcie_write_type1BridgeInt_reg (baseCfgRcRegs, writeRegs->type1BridgeInt));
  }

  /* Power Management Capabilities Registers */
  if (writeRegs->pmCap) {
    pcie_check_result(retVal, pcie_write_pmCap_reg (baseCfgEpRegs, writeRegs->pmCap));
  }
  if (writeRegs->pmCapCtlStat) {
    pcie_check_result(retVal, pcie_write_pmCapCtlStat_reg (baseCfgEpRegs, writeRegs->pmCapCtlStat));
  }

  /*MSI Registers*/
  if (writeRegs->msiCap) {
    pcie_check_result(retVal, pcie_write_msiCap_reg (baseCfgEpRegs, writeRegs->msiCap));
  }
  if (writeRegs->msiLo32) {
    pcie_check_result(retVal, pcie_write_msiLo32_reg (baseCfgEpRegs, writeRegs->msiLo32));
  }
  if (writeRegs->msiUp32) {
    pcie_check_result(retVal, pcie_write_msiUp32_reg (baseCfgEpRegs, writeRegs->msiUp32));
  }
  if (writeRegs->msiData) {
    pcie_check_result(retVal, pcie_write_msiData_reg (baseCfgEpRegs, writeRegs->msiData));
  }

  /*Capabilities Registers*/
  if (writeRegs->pciesCap) {
    pcie_check_result(retVal, pcie_write_pciesCap_reg (baseCfgEpRegs, writeRegs->pciesCap));
  }
  if (writeRegs->deviceCap) {
    pcie_check_result(retVal, pcie_write_deviceCap_reg (baseCfgEpRegs, writeRegs->deviceCap));
  }

  if (writeRegs->devStatCtrl) {
    pcie_check_result(retVal, pcie_write_devStatCtrl_reg (baseCfgEpRegs, writeRegs->devStatCtrl));
  }
  if (writeRegs->linkCap) {
    pcie_check_result(retVal, pcie_write_linkCap_reg (baseCfgEpRegs, writeRegs->linkCap));
  }
  if (writeRegs->linkStatCtrl) {
    pcie_check_result(retVal, pcie_write_linkStatCtrl_reg (baseCfgEpRegs, writeRegs->linkStatCtrl));
  }
  if (writeRegs->slotCap) {
    pcie_check_result(retVal, pcie_write_slotCap_reg (baseCfgRcRegs, writeRegs->slotCap));
  }
  if (writeRegs->slotStatCtrl) {
    pcie_check_result(retVal, pcie_write_slotStatCtrl_reg (baseCfgRcRegs, writeRegs->slotStatCtrl));
  }
  if (writeRegs->rootCtrlCap) {
    pcie_check_result(retVal, pcie_write_rootCtrlCap_reg (baseCfgRcRegs, writeRegs->rootCtrlCap));
  }
  if (writeRegs->rootStatus) {
    pcie_check_result(retVal, pcie_write_rootStatus_reg (baseCfgRcRegs, writeRegs->rootStatus));
  }
  if (writeRegs->devCap2) {
    pcie_check_result(retVal, pcie_write_devCap2_reg (baseCfgEpRegs, writeRegs->devCap2));
  }
  if (writeRegs->devStatCtrl2) {
    pcie_check_result(retVal, pcie_write_devStatCtrl2_reg (baseCfgEpRegs, writeRegs->devStatCtrl2));
  }
  if (writeRegs->linkCtrl2) {
    pcie_check_result(retVal, pcie_write_linkCtrl2_reg (baseCfgEpRegs, writeRegs->linkCtrl2));
  }

  /*Capabilities Extended Registers*/
  if (writeRegs->uncErr) {
    pcie_check_result(retVal, pcie_write_uncErr_reg (baseCfgEpRegs, writeRegs->uncErr));
  }
  if (writeRegs->uncErrMask) {
    pcie_check_result(retVal, pcie_write_uncErrMask_reg (baseCfgEpRegs, writeRegs->uncErrMask));
  }
  if (writeRegs->uncErrSvrty) {
    pcie_check_result(retVal, pcie_write_uncErrSvrty_reg (baseCfgEpRegs, writeRegs->uncErrSvrty));
  }
  if (writeRegs->corErr) {
    pcie_check_result(retVal, pcie_write_corErr_reg (baseCfgEpRegs, writeRegs->corErr));
  }
  if (writeRegs->corErrMask) {
    pcie_check_result(retVal, pcie_write_corErrMask_reg (baseCfgEpRegs, writeRegs->corErrMask));
  }
  if (writeRegs->accr) {
    pcie_check_result(retVal, pcie_write_accr_reg (baseCfgEpRegs, writeRegs->accr));
  }
  if (writeRegs->rootErrCmd) {
    pcie_check_result(retVal, pcie_write_rootErrCmd_reg (baseCfgEpRegs, writeRegs->rootErrCmd));
  }
  if (writeRegs->rootErrSt) {
    pcie_check_result(retVal, pcie_write_rootErrSt_reg (baseCfgEpRegs, writeRegs->rootErrSt));
  }

  /*Port Logic Registers*/
  if (writeRegs->plAckTimer) {
    pcie_check_result(retVal, pcie_write_plAckTimer_reg (baseCfgEpRegs, writeRegs->plAckTimer));
  }
  if (writeRegs->plOMsg) {
    pcie_check_result(retVal, pcie_write_plOMsg_reg (baseCfgEpRegs, writeRegs->plOMsg));
  }
  if (writeRegs->plForceLink) {
    pcie_check_result(retVal, pcie_write_plForceLink_reg (baseCfgEpRegs, writeRegs->plForceLink));
  }
  if (writeRegs->ackFreq) {
    pcie_check_result(retVal, pcie_write_ackFreq_reg (baseCfgEpRegs, writeRegs->ackFreq));
  }
  if (writeRegs->lnkCtrl) {
    pcie_check_result(retVal, pcie_write_lnkCtrl_reg (baseCfgEpRegs, writeRegs->lnkCtrl));
  }
  if (writeRegs->laneSkew) {
    pcie_check_result(retVal, pcie_write_laneSkew_reg (baseCfgEpRegs, writeRegs->laneSkew));
  }
  if (writeRegs->symNum) {
    pcie_check_result(retVal, pcie_write_symNum_reg (baseCfgEpRegs, writeRegs->symNum));
  }
  if (writeRegs->symTimerFltMask) {
    pcie_check_result(retVal, pcie_write_symTimerFltMask_reg (baseCfgEpRegs, writeRegs->symTimerFltMask));
  }
  if (writeRegs->fltMask2) {
    pcie_check_result(retVal, pcie_write_fltMask2_reg (baseCfgEpRegs, writeRegs->fltMask2));
  }
  if (writeRegs->gen2) {
    pcie_check_result(retVal, pcie_write_gen2_reg (baseCfgEpRegs, writeRegs->gen2));
  }

  return retVal;
} /* Pcie_writeRegs */


/*********************************************************************
 * FUNCTION PURPOSE: Configures the Outbound Offset registers 
 *                   for outbound address translation
 ********************************************************************/
pcieRet_e Pcie_cfgObOffset 
(
  Pcie_Handle      handle,   /**< [in] The PCIE LLD instance identifier */
  uint32_t         obAddrLo, /**< [in] Low  Outbound address offset (32bits) */
  uint32_t         obAddrHi, /**< [in] High Outbound address offset (32bits) */
  uint8_t          region    /**< [in] Identifies the Outbound region (0-31) */
)
{

  CSL_Pciess_appRegs *baseAppRegs = (CSL_Pciess_appRegs *)handle; 
  pcieRet_e           retVal      = pcie_RET_OK;
  pcieObOffsetLoReg_t obOffsetLo;
  pcieObOffsetHiReg_t obOffsetHi;
  uint16_t            obAddrLoField;

  pcie_check_handle(handle);
  
  memset (&obOffsetLo, 0, sizeof(obOffsetLo));
  memset (&obOffsetHi, 0, sizeof(obOffsetHi));

  pcie_getbits(obAddrLo, CSL_PCIESS_APP_OB_OFFSET_INDEX_OB_OFFSET_LO, obAddrLoField);
  
  obOffsetLo.enable = 1;
  obOffsetLo.offsetLo = obAddrLoField;

  obOffsetHi.offsetHi = obAddrHi;

  pcie_check_result(retVal, pcie_write_obOffsetLo_reg(baseAppRegs, &obOffsetLo, region));
  pcie_check_result(retVal, pcie_write_obOffsetHi_reg(baseAppRegs, &obOffsetHi, region));

  return retVal;
}


/*********************************************************************
 * FUNCTION PURPOSE: Configures the Inbound Translation registers 
 ********************************************************************/
pcieRet_e Pcie_cfgIbTrans 
(
  Pcie_Handle      handle,   /**< [in] The PCIE LLD instance identifier */
  pcieIbTransCfg_t *ibCfg    /**< [in] Inbound Translation Configuration parameters */
)
{

  CSL_Pciess_appRegs *baseAppRegs = (CSL_Pciess_appRegs *)handle; 
  pcieRet_e           retVal      = pcie_RET_OK;

  pcieIbBarReg_t      ibBar;
  pcieIbStartLoReg_t  ibStartLo;
  pcieIbStartHiReg_t  ibStartHi;
  pcieIbOffsetReg_t   ibOffset;
  
  uint32_t            ibStartLoField;
  uint32_t            ibOffsetField;

  pcie_check_handle(handle);
  
  memset (&ibBar,     0, sizeof(ibBar));
  memset (&ibStartLo, 0, sizeof(ibStartLo));
  memset (&ibStartHi, 0, sizeof(ibStartHi));
  memset (&ibOffset,  0, sizeof(ibOffset));

  ibBar.ibBar = ibCfg->ibBar;
  
  pcie_getbits(ibCfg->ibStartAddrLo, CSL_PCIESS_APP_IB_START_LO_IB_START_LO, ibStartLoField);
  ibStartLo.ibStartLo = ibStartLoField;

  ibStartHi.ibStartHi = ibCfg->ibStartAddrHi;

  pcie_getbits(ibCfg->ibOffsetAddr, CSL_PCIESS_APP_IB_OFFSET_IB_OFFSET, ibOffsetField);
  ibOffset.ibOffset = ibOffsetField;
  

  pcie_check_result(retVal, pcie_write_ibBar_reg    (baseAppRegs, &ibBar,     ibCfg->region));
  pcie_check_result(retVal, pcie_write_ibStartLo_reg(baseAppRegs, &ibStartLo, ibCfg->region));
  pcie_check_result(retVal, pcie_write_ibStartHi_reg(baseAppRegs, &ibStartHi, ibCfg->region));
  pcie_check_result(retVal, pcie_write_ibOffset_reg (baseAppRegs, &ibOffset,  ibCfg->region));

  return retVal;
}


/*********************************************************************
 * FUNCTION PURPOSE: Configures a BAR Register (32bits)
 ********************************************************************/
pcieRet_e Pcie_cfgBar 
(
  Pcie_Handle      handle,   /**< [in] The PCIE LLD instance identifier */
  pcieBarCfg_t    *barCfg    /**< [in] BAR configuration parameters */
)
{
  pcieType0BarIdx_t  type0BarIdx;  
  pcieType1BarIdx_t  type1BarIdx;  
  pcieRegisters_t    setRegs;
  uint32_t           barAddrField = 0;
  pcieRet_e          retVal = pcie_RET_OK;

  memset (&setRegs,     0, sizeof(setRegs));
  memset (&type0BarIdx, 0, sizeof(type0BarIdx));
  memset (&type1BarIdx, 0, sizeof(type1BarIdx));

  if(barCfg->mode == pcie_RC_MODE)
  {
    pcie_getbits(barCfg->base, CSL_PCIE_CFG_SPACE_ROOTCOMPLEX_BAR_BASE_ADDRESS, barAddrField);

    type1BarIdx.reg.base     = barAddrField;
    type1BarIdx.reg.prefetch = barCfg->prefetch;
    type1BarIdx.reg.type     = barCfg->type;
    type1BarIdx.reg.memSpace = barCfg->memSpace;
    type1BarIdx.idx          = barCfg->idx;

    setRegs.type1BarIdx = &type1BarIdx;   
  }
  else
  {
    pcie_getbits(barCfg->base, CSL_PCIE_CFG_SPACE_ENDPOINT_BAR_BASE_ADDRESS, barAddrField);

    type0BarIdx.reg.base     = barAddrField;
    type0BarIdx.reg.prefetch = barCfg->prefetch;
    type0BarIdx.reg.type     = barCfg->type;
    type0BarIdx.reg.memSpace = barCfg->memSpace;
    type0BarIdx.idx          = barCfg->idx;

    setRegs.type0BarIdx = &type0BarIdx;   
  }

  if ((retVal = Pcie_writeRegs (handle, barCfg->location, &setRegs)) != pcie_RET_OK) 
  {
    return retVal;
  }

  return retVal;
}


/*********************************************************************
 * FUNCTION PURPOSE: Returns version number
 ********************************************************************/
uint32_t Pcie_getVersion 
(
  void
)
{
  return pcie_LLD_VERSION_ID;
} /* Pcie_getVersion */

/*********************************************************************
 * FUNCTION PURPOSE: Returns version string
 ********************************************************************/
const char* Pcie_getVersionStr
(
  void
)
{
  return PCIELLDVersionStr;
} /* Pcie_getVersionStr */

/* Nothing past this point */

