/**  
 *  (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include "fw_test.h"
#include "fw_mem_allocator.h"
#include "pautest.h"

#include <ti/drv/qmss/qmss_qm.h>
#include <ti/drv/qmss/device/qmss_device.c>
#include <ti/drv/cppi/device/cppi_device.c>

/* QMSS device specific configuration */
extern Qmss_GlobalConfigParams qmssGblCfgParams[];

/* CPPI device specific configuration */
extern Cppi_GlobalConfigParams  cppiGblCfgParams[];

/* test framework instance */
tFramework_t tFramework;

/* Host Descriptor Region - [Size of descriptor * Number of descriptors] 
 * MUST be 16 byte aligned.
 */
uint8_t *pHostDesc = 0;

/* Static memory allocation for test framework */
uint8_t memPaInst[TF_PA_INST_SIZE]ALIGN(CACHE_LINESZ);
uint8_t memL2Ram[TF_L2_TABLE_SIZE]ALIGN(CACHE_LINESZ);
uint8_t memL3Ram[TF_L3_TABLE_SIZE]ALIGN(CACHE_LINESZ);
uint8_t memUsrStatsLnkTbl[TF_USR_STATS_LNK_TABLE_SIZE]ALIGN(CACHE_LINESZ);


/******************************************************************************
* Macro to convert to IP Register Virtual Address from a mapped base Virtual Address
* Input: virtBaseAddr: Virtual base address mapped using mmap for IP
*        phyBaseAddr: Physical base address for the IP
*        phyRegAddr:  Physical register address
******************************************************************************/
static inline void* FW_GET_REG_VADDR (void * virtBaseAddr, uint32_t phyBaseAddr, uint32_t phyRegAddr)
{
    return((void *)((uint8_t *)virtBaseAddr + (phyRegAddr - phyBaseAddr)));
}

/** ============================================================================
 *   @n@b initQmss
 *
 *   @b Description
 *   @n This API initializes the QMSS LLD on core 0 only.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t initQmss (void)
{
    int32_t                     result;
    Qmss_MemRegInfo             memCfg;
    Qmss_InitCfg                qmssInitConfig;
    Cppi_DescCfg                cppiDescCfg;
    uint32_t                    numAllocated;
    Qmss_GlobalConfigParams     fw_qmssGblCfgParams;

    /* Initialize QMSS */
    memset (&qmssInitConfig, 0, sizeof (Qmss_InitCfg));

    /* Set up QMSS configuration */

    /* Use internal linking RAM */
    qmssInitConfig.linkingRAM0Base  =   0;   
    qmssInitConfig.linkingRAM0Size  =   0;
    qmssInitConfig.linkingRAM1Base  =   0x0;
    qmssInitConfig.maxDescNum       =   TF_NUM_DESC;

    /* Bypass hardware initialization as it is done within Kernel */
    qmssInitConfig.qmssHwStatus     =   QMSS_HW_INIT_COMPLETE;

    fw_qmssGblCfgParams = qmssGblCfgParams[0];

    /* Convert address to Virtual address */ 
    fw_qmssGblCfgParams.qmConfigReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CONFIG_STARVATION_COUNTER_REGS);

    fw_qmssGblCfgParams.qmDescReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_DESCRIPTION_REGS);

    fw_qmssGblCfgParams.qmQueMgmtReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueMgmtProxyReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_PROXY_QUEUE_DEQUEUE_REGS);

    fw_qmssGblCfgParams.qmQueStatReg = 
         FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QUE_PEEK_REGS);
    
    fw_qmssGblCfgParams.qmQueIntdReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_INTD_REGS);

    fw_qmssGblCfgParams.qmPdspCmdReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM1_REGS);
  
    fw_qmssGblCfgParams.qmPdspCmdReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_SCRACH_RAM2_REGS);
        
    fw_qmssGblCfgParams.qmPdspCtrlReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP1_REGS);

    fw_qmssGblCfgParams.qmPdspCtrlReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_ADSP2_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP1_RAM_REGS);

    fw_qmssGblCfgParams.qmPdspIRamReg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_APDSP2_RAM_REGS);

    fw_qmssGblCfgParams.qmStatusRAM = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_QM_STATUS_RAM_REGS);

    fw_qmssGblCfgParams.qmLinkingRAMReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_LINKING_RAM_REGS);

    fw_qmssGblCfgParams.qmMcDMAReg = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_MCDMA_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[0] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER1_REGS);

    fw_qmssGblCfgParams.qmTimer16Reg[1] = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_TIMER2_REGS);

    fw_qmssGblCfgParams.qmQueMgmtDataReg = (void *)((uint32_t)fw_qmssDataVaddr);

    fw_qmssGblCfgParams.qmQueMgmtProxyDataReg = 
        FW_GET_REG_VADDR(fw_qmssDataVaddr,QMSS_DATA_BASE_ADDR,QMSS_DATA_BASE_QUEUE_PROXY_ADDR);

    /* Initialize the Queue Manager */
    result = Qmss_init (&qmssInitConfig, &fw_qmssGblCfgParams);
    if (result != QMSS_SOK)
    {
        printf ("initQmss: Error initializing Queue Manager SubSystem, Error code : %d\n", result);
        return -1;
    }

    /* Start Queue manager on this core */
    Qmss_start ();

    /* Setup the descriptor memory regions. 
     *
     * The Descriptor base addresses MUST be global addresses and
     * all memory regions MUST be setup in ascending order of the
     * descriptor base addresses.
     */
    pHostDesc = (uint8_t*)fw_memAlloc((TF_NUM_DESC * TF_SIZE_DESC), CACHE_LINESZ);

    /* Initialize and setup CPSW Host Descriptors required for example */
    memset (pHostDesc, 0, TF_SIZE_DESC * TF_NUM_DESC);
    memCfg.descBase             =   (uint32_t *) pHostDesc;
    memCfg.descSize             =   TF_SIZE_DESC;
    memCfg.descNum              =   TF_NUM_DESC;
    memCfg.manageDescFlag       =   Qmss_ManageDesc_MANAGE_DESCRIPTOR;
    memCfg.memRegion            =   Qmss_MemRegion_MEMORY_REGION0;
    memCfg.startIndex           =   0;

    /* Insert Host Descriptor memory region */
    result = Qmss_insertMemoryRegion(&memCfg);
    if (result == QMSS_MEMREGION_ALREADY_INITIALIZED)
    {
        printf ("initQmss: Memory Region %d already Initialized \n", memCfg.memRegion);
    }
    else if (result < QMSS_SOK)
    {
        printf ("initQmss: Error Inserting memory region %d, Error code : %d\n", memCfg.memRegion, result);
        return -1;
    }    

    /* Initialize all the descriptors we just allocated on the
     * memory region above. Setup the descriptors with some well
     * known values before we use them for data transfers.
     */
    memset (&cppiDescCfg, 0, sizeof (cppiDescCfg));
    cppiDescCfg.memRegion       =   Qmss_MemRegion_MEMORY_REGION0;
    cppiDescCfg.descNum         =   TF_NUM_DESC;
    cppiDescCfg.destQueueNum    =   TF_Q_FREE_DESC;
    cppiDescCfg.queueType       =   Qmss_QueueType_GENERAL_PURPOSE_QUEUE;
    cppiDescCfg.initDesc        =   Cppi_InitDesc_INIT_DESCRIPTOR;
    cppiDescCfg.descType        =   Cppi_DescType_HOST;
    
    /* By default:
     *      (1) Return descriptors to tail of queue 
     *      (2) Always return entire packet to this free queue
     *      (3) Set that PS Data is always present in start of SOP buffer
     *      (4) Configure free q num < 4K, hence qMgr = 0
     *      (5) Recycle back to the same Free queue by default.
     */
    cppiDescCfg.returnPushPolicy            =   Qmss_Location_TAIL;    
    cppiDescCfg.cfg.host.returnPolicy       =   Cppi_ReturnPolicy_RETURN_ENTIRE_PACKET;    
    cppiDescCfg.cfg.host.psLocation         =   Cppi_PSLoc_PS_IN_DESC;         
    cppiDescCfg.returnQueue.qMgr            =   0;    
    cppiDescCfg.returnQueue.qNum            =   QMSS_PARAM_NOT_SPECIFIED; 
    cppiDescCfg.epibPresent                 =   Cppi_EPIB_EPIB_PRESENT;
    
    /* Initialize the descriptors, create a free queue and push descriptors to a global free queue */
    if ((tFramework.QfreeDesc = Cppi_initDescriptor (&cppiDescCfg, &numAllocated)) <= 0)
    {
        printf ("initQmss: Error Initializing Free Descriptors, Error: %d \n", tFramework.QfreeDesc);
        return -1;
    }
    else
    {
        printf ("initQmss: Initialized Free Descriptors. \n");
    }        
   
    /* Queue Manager Initialization Done */
    return 0;
}

/** ============================================================================
 *   @n@b initCppi
 *
 *   @b Description
 *   @n This API initializes the CPPI LLD, opens the PASS CPDMA and opens up
 *      the Tx, Rx channels required for data transfers.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
int32_t initCppi (void)
{
    int32_t                     result, i;        
    Cppi_CpDmaInitCfg           cpdmaCfg;
    uint8_t                     isAllocated;        
    Cppi_TxChInitCfg            txChCfg;
    Cppi_RxChInitCfg            rxChInitCfg;
    Cppi_GlobalConfigParams     fw_cppiGblCfgParams[CPPI_MAX_CPDMA];

  for (i=0; i<CPPI_MAX_CPDMA; i++)
    fw_cppiGblCfgParams[i] = cppiGblCfgParams[i];


  /* Convert Physical address to Virtual address for LLD access */
  /* SRIO CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_SRIO_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_srioCfgVaddr,CSL_SRIO_CONFIG_REGS,CSL_SRIO_CONFIG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* PASS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_PASS_CPDMA].rxFlowRegs =
        FW_GET_REG_VADDR(fw_passCfgVaddr,CSL_PA_SS_CFG_REGS,CSL_PA_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);

  /* QMSS CPDMA regs */
  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].gblCfgRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_GLOBAL_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxChRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].txSchedRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_TX_SCHEDULER_CFG_REGS);

  fw_cppiGblCfgParams[Cppi_CpDma_QMSS_CPDMA].rxFlowRegs = 
        FW_GET_REG_VADDR(fw_qmssCfgVaddr,CSL_QM_SS_CFG_QUE_PEEK_REGS,CSL_QM_SS_CFG_CPPI_DMA_RX_FLOW_CFG_REGS);


    /* Initialize CPPI LLD */
    result = Cppi_init (fw_cppiGblCfgParams);
    if (result != CPPI_SOK)
    {
        printf ("initCppi: Error initializing CPPI LLD, Error code : %d\n", result);
        return -1;
    }

    /* Initialize PASS CPDMA */
    memset (&cpdmaCfg, 0, sizeof (Cppi_CpDmaInitCfg));
    cpdmaCfg.dmaNum     = Cppi_CpDma_PASS_CPDMA;
    if ((tFramework.tfPaCppiHandle = Cppi_open (&cpdmaCfg)) == NULL)
    {
        printf ("initCppi: Error initializing CPPI for PASS CPDMA %d \n", cpdmaCfg.dmaNum);
        return -1;
    }

    /* Open all CPPI Tx Channels. These will be used to send data to PASS/CPSW */             
    for (i = 0; i < TF_PA_NUM_TX_CPDMA_CHANNELS; i ++)
    {
        txChCfg.channelNum      =   i;       /* CPPI channels are mapped one-one to the PA Tx queues */
        txChCfg.txEnable        =   Cppi_ChState_CHANNEL_DISABLE;  /* Disable the channel for now. */
        txChCfg.filterEPIB      =   0;
        txChCfg.filterPS        =   0;
        txChCfg.aifMonoMode     =   0;
        txChCfg.priority        =   2;
        if ((tFramework.tfPaTxChHnd[i] = Cppi_txChannelOpen (tFramework.tfPaCppiHandle, &txChCfg, &isAllocated)) == NULL)
        {
            printf ("initCppi: Error opening Tx channel %d\n", txChCfg.channelNum);
            return -1;
        }
        Cppi_channelEnable (tFramework.tfPaTxChHnd[i]);
    }

    /* Open all CPPI Rx channels. These will be used by PA to stream data out. */
    for (i = 0; i < TF_PA_NUM_RX_CPDMA_CHANNELS; i++)
    {
        /* Open a CPPI Rx channel that will be used by PA to stream data out. */
        rxChInitCfg.channelNum  =   i; 
        rxChInitCfg.rxEnable    =   Cppi_ChState_CHANNEL_DISABLE; 
        if ((tFramework.tfPaRxChHnd[i] = Cppi_rxChannelOpen (tFramework.tfPaCppiHandle, &rxChInitCfg, &isAllocated)) == NULL)
        {
            printf ("initCppi: Error opening Rx channel: %d \n", rxChInitCfg.channelNum);
            return -1;
        }

        /* Also enable Rx Channel */
        Cppi_channelEnable (tFramework.tfPaRxChHnd[i]);    
    }
    
    /* Clear CPPI Loobpack bit in PASS CDMA Global Emulation Control Register */
    Cppi_setCpdmaLoopback(tFramework.tfPaCppiHandle, 0);   

    /* CPPI Init Done. Return success */
    return 0;
}    

/* Setup all the queues used in the example */
int32_t setupQueues (void)
{
    uint8_t         isAlloc;        
    Qmss_Queue      q;
    Cppi_HostDesc   *hd;
    uint8_t         *bPtr;
    int             i;

    /* The 10 PA transmit queues (corresponding to the 10 tx cdma channels */
    for (i = 0; i < TF_PA_NUM_TX_CPDMA_CHANNELS; i ++)
    {            
        if ((tFramework.QPaTx[i] = Qmss_queueOpen (Qmss_QueueType_PASS_QUEUE, QMSS_PARAM_NOT_SPECIFIED, &isAlloc)) < 0)
        {
            printf ("setupQueues: Qmss_queueOpen failed for PA transmit queue number %d\n", 640+i);
            return -1;
        }            
        Qmss_setQueueThreshold (tFramework.QPaTx[i], 1, 1);
    }
  
    /* The queues with attached buffers */
    tFramework.QLinkedBuf1 = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_LINKED_BUF_Q1, &isAlloc);
    if (tFramework.QLinkedBuf1 < 0)  {
  	    printf ("setupQueues: Qmss_queueOpen failed for queue %d\n", TF_LINKED_BUF_Q1);
  	    return (-1);
    }
  
    tFramework.QLinkedBuf2 = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_LINKED_BUF_Q2, &isAlloc);
    if (tFramework.QLinkedBuf2 < 0)  {
  	    printf ("SetupQueues: Qmss_queueOpen failed for queue %d\n", TF_LINKED_BUF_Q2);
  	    return (-1);
    }
  
    tFramework.QLinkedBuf3 = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_LINKED_BUF_Q3, &isAlloc);
    if (tFramework.QLinkedBuf3 < 0)  {
  	    printf ("SetupQueues: Qmss_queueOpen failed for queue %d\n", TF_LINKED_BUF_Q3);
  	    return (-1);
    }
  
    /* Attach buffers to the queues and push them onto the queue */
    q.qMgr = 0;
    q.qNum = TF_DEF_RET_Q;

    /* Allocate memory for the Packet buffers */
    bPtr = (uint8_t *) fw_memAlloc((TF_LINKED_BUF_Q1_NBUFS * TF_LINKED_BUF_Q1_BUF_SIZE), CACHE_LINESZ);
    if(bPtr == NULL) {
  	    printf ("SetupQueues: memAlloc failed for memQ1\n");
  	    return (-1);
    }

    for (i = 0; i < TF_LINKED_BUF_Q1_NBUFS; i++)  {
  	
        hd = (Cppi_HostDesc *)((uint32_t)Qmss_queuePop (tFramework.QfreeDesc));

        /* The descriptor address returned from the hardware has the 
         * descriptor size appended to the address in the last 4 bits.
         *
         * To get the true descriptor size, always mask off the last 
         * 4 bits of the address.
         */
        hd = (Cppi_HostDesc *)((uint32_t)hd & 0xFFFFFFF0);

        if (hd == NULL)  {
            printf ("setupQueues: Qmss_queuePop returned NULL on pop from queue number %d\n", tFramework.QfreeDesc);
            return (-1);
        }

        /* Populate the free descriptor with the buffer. */
        Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                      TF_LINKED_BUF_Q1_BUF_SIZE);
        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                                 TF_LINKED_BUF_Q1_BUF_SIZE);

        hd->nextBDPtr = (uint32_t)NULL;
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
        Qmss_queuePushDesc (tFramework.QLinkedBuf1, (Ptr)hd);
        bPtr += TF_LINKED_BUF_Q1_BUF_SIZE;
    }
  
    bPtr = (uint8_t *) fw_memAlloc((TF_LINKED_BUF_Q2_NBUFS * TF_LINKED_BUF_Q2_BUF_SIZE), CACHE_LINESZ);
    if(bPtr == NULL) {
  	    printf ("SetupQueues: memAlloc failed for memQ2\n");
  	    return (-1);
    }

    for (i = 0; i < TF_LINKED_BUF_Q2_NBUFS; i++)   {
  	
        hd = (Cppi_HostDesc *)((uint32_t)Qmss_queuePop (tFramework.QfreeDesc));

        /* The descriptor address returned from the hardware has the 
         * descriptor size appended to the address in the last 4 bits.
         *
         * To get the true descriptor size, always mask off the last 
         * 4 bits of the address.
         */
        hd = (Cppi_HostDesc *)((uint32_t)hd & 0xFFFFFFF0);

        if (hd == NULL)  {
            printf ("setupQueues: Qmss_queuePop returned NULL on pop from queue number %d\n", tFramework.QfreeDesc);
            return (-1);
        }

        /* Populate the free descriptor with the buffer. */
        Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                      TF_LINKED_BUF_Q2_BUF_SIZE);
        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                                 TF_LINKED_BUF_Q2_BUF_SIZE);

        hd->nextBDPtr = (uint32_t)NULL;
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
        Qmss_queuePushDesc (tFramework.QLinkedBuf2, (Ptr)hd);
        bPtr += TF_LINKED_BUF_Q2_BUF_SIZE;
    }
  
    bPtr = (uint8_t *) fw_memAlloc((TF_LINKED_BUF_Q3_NBUFS * TF_LINKED_BUF_Q3_BUF_SIZE), CACHE_LINESZ);
    if(bPtr == NULL) {
  	    printf ("SetupQueues: memAlloc failed for memQ3\n");
  	    return (-1);
    }

    for (i = 0; i < TF_LINKED_BUF_Q3_NBUFS; i++)   {
  	
        hd = (Cppi_HostDesc *)((uint32_t)Qmss_queuePop (tFramework.QfreeDesc));

        /* The descriptor address returned from the hardware has the 
         * descriptor size appended to the address in the last 4 bits.
         *
         * To get the true descriptor size, always mask off the last 
         * 4 bits of the address.
         */
        hd = (Cppi_HostDesc *)((uint32_t)hd & 0xFFFFFFF0);

        if (hd == NULL)  {
            printf ("setupQueues: Qmss_queuePop returned NULL on pop from queue number %d\n", tFramework.QfreeDesc);
            return (-1);
        }

        /* Populate the free descriptor with the buffer. */
        Cppi_setData (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                      TF_LINKED_BUF_Q3_BUF_SIZE);
        /* Save original buffer information */
        Cppi_setOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, bPtr,
                                 TF_LINKED_BUF_Q3_BUF_SIZE);

        hd->nextBDPtr = (uint32_t)NULL;
        Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
        Qmss_queuePushDesc (tFramework.QLinkedBuf3, (Ptr)hd);
        bPtr += TF_LINKED_BUF_Q3_BUF_SIZE;
    }
  
    /* The default return queue for descriptors with linked buffers */
    tFramework.QDefRet = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_DEF_RET_Q, &isAlloc);
    if (tFramework.QDefRet < 0)  {
  	    printf ("SetupQueues: Qmss_queueOpen failed for queue %d\n", TF_DEF_RET_Q);
  	    return (-1);
    }
  
    tFramework.QCommonCmdRep = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_COMMON_CMD_REPL_Q, &isAlloc);
    if (tFramework.QCommonCmdRep < 0)  {
  	    printf ("SetupQueues: Qmss_queueOpen failed for queue %d\n", TF_COMMON_CMD_REPL_Q);
  	    return (-1);
    }
    
    /* General purpose queues */
    for (i = 0; i < TF_NUM_GEN_QUEUES; i++)  {
  	
  	    tFramework.QGen[i] = Qmss_queueOpen (Qmss_QueueType_GENERAL_PURPOSE_QUEUE, TF_FIRST_GEN_QUEUE + i, &isAlloc);

 	    if (tFramework.QGen[i] < 0)  {
  	        printf ("SetupQueues: Qmss_queueOpen failed for queue %d\n", TF_FIRST_GEN_QUEUE + i);
  	        return (-1);
  	    }
    }

    /* All done return success. */
    return 0;
}

/* Configure flows */
int setupFlows (void)
{
  Cppi_RxFlowCfg  rxFlowCfg;
  uint8_t         isAlloc;
  int             i;

  /* Configure Rx flow */
  rxFlowCfg.flowIdNum      = tFramework.tfFlowNum = 0;
  rxFlowCfg.rx_dest_qnum   = TF_FIRST_GEN_QUEUE + TF_NUM_GEN_QUEUES -1;   /* Override in PA */
  rxFlowCfg.rx_dest_qmgr   = 0;
  rxFlowCfg.rx_sop_offset  = 0;
  rxFlowCfg.rx_ps_location = Cppi_PSLoc_PS_IN_DESC;
  rxFlowCfg.rx_desc_type   = Cppi_DescType_HOST;
  rxFlowCfg.rx_error_handling = 1;
  rxFlowCfg.rx_psinfo_present = 1;
  rxFlowCfg.rx_einfo_present  = 1;

  rxFlowCfg.rx_dest_tag_lo = 0;
  rxFlowCfg.rx_dest_tag_hi = 0;
  rxFlowCfg.rx_src_tag_lo  = 0;
  rxFlowCfg.rx_src_tag_hi  = 0;

  rxFlowCfg.rx_size_thresh0_en = 1;  
  rxFlowCfg.rx_size_thresh1_en = 1;  
  rxFlowCfg.rx_size_thresh2_en = 0;  
  rxFlowCfg.rx_dest_tag_lo_sel = 0;
  rxFlowCfg.rx_dest_tag_hi_sel = 0;
  rxFlowCfg.rx_src_tag_lo_sel  = 0;
  rxFlowCfg.rx_src_tag_hi_sel  = 0;

  rxFlowCfg.rx_fdq1_qnum = tFramework.QLinkedBuf2;
  rxFlowCfg.rx_fdq1_qmgr = 0;
  rxFlowCfg.rx_fdq0_sz0_qnum = tFramework.QLinkedBuf1;
  rxFlowCfg.rx_fdq0_sz0_qmgr = 0;

  rxFlowCfg.rx_fdq3_qnum = tFramework.QLinkedBuf2;
  rxFlowCfg.rx_fdq3_qmgr = 0;
  rxFlowCfg.rx_fdq2_qnum = tFramework.QLinkedBuf2;
  rxFlowCfg.rx_fdq2_qmgr = 0;

  rxFlowCfg.rx_size_thresh1 = TF_LINKED_BUF_Q2_BUF_SIZE;
  rxFlowCfg.rx_size_thresh0 = TF_LINKED_BUF_Q1_BUF_SIZE;
    
  rxFlowCfg.rx_fdq0_sz1_qnum = tFramework.QLinkedBuf2;
  rxFlowCfg.rx_fdq0_sz1_qmgr = 0;
  rxFlowCfg.rx_size_thresh2  = 0;

  rxFlowCfg.rx_fdq0_sz3_qnum = tFramework.QLinkedBuf3;
  rxFlowCfg.rx_fdq0_sz3_qmgr = 0;
  rxFlowCfg.rx_fdq0_sz2_qnum = tFramework.QLinkedBuf3;
  rxFlowCfg.rx_fdq0_sz2_qmgr = 0;

  tFramework.tfPaFlowHnd0 = Cppi_configureRxFlow (tFramework.tfPaCppiHandle, &rxFlowCfg, &isAlloc);
  if (tFramework.tfPaFlowHnd0 == NULL)  {
    printf ("setupFlows: cppi_ConfigureRxFlow returned NULL on flow 0\n");
    return (-1);
  }
  
  return (0);
}

/* The QM/CPDMA are setup */
int initQm (void)
{
  if (initQmss())  {
  	printf ("initQm: initQmss failed\n");
    return (-1);
  }

  if (initCppi ())  {
  	printf ("initQm: initCppi failed\n");
    return (-1);
  }
  
  if (setupQueues ())  {
  	printf ("initQm: setupQueues failed\n");
    return (-1);
  }

  if (setupFlows ())  {
  	printf ("initQm: setupFlows failed\n");
    return (-1);
  }

  return (0);
}

/* The PA LLD instance is created, the PA firmware is
 * downloaded and started by the Linux Kernel*/
int initPa (void)
{
  paSizeInfo_t  paSize;
  paConfig_t    paCfg;
  paTimestampConfig_t tsCfg;
  int           ret;
  int           sizes[pa_N_BUFS];
  int           aligns[pa_N_BUFS];
  void*         bases[pa_N_BUFS];
  

  memset(&paSize, 0, sizeof(paSizeInfo_t));
  memset(&paCfg, 0, sizeof(paConfig_t));
  
  /* The maximum number of handles that can exists are 32 for L2, and 64 for L3. */
  paSize.nMaxL2 = TF_MAX_NUM_L2_HANDLES;
  paSize.nMaxL3 = TF_MAX_NUM_L3_HANDLES;
  paSize.nUsrStats = pa_USR_STATS_MAX_COUNTERS;
  
  ret = Pa_getBufferReq(&paSize, sizes, aligns);

  if (ret != pa_OK)  {
    printf ("initPa: Pa_getBufferReq() return with error code %d\n", ret);
    return (-1);
  }

  /* The first buffer is used as the instance buffer */
  if ((uint32_t)memPaInst & (aligns[0] - 1))  {
    printf ("initPa: Pa_getBufferReq requires %d alignment for instance buffer, but address is 0x%08x\n", aligns[0], (uint32_t)memPaInst);
    return (-1);
  }

  if (sizeof(memPaInst) < sizes[0])  {
    printf ("initPa: Pa_getBufferReq requires size %d for instance buffer, have only %d\n", sizes[0], sizeof(memPaInst));
    return (-1);
  }

  bases[0] = (void *)memPaInst;


  /* The second buffer is the L2 table */
  if ((uint32_t)memL2Ram & (aligns[1] - 1))  {
    printf ("initPa: Pa_getBufferReq requires %d alignment for buffer 1, but address is 0x%08x\n", aligns[1], (uint32_t)memL2Ram);
    return (-1);
  }

  if (sizeof(memL2Ram) <  sizes[1])  {
    printf ("initPa: Pa_getBufferReq requires %d bytes for buffer 1, have only %d\n", sizes[1], sizeof(memL2Ram));
    return (-1);
  }

  bases[1] = (void *)memL2Ram;

  /* The third buffer is the L3 table */
  if ((uint32_t)memL3Ram & (aligns[2] - 1))  {
    printf ("initPa: Pa_getBufferReq requires %d alignment for buffer 2, but address is 0x%08x\n", aligns[2], (uint32_t)memL3Ram);
    return (-1);
  }

  if (sizeof(memL3Ram) <  sizes[2])  {
    printf ("initPa: Pa_getBufferReq requires %d bytes for buffer 2, have only %d\n", sizes[2], sizeof(memL3Ram));
    return (-1);
  }

  bases[2] = (void *)memL3Ram;
  
  /* The fourth buffer is the User Statistics Link table */
  if ((uint32_t)memUsrStatsLnkTbl & (aligns[3] - 1))  {
    printf ("initPa: Pa_getBufferReq requires %d alignment for buffer 3, but address is 0x%08x\n", aligns[3], (uint32_t)memUsrStatsLnkTbl);
    return (-1);
  }

  if (sizeof(memUsrStatsLnkTbl) <  sizes[3])  {
    printf ("initPa: Pa_getBufferReq requires %d bytes for buffer 3, have only %d\n", sizes[2], sizeof(memUsrStatsLnkTbl));
    return (-1);
  }

  bases[3] = (void *)memUsrStatsLnkTbl;
  
  paCfg.initTable = TRUE;
  paCfg.initDefaultRoute = TRUE;
  paCfg.baseAddr = (uint32_t)fw_passCfgVaddr;
  paCfg.sizeCfg = &paSize;
  
  ret = Pa_create (&paCfg, bases, &tFramework.passHandle);
  if (ret != pa_OK)  {
    printf ("initPa: Pa_create returned with error code %d\n", ret);
    return (-1);
  }

  /* Enable Timer for timestamp */  
  memset(&tsCfg, 0, sizeof(paTimestampConfig_t));
  tsCfg.enable = TRUE;
  tsCfg.factor = pa_TIMESTAMP_SCALER_FACTOR_2;
  
  if(Pa_configTimestamp(tFramework.passHandle, &tsCfg) != pa_OK)
    return (-1);  
 
   return (0);
}


/* Initialize the test framework */
int setupTestFramework (void)
{
    
	/* Create the PA driver instance */
	if (initPa())  {
		printf ("setupTestFramework: initPa returned error, exiting\n");
		return (-1);
	}
	
	/* Setup the QM with associated buffers and descriptors */
	if (initQm())  {
		printf ("setupTestFramework: initQm returned error, exiting\n");
		return (-1);
	}
	
	return (0);
}

/* Check that all the queues are setup correctly */
int verifyTestFramework (void)
{ 
	int i, j;
	int count;
	int returnVal = 0;
	Cppi_HostDesc *hd;
	uint8_t *bufp;
	uint32_t bufLen;
    Qmss_Queue q;
	
	int32_t linkedQ[3];
	int32_t nbufs[]  = { TF_LINKED_BUF_Q1_NBUFS,    TF_LINKED_BUF_Q2_NBUFS,    TF_LINKED_BUF_Q3_NBUFS };
	int32_t bSize[]  = { TF_LINKED_BUF_Q1_BUF_SIZE, TF_LINKED_BUF_Q2_BUF_SIZE, TF_LINKED_BUF_Q3_BUF_SIZE };
	
	linkedQ[0] = tFramework.QLinkedBuf1;
	linkedQ[1] = tFramework.QLinkedBuf2;    
	linkedQ[2] = tFramework.QLinkedBuf3; 
	
	/* Verify that all of the general purpose queues are empty */
	for (i = 0; i < TF_NUM_GEN_QUEUES; i++)  {
		if ((count = Qmss_getQueueEntryCount (tFramework.QGen[i])) != 0)  {
			printf ("verifyTestFramework: Expected 0 entry count for queue %d, found %d entries\n", tFramework.QGen[i], count);
			returnVal = -1;
		}
	}
	
	/* Verify that the number of descriptors in the free descriptor queue is correct */
	count = Qmss_getQueueEntryCount (tFramework.QfreeDesc);
	if (count != (TF_NUM_DESC - TF_LINKED_BUF_Q1_NBUFS - TF_LINKED_BUF_Q2_NBUFS - TF_LINKED_BUF_Q3_NBUFS))  {
		printf ("verifyTestFramework: Expected %d entry count in the free descriptor queue (%d), found %d\n",
				TF_NUM_DESC - TF_LINKED_BUF_Q1_NBUFS - TF_LINKED_BUF_Q2_NBUFS - TF_LINKED_BUF_Q3_NBUFS,
				tFramework.QfreeDesc, count);
		returnVal = -1;
	}

    q.qMgr = 0;
    q.qNum = TF_DEF_RET_Q;
	
	/* Verify the number and sizing of descriptors with linked buffers in the three queues */
	for (j = 0; j < 3; j++)  {
		
		count = Qmss_getQueueEntryCount (linkedQ[j]);
		if (count != nbufs[j])  {
		    printf ("verifyTestFramework: Expected %d entry count in linked buffer queue 1 (%d), found %d\n",
				    nbufs[j], linkedQ[j], count);
		    returnVal = -1;
		}
	
		for (i = 0; i < count; i++)  {
			hd = (Cppi_HostDesc *)(((uint32_t)Qmss_queuePop (linkedQ[j])) & ~15);
			Cppi_getOriginalBufInfo (Cppi_DescType_HOST, (Cppi_Desc *)hd, &bufp, &bufLen);
            Cppi_setReturnQueue (Cppi_DescType_HOST, (Cppi_Desc *)hd, q);
			Qmss_queuePush (linkedQ[j], (Ptr)hd, hd->buffLen, TF_SIZE_DESC, Qmss_Location_TAIL);
				
			if (bufLen != bSize[j])  {
				printf ("verifyTestFramework: Linked buffer queue %d (%d) expected orignal length of %d, found %d\n",
						j, linkedQ[j], bSize[j], bufLen);
				returnVal = -1;
				break;
			}
		}
	}
	
	return (returnVal);  
}


