/******************************************************************************
 * FILE PURPOSE:  Main function routine for Example
 ******************************************************************************
 * FILE NAME:   fw_main.c
 *
 * @brief 
 *  Example to illustrate the usage of PA/QMSS/CPPI from User Land Linux environment 
 *  on ARM
 *
 *  This example application does the following:
 *      (1) Initializes:
 *              (a) Queue Manager (QM) Subsystem 
 *              (b) Packet Accelerator (PA) CPPI DMA
 *
 *      (2) Sets up the CPPI descriptors and Queues required for sending and
 *          receiving data using Ethernet.
 *              (a) Uses Host descriptors
 *              (b) Uses QMSS poll mode
 *
 *      (3) Sets up the example application's configuration (MAC address
 *          it uses to send/recv data; IP address and port number it's listening
 *          on) in PA Subsystem so as to enable the PASS to forward all packets
 *          matching this configuration onto the application for processing.
 *              (a) Switch MAC address configured   =   0x10:0x11:0x12:0x13:0x14:0x15
 *              (b) Example's IP address            =   192.168.1.10
 *              (c) Example App's listening port    =   0x5678
 *
 *      (4) Sends packets onto wire 
 *          (constructed manually in code here with following settings):
 *              (a) Source MAC      =   0x00:0x01:0x02:0x03:0x04:0x05
 *                  Destination MAC =   0x10:0x11:0x12:0x13:0x14:0x15
 *              (b) Source IP       =   192.168.1.1
 *                  Destination IP  =   192.168.1.10
 *              (c) Source Port     =   0x1234
 *                  Destination Port=   0x5678
 *              (d) Payload Data (80 bytes)
 *
 *          The packets sent by the application are looped back at PA by default.The packets 
 *          are received and passed back up to the example application for processing.
 *      
 *      (5) Application receives all packets using QM poll mode
 *          Validates received packet against data sent.
 *
 *  Example Test Setup:
 *
 *          EVM for an SOC example running Linux environment on ARM processor.
 *          If packets are routed to network. You could put another PC on the Hub to observe packets 
 *          being sent onto wire.
 *
 *  \par
 *  ============================================================================
 *  @n   (C) Copyright 2012, Texas Instruments, Inc.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h> 
#include <time.h>
#include <pthread.h>
#include <string.h>
#include "fw_mem_allocator.h"

void  *topLevelTest (void *args);


/* The exit code is a global. This is used so
 * the clock function can terminate the program with
 * the proper exit code 
 */
int exitCode;

/**************************************************************
************************** DEFINITIONS ************************
***************************************************************/
/* Number of packets to be used for testing the example. */
#define     MAX_NUM_PACKETS     10

/* Counters to track number of packets sent/received by this application */
extern volatile uint32_t                gRxCounter, gTxCounter;

/*
 * Default test configuration for the silicon
 *
 * Packets being looped back at PASS
 *    passLpbkMode = PASS_LOOPBACK_INTERNAL
 *    passLpbkMode = PASS_LOOPBACK_NONE : To disable the Loopback
 */
int32_t passLpbkMode = PASS_LOOPBACK_INTERNAL;

#define MAX_RETRIES 5

/**************************************************************
**************** EXAMPLE APP FUNCTIONS ************************
***************************************************************/




/** ============================================================================
 *   @n@b fw_test
 *
 *   @b Description
 *   @n Example application that sets up the application, sends, receives
 *      data.
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return
 *   @n None
 *
 * =============================================================================
 */
void* fw_test (void *args)
{
    int32_t         i;

    System_printf ("**************************************************\n");
    System_printf ("******* Ethernet Single Core Example Start *******\n");
    System_printf ("**************************************************\n");

    /* Initialize the components required to run the example:
     *  (1) QMSS
     *  (2) CPPI
     *  (3) Ethernet switch subsystem + MDIO + SGMII
     */
    /* Initialize QMSS */
    if (Init_Qmss () != 0)
    {
        System_printf ("QMSS init failed \n");
        return;
    }
    else
    {
        System_printf ("QMSS successfully initialized \n");            
    }

    /* Initialize CPPI */
    if (Init_Cppi () != 0)
    {
        System_printf ("CPPI init failed \n");
        return;
    }
    else
    {
        System_printf ("CPPI successfully initialized \n");            
    }

    /* Init PA LLD */
    if (Init_PASS () != 0)
    {
        System_printf ("PASS init failed \n");
        return;
    }
    else
    {
        System_printf ("PASS successfully initialized \n");            
    }

    System_printf ("Ethernet subsystem  initializion bypassed \n");

    /* Setup Tx */
    if (Setup_Tx () != 0)
    {
        System_printf ("Tx setup failed \n");
        return;
    }
    else
    {
        System_printf ("Tx setup successfully done \n");            
    }

    /* Setup Rx */
    if (Setup_Rx () != 0)
    {
        System_printf ("Rx setup failed \n");
        return;
    }
    else
    {
        System_printf ("Rx setup successfully done \n");            
    }

    /* Setup PA */
    if (Setup_PASS () != 0)
    {
        System_printf ("PASS setup failed \n");
        return;
    }
    else
    {
        System_printf ("PASS setup successfully done \n");            
    }

    System_printf ("PA Stats before Packet Transmission BEGIN ********* ... \n");
    if (getPaStats ())  {
        System_printf ("Function getPaStats failed\n");
        return;
    }
    
    System_printf ("PA Stats before Packet Transmission END ********* ... \n");
    
    /* Run some data through and verify transfer worked */
    //mdebugHaltPdsp(0);
    System_printf ("Packet Transmission Start ... \n");
    for (i = 0; i < MAX_NUM_PACKETS; i ++)
    {
        if (SendPacket () != 0)
        {
            System_printf ("Packet %d send failed \n", i);
            return;
        }
    }     
    
    /* Wait until all packet reception is done */
    System_printf ("Packet Transmission Done.\nWait for all packets to be Received ... \n");

    i =0;
    while ((gRxCounter != gTxCounter) && (i < MAX_RETRIES)) {
        if(ReceivePacket() == -1)
        {
            System_printf("Verififcation Failed for Received %d packets so far...\n", gRxCounter);
        }
        CycleDelay (10000);   
        i++;        
    }
    
    System_printf("Received %d packets so far...\n", gRxCounter);

    System_printf ("PA Stats After Packet Transmission BEGIN ********* ... \n");
    if (getPaStats ())  {
        System_printf ("Function getPaStats failed\n");
        return;
    }
    
    System_printf ("PA Stats After Packet Transmission END ********* ... \n");
    
    System_printf ("Packets Sent\t\t=\t%d \nPackets Received\t=\t%d \nExample Done! \n", gTxCounter,  gRxCounter);
  
    System_printf ("**************************************************\n");
    System_printf ("******** Ethernet Single Core Example End ********\n");
    System_printf ("**************************************************\n");
    
}


typedef pthread_t task_handle;

#define DEFAULT_STACK_SIZE  0x8000
/** ============================================================================
 *   @n@b task_create
 *
 *   @b Description
 *   @n Create thread to run the test program
 *
 *   @param[in]  
 *   @n None
 * 
 *   @return    int32_t
 *              -1      -   Error
 *              0       -   Success
 * =============================================================================
 */
static int task_create ( void *(start_routine)(void*), void* args, void* handle)
{
    int max_priority, err;
    pthread_t thread;
    pthread_attr_t attr;
    struct sched_param param;

    max_priority = sched_get_priority_max(SCHED_FIFO);
    err = pthread_attr_init(&attr);
    if (err) {
        printf("pthread_attr_init failed: (%s)\n", strerror(err));
        return err;
    }
    err = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (err) {
        printf("pthread_attr_setdetachstate failed: (%s)\n", strerror(err));
        return err;
    }
    err = pthread_attr_setstacksize(&attr, DEFAULT_STACK_SIZE);
    if (err) {
        printf("pthread_attr_setstacksize failed: (%s)\n", strerror(err));
        return err;
    }
    err = pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
    if (err) {
        printf("pthread_attr_setinheritsched failed: (%s)\n", strerror(err));
        return err;
    }
    err = pthread_attr_setschedpolicy(&attr, SCHED_FIFO);
    if (err) {
        printf("pthread_attr_setschedpolicy failed: (%s)\n", strerror(err));
        return err;
    }
    memset(&param, 0, sizeof(param));
    param.sched_priority = max_priority;
    err = pthread_attr_setschedparam(&attr, &param);
    if (err) {
        printf("pthread_attr_setschedparam failed: (%s)\n", strerror(err));
        return err;
    }
    if (err) return err;
    err = pthread_create(&thread, &attr, start_routine, args);
    if (err) {
        printf("pthread_create failed: (%s)\n", strerror(err));
        return err;
    }
    if (err) return err;
    *(pthread_t*)handle = thread;
    return 0;
}

/** ============================================================================
 *   @n@b task_wait
 *
 *   @b Description
 *   @n Wait for Task completion
 * 
 *   @return    void
 * =============================================================================
 */
static void task_wait (void *handle)
{
    pthread_join(*((pthread_t*)handle), NULL);
    return;
}
/** ============================================================================
 *   @n@b task_sleep
 *
 *   @b Description
 *   @n Sleep the thread for msec duration
 * 
 *   @return    void
 * =============================================================================
 */

static void task_sleep(int time_in_msec)
{
    pthread_mutex_t fake_mutex = PTHREAD_MUTEX_INITIALIZER;
    pthread_cond_t fake_cond = PTHREAD_COND_INITIALIZER;
    struct timespec ts;
    int rt;
    unsigned int sec, nsec;

    sec = time_in_msec/1000;
    nsec = (time_in_msec - (sec*1000)) * 1000000;

    /* Use the wall-clock time */
    clock_gettime(CLOCK_REALTIME, &ts);

    ts.tv_sec += sec;
    ts.tv_nsec += nsec;

    pthread_mutex_lock(&fake_mutex);
    rt = pthread_cond_timedwait(&fake_cond, &fake_mutex, &ts);
    pthread_mutex_unlock(&fake_mutex);
}
/** ============================================================================
 *   @n@b main
 *
 *   @b Description
 *   @n test application main
 * 
 *   @return    int
 * =============================================================================
 */
int main() {
    task_handle test_th;
    int status;

    fw_osalInit();

    if (fw_memAllocInit((uint8_t*)MSMC_SRAM_BASE_ADDR,
                                  MSMC_TEST_PERM_MEM_SZ) == fw_FALSE) {
        printf("ERROR: \"Top Level Test\" fw_memAllocInit failed\n");
        return (-1);
    }


    /* Create virtual memory maps */
    /* QMSS CFG Regs */
    fw_qmssCfgVaddr = fw_memMap((void*)QMSS_CFG_BASE_ADDR,
                                            QMSS_CFG_BLK_SZ);
    if (!fw_qmssCfgVaddr)
    {
        printf("ERROR: Failed to map QMSS CFG registers\n");
        return (-1);
    }
#ifdef EXT_DEBUG
    printf("main:QMSS_CFG_BASE_ADDR:0x%x Memory mapped at address %p.\n",(void*)QMSS_CFG_BASE_ADDR, fw_qmssCfgVaddr); 
#endif

    /* QMSS DATA Regs */
    fw_qmssDataVaddr = fw_memMap((void*)QMSS_DATA_BASE_ADDR,
                                            QMSS_DATA_BLK_SZ);
    if (!fw_qmssDataVaddr)
    {
        printf("ERROR: Failed to map QMSS DATA registers\n");
        return (-1);
    }
#ifdef EXT_DEBUG
    printf("main:QMSS_DATA_BASE_ADDR:0x%x Memory mapped at address %p.\n",(void*)QMSS_DATA_BASE_ADDR, fw_qmssDataVaddr);
#endif

    /* SRIO CFG Regs */
    fw_srioCfgVaddr = fw_memMap((void*)SRIO_CFG_BASE_ADDR,
                                            SRIO_CFG_BLK_SZ);
    if (!fw_srioCfgVaddr)
    {
        printf("ERROR: Failed to map SRIO CFG registers\n");
        return (-1);
    }
#ifdef EXT_DEBUG
    printf("main:SRIO_CFG_BASE_ADDR:0x%x Memory mapped at address %p.\n",(void*)SRIO_CFG_BASE_ADDR, fw_srioCfgVaddr);
#endif

    /* PASS CFG Regs */
    fw_passCfgVaddr = fw_memMap((void*)PASS_CFG_BASE_ADDR,
                                            PASS_CFG_BLK_SZ);
    if (!fw_passCfgVaddr)
    {
        printf("ERROR: Failed to map PASS CFG registers\n");
        return (-1);
    }
#ifdef EXT_DEBUG
    printf("main:PASS_CFG_BASE_ADDR:0x%x Memory mapped at address %p.\n",(void*)PASS_CFG_BASE_ADDR, fw_passCfgVaddr);
#endif


    if (status = task_create(topLevelTest, NULL, &test_th)) {
        printf("ERROR: \"Top Level Test\" task-create failed (%d)\n", status);
        return (-1);
    }

    task_wait(&test_th);
    fw_osalshutdown();
    return 0;
}

/** ============================================================================
 *   @n@b topLevelTest
 *
 *   @b Description
 *   @n Routine running as a separate thread
 * 
 *   @return    int
 * =============================================================================
 */
void *topLevelTest (void *args)
{
    task_handle fw_test_th;
    int status;

    fw_osalInit();

    if (status = task_create(fw_test, NULL, &fw_test_th)) {
        printf("ERROR: \"fwTest\" task-create failed (%d)\n", status);
        return;
    }

    task_wait(&fw_test_th);
    pthread_exit((void*) 0);
    return;
}


