/* NXP I3C Master IP character device for NFC Controller tests */
/* TODO: Add a proper license */
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/atomic.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/io.h>
#include <linux/iopoll.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/bitops.h>
#include <linux/err.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/of_irq.h>
#include <linux/i3c/master.h>
#include <linux/i3c/device.h>
#include <linux/i3c/ccc.h>
#include <linux/list.h>
#include <linux/platform_device.h>
#include <linux/i3c/nxp_i3c.h>
#include <linux/uaccess.h>

#define DRIVER_NAME "i3c-nxp"
#define DEVICE_NAME "i3c-nxp"
#define CLASS_NAME "i3c"

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Ricardo Barbedo");
MODULE_AUTHOR("Valentin Saindon");
MODULE_DESCRIPTION("NXP I3C Master IP for NFCC testing");
MODULE_VERSION("0.1");

#define LEFT_SHIFT(shift, value) ((value) << (shift))

/* Register location */
#define MCONFIG_REG 0x00
#define MCTRL_REG 0x084
#define MSTATUS_REG 0x088
#define IBIRULES_REG 0x08C
#define MINTSET_REG 0x090
#define MINTCLR_REG 0x094
#define MINTMASKED_REG 0x098
#define MERRWARN_REG 0x09C
#define MDMACTRL_REG 0x0A0
#define MDATACTRL_REG 0x0AC
#define MWDATAB_REG 0x0B0
#define MWDATABE_REG 0x0B4
#define MWDATAH_REG 0x0B8
#define MWDATAHE_REG 0x0BC
#define MRDATAB_REG 0x0C0
#define MRDATAH_REG 0x0C4
#define MWMSG_SDR_REG 0x0D0
#define MRMSG_SDR_REG 0x0D4
#define MWMSG_DDR_REG 0x0D8
#define MRMSG_DDR_REG 0x0DC
#define MDYNADDR_REG 0x0E4

/* MCONFIG register masks */
#define MCONFIG_MSTENA_SHIFT 0
#define MCONFIG_MSTENA_MASK GENMASK(1, 0)
#define MCONFIG_MSTENA(x) (((x)&MCONFIG_MSTENA_MASK) >> MCONFIG_MSTENA_SHIFT)
#define MCONFIG_DISTO_SHIFT 3
#define MCONFIG_DISTO BIT(3)
#define MCONFIG_HKEEP_SHIFT 4
#define MCONFIG_HKEEP_MASK GENMASK(5, 4)
#define MCONFIG_HKEEP(x) (((x)&MCONFIG_HKEEP_MASK) >> MCONFIG_HKEEP_SHIFT)
#define MCONFIG_ODSTOP_SHIFT 6
#define MCONFIG_ODSTOP BIT(6)
#define MCONFIG_PERLOW_SHIFT 7
#define MCONFIG_PERLOW BIT(7)
#define MCONFIG_PPBAUD_SHIFT 8
#define MCONFIG_PPBAUD_MASK GENMASK(11, 8)
#define MCONFIG_PPBAUD(x) (((x)&MCONFIG_PPBAUD_MASK) >> MCONFIG_PPBAUD_SHIFT)
#define MCONFIG_PPLOW_SHIFT 12
#define MCONFIG_PPLOW_MASK GENMASK(15, 12)
#define MCONFIG_PPLOW(x) (((x)&MCONFIG_PPLOW_MASK) >> MCONFIG_PPLOW_SHIFT)
#define MCONFIG_ODBAUD_SHIFT 16
#define MCONFIG_ODBAUD_MASK GENMASK(23, 16)
#define MCONFIG_ODBAUD(x) (((x)&MCONFIG_ODBAUD_MASK) >> MCONFIG_ODBAUD_SHIFT)
#define MCONFIG_ODHPP_SHIFT 24
#define MCONFIG_ODHPP BIT(24)
#define MCONFIG_SKEW_SHIFT 25
#define MCONFIG_SKEW_MASK GENMASK(27, 25)
#define MCONFIG_SKEW(x) (((x)&MCONFIG_SKEW_MASK) >> MCONFIG_SKEW_SHIFT)
#define MCONFIG_I2CBAUD_SHIFT 28
#define MCONFIG_I2CBAUD_MASK GENMASK(31, 28)
#define MCONFIG_I2CBAUD(x) (((x)&MCONFIG_I2CBAUD_MASK) >> MCONFIG_I2CBAUD_SHIFT)

/* MCTRL register masks */
#define MCTRL_REQUEST_SHIFT 0
#define MCTRL_REQUEST_MASK GENMASK(2, 0)
#define MCTRL_TYPE_SHIFT 4
#define MCTRL_TYPE_MASK GENMASK(5, 4)
#define MCTRL_IBIRESP_SHIFT 6
#define MCTRL_IBIRESP_MASK GENMASK(7, 6)
#define MCTRL_IBIRESP_WRITE(x)                                                 \
	(((x) << MCTRL_IBIRESP_SHIFT) & MCTRL_IBIRESP_MASK)
#define MCTRL_READ BIT(8)
#define MCTRL_ADDR_SHIFT 9
#define MCTRL_ADDR_MASK GENMASK(15, 9)
#define MCTRL_ADDR(x) (((x) << MCTRL_ADDR_SHIFT) & MCTRL_ADDR_MASK)
#define MCTRL_RDTERM_SHIFT 16
#define MCTRL_RDTERM_MASK GENMASK(23, 16)
#define MCTRL_RDTERM(x) (((x) << MCTRL_RDTERM_SHIFT) & MCTRL_RDTERM_MASK)
// #define MCTRL_RDTERM(x) (((x)&MCTRL_RDTERM_MASK) <<)

/* MCTRL_REQUEST value */
#define MCTRL_REQUEST_NONE 0
#define MCTRL_REQUEST_STARTADDR 1
#define MCTRL_REQUEST_STOP 2
#define MCTRL_REQUEST_IBIACKNACK 3
#define MCTRL_REQUEST_PROCESSDAA 4
#define MCTRL_REQUEST_FORCEEXIT 6
#define MCTRL_REQUEST_AUTOIBI 7

/* MCTRL_TYPE value */
#define MCTRL_TYPE_I3C 0
#define MCTRL_TYPE_I2C 1
#define MCTRL_TYPE_DDRHDR 2
#define MCTRL_TYPE_WTF 3

#define MCTRL_IBIRESP_ACKNOBYTE 0
#define MCTRL_IBIRESP_NACK 1
#define MCTRL_IBIRESP_ACKWITHBYTE 2
#define MCTRL_IBIRESP_MANUAL 3

/* MSTATUS register masks */
#define MSTATUS_STATE_SHIFT 0
#define MSTATUS_STATE_MASK GENMASK(2, 0)
#define MSTATUS_STATE(x) (((x) << MSTATUS_STATE_SHIFT) & MSTATUS_STATE_MASK)
#define MSTATUS_STATE_READ(x) (((x)&MSTATUS_STATE_MASK) >> MSTATUS_STATE_SHIFT)
#define MSTATUS_BETWEEN BIT(4)
#define MSTATUS_NACKED BIT(5)
#define MSTATUS_IBITYPE
#define MSTATUS_IBITYPE_SHIFT 6
#define MSTATUS_IBITYPE_MASK GENMASK(7, 6)
#define MSTATUS_IBITYPE(x) (((x)&MSTATUS_IBITYPE_MASK) >> MSTATUS_IBITYPE_SHIFT)
#define MSTATUS_SLVSTART BIT(8)
#define MSTATUS_MCTRLDONE BIT(9)
#define MSTATUS_COMPLETE BIT(10)
#define MSTATUS_RXPEND BIT(11)
#define MSTATUS_TXNOTFULL BIT(12)
#define MSTATUS_IBIWON BIT(13)
#define MSTATUS_ERRWARN BIT(15)
#define MSTATUS_NOWMASTER BIT(19)
#define MSTATUS_IBIADDR_SHIFT 24
#define MSTATUS_IBIADDR_MASK GENMASK(30, 24)
#define MSTATUS_IBIADDR(x) (((x)&MSTATUS_IBIADDR_MASK) >> MSTATUS_IBIADDR_SHIFT)

#define MSTATUS_STATE_IDLE 0
#define MSTATUS_STATE_SLVREQ 1
#define MSTATUS_STATE_MSGSDR 2
#define MSTATUS_STATE_NORMACT 3
#define MSTATUS_STATE_DDR 4
#define MSTATUS_STATE_DDA 5
#define MSTATUS_STATE_IBIACK 6
#define MSTATUS_STATE_IBIRCV 7

#define MSTATUS_IBITYPE_NONE 0
#define MSTATUS_IBITYPE_IBI 1
#define MSTATUS_IBITYPE_MR 2
#define MSTATUS_IBITYPE_HJ 3

// struct nxp_ic3_status {
// 	unsigned state : 3;
// 	unsigned between : 1;
// 	unsigned nacked : 1;
// 	unsigned ibi_type : 2;
// 	unsigned slv_start : 1;
// 	unsigned mctrl_done : 1;
// 	unsigned complete : 1;
// 	unsigned rxpend : 1;
// 	unsigned tx_not_full : 1;
// 	unsigned ibi_won : 1;
// 	unsigned err_warn : 1;
// 	unsigned now_master : 1;
// 	unsigned ibi_addr : 7;
// };

// typedef union i3c_status_bit_converter {
// 	struct nxp_ic3_status bf;
// 	u32 i;
// } i3c_status_bit_converter;

/* IBIRULES register masks */
#define IBIRULES_NOBYTE BIT(31)
#define IBIRULES_MSB0 BIT(30)
#define IBIRULES_ADDR1_SHIFT 24
#define IBIRULES_ADDR1_MASK GENMASK(24, 19)
#define IBIRULES_ADDR2_SHIFT 18
#define IBIRULES_ADDR2_MASK GENMASK(23, 18)
#define IBIRULES_ADDR3_SHIFT 12
#define IBIRULES_ADDR3_MASK GENMASK(17, 12)
#define IBIRULES_ADDR4_SHIFT 6
#define IBIRULES_ADDR4_MASK GENMASK(11, 6)
#define IBIRULES_ADDR5_SHIFT 0
#define IBIRULES_ADDR5_MASK GENMASK(5, 0)

/* MERRWARN register masks */
#define MERRWARN_ORUN BIT(0)
#define MERRWARN_URUN BIT(1)
#define MERRWARN_NACK BIT(2)
#define MERRWARN_WRABT BIT(3)
#define MERRWARN_TERM BIT(4)
#define MERRWARN_HPAR BIT(9)
#define MERRWARN_HCRC BIT(10)
#define MERRWARN_OREAD BIT(16)
#define MERRWARN_OWRITE BIT(17)
#define MERRWARN_MSGERR BIT(18)
#define MERRWARN_INVREQ BIT(19)
#define MERRWARN_TIMEOUT BIT(20)

/* MDMACTRL register masks */
#define MERRWARN_DMAFB_SHIFT 0
#define MERRWARN_DMAFB_MASK GENMASK(1, 0)
#define MERRWARN_DMATB_SHIFT 2
#define MERRWARN_DMATB_MASK GENMASK(3, 2)
#define MERRWARN_DMAWIDTH_SHIFT 4
#define MERRWARN_DMAWIDTH_MASK GENMASK(5, 4)

/* DATACTRL register masks */
#define DATACTRL_FLUSHTB BIT(0)
#define DATACTRL_FLUSHFB BIT(1)
#define DATACTRL_UNLOCK BIT(3)
#define DATACTRL_TXTRIG_SHIFT 4
#define DATACTRL_TXTRIG_MASK GENMASK(5, 4)
#define DATACTRL_RXTRIG_SHIFT 6
#define DATACTRL_RXTRIG_MASK GENMASK(7, 6)
#define DATACTRL_TXCOUNT_SHIFT 16
#define DATACTRL_TXCOUNT_MASK GENMASK(20, 16)
#define DATACTRL_RXCOUNT_SHIFT 24
#define DATACTRL_RXCOUNT_MASK GENMASK(28, 24)
#define DATACTRL_TXFULL BIT(30)
#define DATACTRL_RXEMPTY BIT(31)

/* MWDATAB register masks */
#define MWDATAB_VALUE_SHIFT 0
#define MWDATAB_VALUE_MASK GENMASK(7, 0)
#define MWDATAB_END BIT(7)

/* MWDATABE register masks */
#define MWDATABE_VALUE_SHIFT 0
#define MWDATABE_VALUE_MASK GENMASK(7, 0)

/* MWDATAH register masks */
#define MWDATAH_DATA0_SHIFT 0
#define MWDATAH_DATA0_MASK GENMASK(7, 0)
#define MWDATAH_DATA1_SHIFT 8
#define MWDATAH_DATA1_MASK GENMASK(15, 8)
#define MWDATAH_END BIT(16)

/* WDATAHE register masks */
#define WDATAHE_DATA0_SHIFT 0
#define WDATAHE_DATA0_MASK GENMASK(7, 0)
#define WDATAHE_DATA1_SHIFT 8
#define WDATAHE_DATA1_MASK GENMASK(15, 8)

/* MRDATAB register masks */
#define MRDATAB_VALUE_SHIFT 0
#define MRDATAB_VALUE_MASK GENMASK(7, 0)

/* RDATAH register masks */
#define RDATAH_LSB_SHIFT 0
#define RDATAH_LSB_MASK GENMASK(7, 0)
#define RDATAH_MSB_SHIFT 8
#define RDATAH_MSB_MASK GENMASK(15, 8)

/* MWMSG_SDR register masks */
#define MWMSG_SDR_DATA_SHIFT 0
#define MWMSG_SDR_DATA_MASK GENMASK(15, 0)
/* Register 0:15 can be take 2 bit masks depending on first msg or no  */
#define MWMSG_SDR_LEN_SHIFT 11
#define MWMSG_SDR_LEN_MASK GENMASK(15, 11)
#define MWMSG_SDR_I2C BIT(12)
#define MWMSG_SDR_END BIT(8)
#define MWMSG_SDR_ADDR_SHIFT 1
#define MWMSG_SDR_ADDR_MASK GENMASK(7, 1)
#define MWMSG_SDR_DIR BIT(0)

/* MRMSG_SDR register mask */
#define MRMSG_SDR_DATA_SHIFT 0
#define MRMSG_SDR_DATA_MASK GENMASK(15, 0)

/* MWMSG_DDR register masks */
#define MWMSG_DDR_ADDRCMD_SLVADDR_SHIFT 9
#define MWMSG_DDR_ADDRCMD_SLVADDR_MASK GENMASK(15, 9)
#define MWMSG_DDR_ADDRCMD_RESERVED BIT(8)
#define MWMSG_DDR_ADDRCMD_RW BIT(7)
#define MWMSG_DDR_ADDRCMD_CMD_SHIFT 0
#define MWMSG_DDR_ADDRCMD_CMD_MASK GENMASK(6, 0)
#define MWMSG_DDR_DATA_SHIFT 0
#define MWMSG_DDR_DATA_MASK GENMASK(15, 0)
#define MWMSG_DDR_END BIT(14)
#define MWMSG_DDR_LEN_SHIFT 0
#define MWMSG_DDR_LEN_MASK GENMASK(9, 0)

/* MRMSG_DDR register masks */
#define MRMSG_DDR_DATA_SHIFT 0
#define MRMSG_DDR_DATA_MASK GENMASK(15, 0)

/* MDYNADDR register masks */
#define MDYNADDR_DAVALID BIT(0)
#define MDYNADDR_DADDR_SHIFT 1
#define MDYNADDR_DADDR_MASK GENMASK(7, 1)

/* Value defined from ip information. Cannot be read from ip registers */
#define I3C_NXP_IP_FIFO_DEPTH 8

/*
 * Timeout in us in busy loop while waiting for the status interrupt signaling
 * the end of the SPMI frame.
 */
#define INTERRUPT_STATUS_POLL_TIMEOUT 20
/* Poll sleep period  */
#define INTERRUPT_STATUS_POLL_SLEEP 0

#define INTERRUPT_STATUS_IDLE_TIMEOUT 1000000
/* Poll sleep period  */
#define INTERRUPT_STATUS_IDLE_SLEEP 10

// Can process only one command a time with the current master IP
#define MASTER_CMD_FIFO_DEPH 1

#define MAX_DEVS 16

static struct class *i3c_nxp_class = NULL;
static struct device *i3c_nxp_device = NULL;
static int major_number = 0;

struct nxp_i3c_cmd {
	u8 rnw;
	u32 cmd_ctrl;
	u16 len;
	const void *tx_buf;
	void *rx_buf;
	unsigned error;
};

struct nxp_i3c_xfer {
	struct list_head node;
	struct completion comp;
	int ret;
	unsigned status;
	unsigned int ncmds;
	struct nxp_i3c_cmd cmds[0];
};

struct nxp_i3c_device_slot {
	u8 id;
	u16 static_addr;
	u8 dynamic_addr;
	u8 active;
	u8 ibi;
	u8 type;
};

struct nxp_i3c_master {
	// struct work_struct hj_work;
	struct i3c_master_controller base;
	u8 bus_in_i3c;
	u8 bus_in_ddr;
	u8 bus_in_sdr;
	i3c_config conf;
	// u32 free_rr_slots;
	// unsigned int maxdevs;
	struct {
		unsigned int rules; // IBIRULES register configuration
		u8 resp; // Response method if the master receive an IBI from slave
		// struct i3c_dev_desc **slots;
		u16 addr;
		u8 type;
		struct work_struct slvreq_work;
		struct work_struct rxbyte_work;
		u8 enable;
		spinlock_t lock;
	} ibi;
	struct {
		struct list_head list;
		struct nxp_i3c_xfer *cur;
		spinlock_t lock;
	} xferqueue;
	struct {
		struct work_struct hjwork;
		unsigned int mode;
	} hj;
	void __iomem *regs;
	unsigned interrupt_set;
	// unsigned int irq;
	// struct cdns_i3c_master_caps caps; // to configure differents fifos capacity
	/* TODO : Manage the IP FIFO depth capacty (Read Fifo, Write Fifo, etc) */
};

struct nxp_i3c_device_data {
	struct cdev cdev;
	struct nxp_i3c_master *master;
	int irq;
	struct task_struct *task;
	DECLARE_KFIFO(evt_fifo, struct ibi_event, EVENT_FIFO_SIZE);
	struct ibi_event *ibi_event;
	atomic_t available;
};

static struct nxp_i3c_device_data my_device_data;

static inline struct nxp_i3c_master *
to_nxp_i3c_master(struct i3c_master_controller *master)
{
	return container_of(master, struct nxp_i3c_master, base);
}

/*
 * Read STATUS register until the bits represented by the given mask change
 * @master: i3c master instance
 * @status_to_trig: Mask of bits to trigger
 *
 * Return 0 if success, -ETIMEOUT if timeout is reached, and negative value of
 * MERRWARN register if ERRWARN flag of MSTATUS register is set to 1.
 */
static int nxp_i3c_master_trigger_cmd(struct nxp_i3c_master *master,
				      u32 status_to_trig)
{
	int err = 0;
	u32 status = 0;

	err = readl_poll_timeout(master->regs + MINTMASKED_REG, status,
				 status & (status_to_trig | MSTATUS_ERRWARN),
				 INTERRUPT_STATUS_POLL_SLEEP,
				 INTERRUPT_STATUS_POLL_TIMEOUT);
	writel(status_to_trig, master->regs + MINTCLR_REG);
	if (!err) {
		printk(KERN_INFO "SPMI NXP: Trigger command timed out\n");
		return err;
	}

	if (status & MSTATUS_ERRWARN)
		return (-readl(master->regs + MERRWARN_REG));

	return readl(master->regs + MSTATUS_REG);
}

static void nxp_i3c_master_clear_status(struct nxp_i3c_master *master)
{
	writel(0xFFFFFF, master->regs + MSTATUS_REG);
}

static void nxp_i3c_master_flush_buffers(struct nxp_i3c_master *master)
{
	writel(DATACTRL_FLUSHFB | DATACTRL_FLUSHTB,
	       master->regs + MDATACTRL_REG);
}

static void nxp_i3c_master_update_bus_status(struct nxp_i3c_master *master,
					     unsigned status)
{
	master->bus_in_ddr = (MSTATUS_STATE_READ(status) == MSTATUS_STATE_DDR);
	master->bus_in_sdr =
		(MSTATUS_STATE_READ(status) == MSTATUS_STATE_MSGSDR);
}

static void nxp_i3c_master_clear_err_warn(struct nxp_i3c_master *master)
{
	writel(0xFFFFFF, master->regs + MERRWARN_REG);
}

static int nxp_i3c_master_write_exit(struct nxp_i3c_master *master)
{
	int res = 0;
	writel(MCTRL_REQUEST_FORCEEXIT, master->regs + MCTRL_REG);

	res = nxp_i3c_master_trigger_cmd(master, MSTATUS_MCTRLDONE);
	if (res < 0) {
		nxp_i3c_master_clear_err_warn(master);
		return res;
	}
	nxp_i3c_master_clear_status(master);

	nxp_i3c_master_update_bus_status(master, res);

	return res;
}

static int nxp_i3c_master_write_stop(struct nxp_i3c_master *master)
{
	int res = 0;
	writel(MCTRL_REQUEST_STOP, master->regs + MCTRL_REG);

	res = nxp_i3c_master_trigger_cmd(master, MSTATUS_MCTRLDONE);
	if (res < 0) {
		nxp_i3c_master_clear_err_warn(master);
		return res;
	}

	nxp_i3c_master_update_bus_status(master, res);

	return res;
}

static void nxp_i3c_master_ibi_init_rules(struct nxp_i3c_master *master,
					  unsigned flags)
{
	/* Initialize IBI rules with MSB0 enable */
	writel(IBIRULES_MSB0, master->regs + IBIRULES_REG);
}

static void nxp_i3c_master_ibi_update_rules(struct nxp_i3c_master *master,
					    unsigned flags)
{
	int current_ibi;
	current_ibi = readl(master->regs + IBIRULES_REG);
	writel(current_ibi | flags, master->regs + IBIRULES_REG);
}

static struct nxp_i3c_xfer *
nxp_i3c_master_xfer_alloc(struct nxp_i3c_master *master, unsigned int ncmds)
{
	struct nxp_i3c_xfer *xfer;

	xfer = kzalloc(sizeof(*xfer) + sizeof(struct nxp_i3c_cmd) * ncmds,
		       GFP_KERNEL);
	if (!xfer)
		return NULL;

	INIT_LIST_HEAD(&xfer->node);
	xfer->ncmds = ncmds;
	xfer->ret = -ETIMEDOUT;

	return xfer;
}

static void nxp_i3c_master_xfer_free(struct nxp_i3c_xfer *xfer)
{
	kfree(xfer);
}

static void nxp_i3c_master_xfer_start_locked(struct nxp_i3c_master *master)
{
	struct nxp_i3c_xfer *xfer = master->xferqueue.cur;
	unsigned int i, idx, ctrl_status;
	const char *payload;
	struct nxp_i3c_cmd *cmd;
	if (!xfer)
		return;

	/* I3C API doesn't have a cmd FIFO and cannot process more than one command
	   a time. Limite command to 1 max */
	if (xfer->ncmds > 1) {
		printk(KERN_ERR
		       "NXP I3C : unable to process more than 1 xfer a time\n");
		return;
	}

	ctrl_status = readl(master->regs + MDATACTRL_REG);
	if (!(ctrl_status & DATACTRL_RXEMPTY)) {
		printk(KERN_ERR
		       "NXP I3C : cannot process xfer frame because rx fifo is not \
			   empty.\n");
		return;
	}

	if (ctrl_status & DATACTRL_TXFULL) {
		printk(KERN_ERR
		       "NXP I3C : cannot process xfer frame because tx fifo is full.\n");
	}

	// enable interruption for command completion or error/warning */
	nxp_i3c_master_set_interrupt_mask(
		master, MSTATUS_COMPLETE | MSTATUS_ERRWARN | MSTATUS_MCTRLDONE);
	// writel(readl(master->regs + MINTSET_REG) | MSTATUS_COMPLETE |
	// 	       MSTATUS_ERRWARN | MSTATUS_MCTRLDONE,
	//        master->regs + MINTSET_REG);

	/* TODO : Support for IBI event */

	/* Process commands list stored in xfer (private transfert)
	 * (option 1 of nxp i3c master ip)
     */
	for (i = 0; i < xfer->ncmds; i++) {
		cmd = &xfer->cmds[i];
		writel(cmd->cmd_ctrl, master->regs + MCTRL_REG);
		payload = cmd->tx_buf;
		if (cmd->rnw) {
			for (idx = 0; idx < (cmd->len - 1); idx++) {
				writel(payload[idx],
				       master->regs + MWDATAB_REG);
				if (!nxp_i3c_master_trigger_cmd(
					    master, MSTATUS_TXNOTFULL))
					return;
			}
			writel(payload[cmd->len - 1],
			       master->regs + MWDATABE_REG);
		}
	}
}

static void nxp_i3c_master_xfer_end_locked(struct nxp_i3c_master *master,
					   u32 isr)
{
	struct nxp_i3c_xfer *xfer = master->xferqueue.cur;
	int err = 0;
	int ret = 0;
	int idx = 0;
	char *payload;
	u32 status;

	if (!xfer)
		return;

	// enable interruption for command completion or error/warning */
	nxp_i3c_master_set_interrupt_mask(master, ~MSTATUS_COMPLETE &
							  ~MSTATUS_ERRWARN &
							  ~MSTATUS_MCTRLDONE);
	// if (!(isr & MSTATUS_COMPLETE))
	// 	return;
	xfer->status = readl(master->regs + MSTATUS_REG);
	xfer->cmds[0].error = 0;
	if (isr & MSTATUS_ERRWARN) {
		err = readl(master->regs + MERRWARN_REG);

		printk(KERN_ERR
		       "NXP I3C : Error during xfer exchange. Err %d\n",
		       err);
		xfer->cmds[0].error = err;

		switch (err) {
		case MERRWARN_ORUN:
		case MERRWARN_URUN:
		case MERRWARN_NACK:
		case MERRWARN_WRABT:
		case MERRWARN_TERM:
			ret = -EIO;
			break;
		case MERRWARN_OREAD:
		case MERRWARN_OWRITE:
			ret = -ENOSPC;
			break;
		case MERRWARN_TIMEOUT:
			ret = -ETIMEDOUT;
			break;
		case MERRWARN_INVREQ:
		default:
			ret = -EINVAL;
			break;
		}
	} else if (xfer->cmds[0].rnw && (isr & MSTATUS_RXPEND)) {
		idx = 0;
		payload = xfer->cmds[0].rx_buf;
		for (status = readl(master->regs + MSTATUS_REG);
		     !(status & MSTATUS_COMPLETE) ||
		     ((status & MSTATUS_RXPEND) && idx < xfer->cmds[0].len);
		     status = readl(master->regs + MSTATUS_REG), idx++) {
			payload[idx] = readl(master->regs + MRDATAB_REG);
		}
		if (idx != xfer->cmds[0].len)
			xfer->cmds[0].len = idx;
	}

	xfer->ret = ret;
	complete(&xfer->comp);

	xfer = list_first_entry_or_null(&master->xferqueue.list,
					struct nxp_i3c_xfer, node);
	if (xfer)
		list_del_init(&xfer->node);

	master->xferqueue.cur = xfer;
	nxp_i3c_master_xfer_start_locked(master);
}

static void nxp_i3c_master_xfer_queue(struct nxp_i3c_master *master,
				      struct nxp_i3c_xfer *xfer)
{
	unsigned long flags;

	init_completion(&xfer->comp);
	spin_lock_irqsave(&master->xferqueue.lock, flags);
	if (master->xferqueue.cur) {
		list_add_tail(&xfer->node, &master->xferqueue.list);
	} else {
		master->xferqueue.cur = xfer;
		nxp_i3c_master_xfer_start_locked(master);
	}
	spin_unlock_irqrestore(&master->xferqueue.lock, flags);
}

static void nxp_i3c_master_xfer_unqueue(struct nxp_i3c_master *master,
					struct nxp_i3c_xfer *xfer)
{
	unsigned long flags;

	spin_lock_irqsave(&master->xferqueue.lock, flags);
	if (master->xferqueue.cur == xfer) {
		u32 status;

		status = readl(master->regs + MSTATUS_REG);
		nxp_i3c_master_update_bus_status(master, status);

		if (master->bus_in_ddr)
			nxp_i3c_master_write_exit(master);
		else
			nxp_i3c_master_write_stop(master);

		/* Wait for the buffer to come back in idle mode */
		readl_poll_timeout_atomic(master->regs + MSTATUS_REG, status,
					  status & MSTATUS_STATE_IDLE,
					  INTERRUPT_STATUS_IDLE_SLEEP,
					  INTERRUPT_STATUS_IDLE_TIMEOUT);

		master->xferqueue.cur = NULL;
		nxp_i3c_master_update_bus_status(master, status);
		nxp_i3c_master_flush_buffers(master);
		nxp_i3c_master_clear_status(master);
	} else {
		list_del_init(&xfer->node);
	}
	spin_unlock_irqrestore(&master->xferqueue.lock, flags);
}

static int nxp_i3c_master_get_err(struct nxp_i3c_cmd *cmd)
{
	switch (cmd->error) {
	case MERRWARN_MSGERR:
		return I3C_ERROR_M1;
	case MERRWARN_NACK:
		return I3C_ERROR_M2;
	default:
		break;
	}

	return I3C_ERROR_UNKNOWN;
}

static void nxp_i3c_master_set_config(struct nxp_i3c_master *master)
{
	u32 config_value = 0;

	config_value |= master->conf.stena << MCONFIG_MSTENA_SHIFT |
			master->conf.disto << MCONFIG_DISTO_SHIFT |
			master->conf.hkeep << MCONFIG_HKEEP_SHIFT |
			master->conf.odstop << MCONFIG_ODSTOP_SHIFT |
			master->conf.perlow << MCONFIG_PERLOW_SHIFT |
			master->conf.ppbaud << MCONFIG_PPBAUD_SHIFT |
			master->conf.odbaud << MCONFIG_ODBAUD_SHIFT |
			master->conf.odhpp << MCONFIG_ODHPP_SHIFT |
			master->conf.skew << MCONFIG_SKEW_SHIFT |
			master->conf.i2cbaud << MCONFIG_I2CBAUD_SHIFT;

	writel(config_value, master->regs + MCONFIG_REG);

	printk(KERN_INFO "NXP I3C IP : config written into master ip.\n");
}

static void nxp_i3c_master_set_interrupt_mask(struct nxp_i3c_master *master,
					      u32 interrupt_flag)
{
	u32 interrupt_config = master->interrupt_set | interrupt_flag;
	writel(interrupt_config, master->regs + MINTSET_REG);

	printk(KERN_INFO "NXP I3C IP : interrupt written into master ip.\n");
}

static void nxp_i3c_master_ibi_upd_rules(struct nxp_i3c_master *master,
					 u32 ibirules)
{
	master->ibi.rules = ibirules;
}

static void nxp_i3c_master_ibi_upd_resp(struct nxp_i3c_master *master,
					u8 ibiresp)
{
	master->ibi.resp = ibiresp;
}

static void nxp_i3c_master_ibi_set_config(struct nxp_i3c_master *master,
					  u8 enable)
{
	master->ibi.enable = enable;
	if (enable) {
		master->interrupt_set &= MSTATUS_SLVSTART;
	} else {
		master->interrupt_set &= ~MSTATUS_SLVSTART;
		master->ibi.resp &= MCTRL_IBIRESP_ACKNOBYTE;
	}

	writel(master->ibi.rules, master->regs + IBIRULES_REG);
	nxp_i3c_master_set_interrupt_mask(master, 0);
}

static int nxp_i3c_master_bus_init(struct i3c_master_controller *m)
{
	struct nxp_i3c_master *master = to_nxp_i3c_master(m);
	// struct i3c_bus *bus = i3c_master_get_bus(m);
	struct i3c_device_info info = {};
	int err, status, ret;
	printk(KERN_INFO "NXP I3C : Bus initialization.\n");

	/* Configuration of BUS mode */
	/* TODO : hceck if IP configuration is needed from this value .*/
	/*switch (bus->mode) {
	case I3C_BUS_MODE_PURE:
		break;

	case I3C_BUS_MODE_MIXED_FAST:
		break;

	case I3C_BUS_MODE_MIXED_SLOW:
		break;

	case I3C_BUS_MODE_MIXED_LIMITED :
		brea;

	default:
		return -EINVAL;
	}*/

	/* First, stop the bus to know its status */
	master->bus_in_i3c = 0;
	master->bus_in_sdr = 0;
	master->bus_in_ddr = 0;

	err = nxp_i3c_master_write_stop(master);
	if (err < 0) {
		printk(KERN_ERR
		       "NXP I3C : unable to init MASTER IP, err = %d \n",
		       err);
	}

	/* configure fgpa */
	nxp_i3c_master_set_config(master);

	status = readl(master->regs + MSTATUS_REG);
	nxp_i3c_master_update_bus_status(master, status);

	/* Get an address for the master. */
	ret = i3c_master_get_free_addr(m, 0);
	if (ret < 0)
		return ret;

	writel((ret & GENMASK(6, 0)) << 1, master->regs + MDYNADDR_REG);

	memset(&info, 0, sizeof(info));
	info.dyn_addr = ret;

	ret = i3c_master_set_info(&master->base, &info);
	if (ret)
		return ret;

	nxp_i3c_master_ibi_init_rules(master);
	master->hj.mode = NXP_I3C_HOTJOIN_ACK;

	return 0;
}

static int nxp_i3c_master_send_empty_header(struct nxp_i3c_master *master)
{
	struct nxp_i3c_xfer *xfer;
	struct nxp_i3c_cmd *ccmd;
	int ret = 0;
	char *payload;

	printk(KERN_DEBUG "NXP I3C : Exec CCC broadcast command.\n");
	if (cmd->ndests != 1) {
		printk(KERN_ERR
		       "NXP I3C : error, wrong boardcast CCC command format broadcast but mot than 1 dest configureed.\n ");
		return -EINVAL;
	}

	xfer = nxp_i3c_master_xfer_alloc(master, 1);
	if (!xfer)
		return -ENOMEM;

	ccmd = xfer->cmds;
	ccmd->rnw = 0;
	ccmd->len = 0;
	ccmd->tx_buf = payload;
	ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
			 (I3C_BROADCAST_ADDR << MCTRL_ADDR_SHIFT) |
			 (MCTRL_TYPE_I3C << MCTRL_TYPE_SHIFT |
			 MCTRL_IBIRESP_WRITE( master->ibi.resp);

	nxp_i3c_master_xfer_queue(master, xfer);
	if (!wait_for_completion_timeout(&xfer->comp, msecs_to_jiffies(1000)))
		nxp_i3c_master_xfer_unqueue(master, xfer);

	ret = xfer->ret;
	cmd->err = nxp_i3c_master_get_err(&xfer->cmds[0]);
	nxp_i3c_master_xfer_free(xfer);

	kfree(payload);

	return ret;
}

static int nxp_i3c_master_ccc_broadcast(struct nxp_i3c_master *master,
					struct i3c_ccc_cmd *cmd)
{
	struct nxp_i3c_xfer *xfer;
	struct nxp_i3c_cmd *ccmd;
	int ret = 0;
	int payload_len = 1;
	char *payload, *data;

	printk(KERN_DEBUG "NXP I3C : Exec CCC broadcast command.\n");
	if (cmd->ndests != 1) {
		printk(KERN_ERR
		       "NXP I3C : error, wrong boardcast CCC command format broadcast but mot than 1 dest configureed.\n ");
		return -EINVAL;
	}

	xfer = nxp_i3c_master_xfer_alloc(master, 1);
	if (!xfer)
		return -ENOMEM;

	ccmd = xfer->cmds;
	ccmd->rnw = cmd->rnw;

	/*
	 * In case of CCC command type Direct (Set or Get) the CCC is sended as
	 * broadcast first with no data. So in this ccc_broadcast function,
	 * just check if the cmd is rwn or not.
	 * Depending on that, insert the data after the CCC byte
	 */
	if (cmd->id & I3C_CCC_DIRECT)
		payload_len += cmd->dests[0].payload.len;

	data = cmd->dests[0].payload.data;
	payload = kmalloc(payload_len * sizeof(char), GFP_KERNEL);
	payload[0] = (char)cmd->id;

	if (!(cmd->id & I3C_CCC_DIRECT))
		memcpy(payload + 1, data, cmd->dests[0].payload.len);

	ccmd->tx_buf = payload;
	ccmd->len = payload_len;

	ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
			 (I3C_BROADCAST_ADDR << MCTRL_ADDR_SHIFT) |
			 (MCTRL_TYPE_I3C << MCTRL_TYPE_SHIFT);

	nxp_i3c_master_xfer_queue(master, xfer);
	if (!wait_for_completion_timeout(&xfer->comp, msecs_to_jiffies(1000)))
		nxp_i3c_master_xfer_unqueue(master, xfer);

	ret = xfer->ret;
	cmd->err = nxp_i3c_master_get_err(&xfer->cmds[0]);
	nxp_i3c_master_xfer_free(xfer);

	kfree(payload);

	return ret;
}

static int nxp_i3c_master_ccc_cmd_get(struct nxp_i3c_master *master,
				      struct i3c_ccc_cmd *cmd)
{
	struct nxp_i3c_xfer *xfer;
	struct nxp_i3c_cmd *ccmd;
	int ret = 0;
	printk(KERN_DEBUG "NXP I3C : Exec CCC hget command.\n");

	/* First broadcast the CCC command */
	nxp_i3c_master_ccc_broadcast(master, cmd);

	xfer = nxp_i3c_master_xfer_alloc(master, 1);
	if (!xfer)
		return -ENOMEM;

	/* Followed by a start addr + read data action*/
	ccmd = xfer->cmds;
	ccmd->rnw = cmd->rnw;
	ccmd->rx_buf = cmd->dests[0].payload.data;
	ccmd->len = cmd->dests[0].payload.len;
	ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
			 (cmd->dests[0].addr << MCTRL_ADDR_SHIFT) | MCTRL_READ |
			 (MCTRL_TYPE_I3C << MCTRL_TYPE_SHIFT) |
			 MCTRL_RDTERM(ccmd->len);

	nxp_i3c_master_xfer_queue(master, xfer);
	if (!wait_for_completion_timeout(&xfer->comp, msecs_to_jiffies(1000)))
		nxp_i3c_master_xfer_unqueue(master, xfer);

	ret = xfer->ret;
	cmd->err = nxp_i3c_master_get_err(&xfer->cmds[0]);
	nxp_i3c_master_xfer_free(xfer);

	return ret;
}

static int nxp_i3c_master_ccc_direct(struct nxp_i3c_master *master,
				     struct i3c_ccc_cmd *cmd)
{
	struct nxp_i3c_xfer *xfer;
	struct nxp_i3c_cmd *ccmd;
	int ret = 0;

	printk(KERN_DEBUG "NXP I3C : Exec CCC direct command.\n");

	/* First broadcast the CCC command */
	nxp_i3c_master_ccc_broadcast(master, cmd);

	xfer = nxp_i3c_master_xfer_alloc(master, 1);
	if (!xfer)
		return -ENOMEM;

	/* Followed by a start addr + read data action*/
	ccmd = xfer->cmds;
	ccmd->rnw = cmd->rnw;
	ccmd->tx_buf = cmd->dests[0].payload.data;
	ccmd->len = cmd->dests[0].payload.len;
	ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
			 (cmd->dests[0].addr << MCTRL_ADDR_SHIFT) |
			 (MCTRL_TYPE_I3C << MCTRL_TYPE_SHIFT);

	nxp_i3c_master_xfer_queue(master, xfer);
	if (!wait_for_completion_timeout(&xfer->comp, msecs_to_jiffies(1000)))
		nxp_i3c_master_xfer_unqueue(master, xfer);

	ret = xfer->ret;
	cmd->err = nxp_i3c_master_get_err(&xfer->cmds[0]);
	nxp_i3c_master_xfer_free(xfer);

	return ret;
	/* TODO : check if any action to do for command SETDASA or SETNEWDA*/
}

static int nxp_i3c_master_send_ccc_cmd(struct i3c_master_controller *m,
				       struct i3c_ccc_cmd *cmd)
{
	struct nxp_i3c_master *master = to_nxp_i3c_master(m);
	int ret = 0;

	printk(KERN_DEBUG "NXP I3C : Exec CCC command.\n");

	/* If the command need to retrieve dat afrom device it's a Direct Get type */
	if (cmd->rnw) {
		ret = nxp_i3c_master_ccc_cmd_get(master, cmd);
	} else {
		if (cmd->id & I3C_CCC_DIRECT) {
			ret = nxp_i3c_master_ccc_direct(master, cmd);
		} else {
			ret = nxp_i3c_master_ccc_broadcast(master, cmd);
		}
	}

	return ret;
}

static int nxp_i3c_master_priv_xfers(struct i3c_dev_desc *dev,
				     struct i3c_priv_xfer *i3c_xfers,
				     int nxfers)
{
	struct i3c_master_controller *m = i3c_dev_get_master(dev);
	struct nxp_i3c_master *master = to_nxp_i3c_master(m);
	struct nxp_i3c_cmd *ccmd = NULL;
	struct nxp_i3c_xfer *nxp_xfer;
	unsigned i, ret = 0;

	if (!i3c_xfers)
		return 0;

	// if(nxfers > MASTER_CMD_FIFO_DEPH)
	// 	return -EOPNOTSUPP;

	for (i = 0; i < nxfers; i++) {
		nxp_xfer = nxp_i3c_master_xfer_alloc(master, 1);
		if (!nxp_xfer)
			return -ENOMEM;

		ccmd = &nxp_xfer->cmds[0];

		ccmd->rnw = i3c_xfers[i].rnw;
		ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
				 (dev->info.dyn_addr << MCTRL_ADDR_SHIFT) |
				 (MCTRL_TYPE_I3C << MCTRL_TYPE_SHIFT);

		ccmd->len = i3c_xfers[i].len;

		if (ccmd->rnw) {
			ccmd->cmd_ctrl |= MCTRL_READ;
			ccmd->cmd_ctrl |= MCTRL_RDTERM(ccmd->len);
			ccmd->rx_buf = i3c_xfers[i].data.in;
		} else {
			ccmd->tx_buf = i3c_xfers[i].data.out;
		}

		nxp_i3c_master_xfer_queue(master, nxp_xfer);
		if (!wait_for_completion_timeout(&nxp_xfer->comp,
						 msecs_to_jiffies(1000)))
			nxp_i3c_master_xfer_unqueue(master, nxp_xfer);

		ret = nxp_xfer->ret;
		i3c_xfers[i].err = nxp_i3c_master_get_err(&nxp_xfer->cmds[0]);
		nxp_i3c_master_xfer_free(nxp_xfer);

		if (!ret)
			break;
	}

	return ret;
}

static int nxp_i3c_master_i2c_xfers(struct i2c_dev_desc *dev,
				    const struct i2c_msg *i2c_xfers, int nxfers)
{
	struct i3c_master_controller *m = i2c_dev_get_master(dev);
	struct nxp_i3c_master *master = to_nxp_i3c_master(m);
	int i, ret = 0;
	struct nxp_i3c_xfer *nxp_xfer;
	struct nxp_i3c_cmd *ccmd = NULL;

	if (!i2c_xfers)
		return 0;

	for (i = 0; i < nxfers; i++) {
		nxp_xfer = nxp_i3c_master_xfer_alloc(master, 1);
		if (!nxp_xfer)
			return -ENOMEM;

		ccmd = &nxp_xfer->cmds[0];

		ccmd->cmd_ctrl = MCTRL_REQUEST_STARTADDR |
				 (i2c_xfers[i].addr << MCTRL_ADDR_SHIFT) |
				 (MCTRL_TYPE_I2C << MCTRL_TYPE_SHIFT);

		ccmd->len = i2c_xfers[i].len;

		if (i2c_xfers[i].flags & I2C_M_RD) {
			ccmd->rnw = 1; // set to True
			ccmd->cmd_ctrl |= MCTRL_READ;
			ccmd->cmd_ctrl |= MCTRL_RDTERM(ccmd->len);
			ccmd->rx_buf = i2c_xfers[i].buf;
		} else {
			ccmd->tx_buf = i2c_xfers[i].buf;
		}

		nxp_i3c_master_xfer_queue(master, nxp_xfer);
		if (!wait_for_completion_timeout(&nxp_xfer->comp,
						 msecs_to_jiffies(1000)))
			nxp_i3c_master_xfer_unqueue(master, nxp_xfer);

		ret = nxp_xfer->ret;
		nxp_i3c_master_xfer_free(nxp_xfer);

		if (!ret)
			break;
	}

	return ret;
}

static int nxp_i3c_master_do_daa(struct i3c_master_controller *m)
{
	struct nxp_i3c_master *master = to_nxp_i3c_master(m);
	unsigned i = 0;
	int ret, slot;
	u8 addrs[MAX_DEVS] = {};
	u8 last_addr = 0;
	static struct nxp_i3c_xfer *xfer;
	struct nxp_i3c_cmd *ccmd = NULL;
	/* Device Response Byte Number */
	unsigned DRBN = 8;
	char device_data_byte[DRBN];

	/* Protocol remainder :
	 * After receiving a CCC with ENTDAA, a slave device will send 8 bytes of data
	 * 15 bits : Manufacturer ID
	 * 1 bits : Provisioned id Type
	 * 32 bits : Provisioned id
	 * 8 bits : BCR (Bus Characteristics Register)
	 * 8 bits : DCR (Device Characteristic Register)
	 * If any of these bytes is missing the slace DAA cannot by processed
	 */
	xfer = nxp_i3c_master_xfer_alloc(master, 1);
	if (!xfer)
		return -ENOMEM;

	memset(device_data_byte, 0, DRBN * sizeof(char));
	/* exec DAA request and read the 8 bytes */
	ccmd = xfer->cmds;
	ccmd->rnw = 1;
	ccmd->rx_buf = device_data_byte;
	ccmd->len = DRBN;
	ccmd->cmd_ctrl = MCTRL_REQUEST_PROCESSDAA;

	nxp_i3c_master_xfer_queue(master, xfer);
	if (!wait_for_completion_timeout(&xfer->comp, msecs_to_jiffies(1000)))
		nxp_i3c_master_xfer_unqueue(master, xfer);
	ret = xfer->ret;
	// cmd->err = nxp_i3c_master_get_err(&xfer->cmds[0]);

	while (!(xfer->status & MSTATUS_COMPLETE)) {
		if (xfer->status & MSTATUS_NACKED) {
			nxp_i3c_master_write_stop(master);
			return -ENOMSG;
		}

		if (xfer->cmds[0].len != DRBN) {
			printk(KERN_ERR
			       "NXP I3C : DAA error. Waited %d byte from slave device but %d byte received.\n",
			       DRBN, xfer->cmds[0].len);
			continue;
		}

		ret = i3c_master_get_free_addr(m, last_addr + 1);
		if (ret < 0)
			return -ENOSPC;

		last_addr = ret;
		addrs[slot] = last_addr;

		/* Write the new DA for the slave */
		writel(last_addr, master->regs + MWDATAB_REG);

		/* Reinit the xfer frame & rx payload */
		ccmd->len = DRBN;
		memset(device_data_byte, 0, DRBN);
		/* Process a new DAA Request to Apply the new slave DA
		 * It will launch new slave processing */
		nxp_i3c_master_xfer_queue(master, xfer);
		if (!wait_for_completion_timeout(&xfer->comp,
						 msecs_to_jiffies(1000)))
			nxp_i3c_master_xfer_unqueue(master, xfer);
		slot++;
	}

	ret = readl(master->regs + MDATACTRL_REG);
	if (ret & DATACTRL_RXEMPTY)

		nxp_i3c_master_xfer_free(xfer);

	for (i = 0; i < slot; i++)
		i3c_master_add_i3c_dev_locked(m, addrs[i]);

	i3c_master_defslvs_locked(&master->base);

	return 0;
}

static void nxp_i3c_master_slvreq(struct work_struct *work)
{
	struct nxp_i3c_master *master =
		container_of(work, struct nxp_i3c_master, ibi.slvreq_work);

	nxp_i3c_master_send_empty_header(master);
}

static void nxp_i3c_master_ibi_hj(struct work_struct *work)
{
	struct nxp_i3c_master *master =
		container_of(work, struct nxp_i3c_master, ibi.slvreq_work);

	printk(KERN_INFO "I3C NXP : Receive HotJoin.\n");
	switch (master->hj.mode) {
	case NXP_I3C_HOTJOIN_ACK:
		writel(MCTRL_REQUEST_IBIACKNACK | MCTRL_IBIRESP_ACKNOBYTE,
		       master->regs + MCTRL_REG);
		break;
	case NXP_I3C_HOTJOIN_NACK:
		writel(MCTRL_REQUEST_IBIACKNACK | MCTRL_IBIRESP_NACK,
		       master->regs + MCTRL_REG);
		break;
	case NXP_I3C_HOTJOIN_MANUAL:
		writel(NXP_I3C_HOTJOIN_ACK);
		break;
	}

	nxp_i3c_master_write_stop(master);

	// save the event in the FIFO
	my_device_data.ibi_event->type = NXP_I3C_IBITYPE_HJ;
	my_device_data.ibi_event->wait_for_manual_action = 0;
	my_device_data.ibi_event->len = 0;
	memset(my_device_data.ibi_event->data, 0, len * sizeof(__u8));
	my_device_data.ibi_event->dyn_addr = NXP_I3C_HOTJOIN_ADDR;

	kfifo_in(&(my_device_data.evt_fifo), my_device_data.ibi_event, 1);

	if ((my_device_data.task != NULL) && kfifo_empty) {
		memset(&info, 0, sizeof(struct siginfo));
		info.si_signo = SIGHJ;
		info.si_code = SI_QUEUE;
		if (send_sig_info(SIGIBI, &info, my_device_data.task) < 0) {
			printk(KERN_INFO "I3C NXP: "
					 "Could not send signal to user\n");
		}
	}

	err = i3c_master_do_daa(&master->base);
	if (err)
		printk(KERN_ERR
		       "I3C NXP : Receive HotJoin, unable to perform DAA.\n");
}

static int nxp_i3c_master_ibi_read(struct nxp_i3c_master *master, u8 addr)
{
	unsigned int status, idx, kfifo_empty;
	u8 read_data[255];
	struct siginfo info;

	kfifo_empty = kfifo_is_empty(&(dev_data.evt_fifo));

	for(status = readl(master->regs + MCTRL_REG), idx=0;
		(status & MSTATUS_RXPEND) & !(status & MSTATUS_ERRWARN);
		status = readl(master->regs + MCTRL_REG, idx++){
		read_data[idx] = readl(master->reg + MRDATAB_REG);
	}

	if( status & MSTATUS_ERRWARN)
		nxp_i3c_master_process_error(master);

	// save the event in the FIFO
	my_device_data.ibi_event->type = NXP_I3C_IBITYPE_IBI;
	my_device_data.ibi_event->wait_for_manual_action = 0;
	my_device_data.ibi_event->len = idx;
	memset(my_device_data.ibi_event->data, 0 , len * sizeof(__u8));
	memcpy(my_device_data.ibi_event->data, read_data, len);
	my_device_data.ibi_event->dyn_addr = addr;

	kfifo_in(&(my_device_data.evt_fifo), my_device_data.ibi_event, 1);

	if ((my_device_data.task != NULL) && kfifo_empty) {
		memset(&info, 0, sizeof(struct siginfo));
		info.si_signo = SIGIBI;
		info.si_code = SI_QUEUE;
		if (send_sig_info(SIGIBI, &info, my_device_data.task) < 0) {
			printk(KERN_INFO "I3C NXP: "
					 "Could not send signal to user\n");
		}
	}

	nxp_i3c_master_clear_status(master);
	nxp_i3c_master_flush_buffers(master);
}

/*
 * This function must be called after an IBIWON interruption to process the IBI
 * (Detect if it's a HotJoin Or Data IBI)
 */
static int nxp_i3c_master_ibi_demux(struct nxp_i3c_master *master,
				    unsigned status)
{
	unsigned err = 0;
	u8 addr = = MSTATUS_IBIADDR(status);

	/* First wait for the COMPLETE status */
	err = nxp_i3c_master_trigger_cmd(master, MSTATUS_COMPLETE);
	if (!err)
		return err;

	switch (MSTATUS_IBITYPE(status)) {
	case MSTATUS_IBITYPE_IBI:
		/*
		 * Here read the data if the IBI is an IBI with byte
		 */
		master->ibi.addr = addr;
		err = nxp_i3c_master_ibi_read(master, addr);
		break;
	/* Reception of Master Request from a secondary master on the bus
	 * Actually not compatible with this feature, so will always repond nack*/
	case MSTATUS_IBITYPE_MR:
		/* TODO : include support for secondary Master on I3C bus */
		writel(MCTRL_REQUEST_IBIACKNACK |
			       MCTRL_IBIRESP_WRITE(MCTRL_IBIRESP_NACK),
		       master->base + MCTRL_REG);
		break;
	case MSTATUS_IBITYPE_HJ:
		if (addr == NXP_I3C_HOTJOIN_ADDR)
			queue_work(master->base.wq, &master->hj.hjwork);
		break;
	case MSTATUS_IBITYPE_NONE:
	default:
		break;
	}

	return 0;
}

static int nxp_i3c_master_process_error(struct nxp_i3c_master *master)
{
	int err;

	err = readl(master->regs + MERRWARN_REG);

	if (err == MERRWARN_NACK) {
		nxp_i3c_master_flush_buffers(master);
		nxp_i3c_master_clear_status(master);
	}

	nxp_i3c_master_clear_err_warn(master);

	return err;
}

irqreturn_t nxp_i3c_master_interrupt(int irq, void *data)
{
	struct nxp_i3c_master *master = data;
	u32 interrupt_status, status;

	interrupt_status = readl(master->regs + MINTMASKED_REG);
	status = readl(master->regs + MSTATUS_REG);
	/* confirm that the catched IRQ come from a followed status flag */
	if (!(interrupt_status & readl(master->regs + MINTSET_REG)))
		return IRQ_NONE;

	/* Request by a slave to send a start & allow the reception of IBI */
	if (interrupt_status & MSTATUS_SLVSTART) {
		// send a Start to the bus to offer the possibility to the slave of
		// process an IBI
		queue_work(master->base.wq, &master->ibi.slvreq_work);
	} else if (interrupt_status & MSTATUS_IBIWON) {
		nxp_i3c_master_ibi_demux(master, status);
	} else {
		spin_lock(&master->xferqueue.lock);
		nxp_i3c_master_xfer_end_locked(master, interrupt_status);
		spin_unlock(&master->xferqueue.lock);
	}

	return IRQ_HANDLED;
}

/*
 *   -------------------------------------------------------------
 *  Bloc containing function for nxp i3c char device
 *   -------------------------------------------------------------
 */

static int nxp_i3c_chardev_open(struct inode *inode, struct file *file)
{
	printk(KERN_INFO "I3C NXP: Device opened");
	// if (!atomic_dec_and_test(&dev_data.available)) {
	// 	atomic_inc(&dev_data.available);
	// 	return -EBUSY;
	// }
	if (!atomic_dec_and_test(&my_device_data.available)) {
		atomic_inc(&my_device_data.available);
		return -EBUSY;
	}
	return 0;
}

static struct i3c_dev_desc *
nxp_i3c_tool_search_i3c_dev(struct nxp_i3c_master *master, u8 dyn_addr)
{
	struct i3c_dev_desc *i3cdev;
	i3c_bus_for_each_i3cdev (&master->base.bus, i3c_dev) {
		if (i3cdev->info.dyn_addr == dyn_addr) {
			return i3cdev;
		}
	}
	return NULL;
}

static struct i2c_dev_desc *
nxp_i3c_tool_search_i2c_dev(struct nxp_i3c_master *master, u8 static_addr)
{
	struct i2c_dev_desc *i2cdev;
	i3c_bus_for_each_i2cdev (&master->base.bus, i2cdev) {
		if (i2cdev->addr == static_addr) {
			return i2cdev;
		}
	}
	return NULL;
}

static int nxp_i3c_tool_write_i3c_buffer(struct nxp_i3c_master *master,
					 i3c_cmd_param *cmd_params)
{
	struct i3c_dev_desc *i3cdev;
	struct i3c_priv_xfer *xfers;
	int err = 0;
	i3cdev = nxp_i3c_tool_search_i3c_dev(master, cmd_params->dyn_addr);
	if (i3cdev == NULL)
		return -EINVAL;

	// First control if the device is known by the bus
	xfers = kmalloc(sizeof(struct i3c_priv_xfer), GFP_KERNEL);
	xfers->rnw = 0;
	xfers->len = cmd_params->buffer_len;
	xfers->data.out = cmd_params->write_data;
	xfers->err = 0;
	err = i3c_device_do_priv_xfers(i3cdev->dev, xfers, 1);

	kfree(xfers);

	return err;
}

static int nxp_i3c_tool_read_i3c_buffer(struct nxp_i3c_master *master,
					i3c_cmd_param *cmd_params,
					u8 *read_buffer)
{
	struct i3c_dev_desc *i3cdev;
	struct i3c_priv_xfer *xfers;
	int err = 0;
	i3cdev = nxp_i3c_tool_search_i3c_dev(master, cmd_params->dyn_addr);
	if (i3cdev == NULL)
		return -EINVAL;

	// First control if the device is known by the bus
	xfers = kmalloc(sizeof(struct i3c_priv_xfer), GFP_KERNEL);
	xfers->rnw = 1;
	xfers->len = cmd_params->buffer_len;
	xfers->data.in = read_buffer;
	xfers->err = 0;
	err = i3c_device_do_priv_xfers(i3cdev->dev, xfers, 1);
	kfree(xfers);
	return err;
}

static int nxp_i3c_tool_write_i2c_buffer(struct nxp_i3c_master *master,
					 i3c_cmd_param *cmd_params)
{
	struct i2c_dev_desc *i2cdev;
	struct i2c_msg *i2c_msg;
	int err = 0;
	i2cdev = nxp_i3c_tool_search_i2c_dev(master, cmd_params->static_addr);
	if (i2cdev == NULL)
		return -EINVAL;

	i2c_msg = kmalloc(sizeof(struct i2c_msg), GFP_KERNEL);
	i2c_msg->addr = cmd_params->static_addr;
	/* TODO Check if any I2C flag flag management is necessary*/
	i2c_msg->flags = 0;
	i2c_msg->buf = cmd_params->write_data;
	i2c_msg->len = cmd_params->buffer_len;
	err = master->base.i2c.algo->master_xfer(&master->base.i2c, i2c_msg, 1);
	// err = i3c_master_i2c_adapter_xfer(i2c_msg, master->base.i2c, 1);
	kfree(i2c_msg);
	return err;
}

static int nxp_i3c_tool_read_i2c_buffer(struct nxp_i3c_master *master,
					i3c_cmd_param *cmd_params,
					u8 *read_buffer)
{
	struct i2c_dev_desc *i2cdev;
	struct i2c_msg *i2c_msg;
	int err = 0;
	i2cdev = nxp_i3c_tool_search_i2c_dev(master, cmd_params->static_addr);
	if (i2cdev == NULL)
		return -EINVAL;

	i2c_msg = kmalloc(sizeof(struct i2c_msg), GFP_KERNEL);
	i2c_msg->addr = cmd_params->static_addr;
	/* TODO Check if any I2C flag flag management is necessary*/
	i2c_msg->flags = I2C_M_RD;
	i2c_msg->buf = read_buffer;
	i2c_msg->len = cmd_params->buffer_len;
	err = master->base.i2c.algo->master_xfer(&master->base.i2c, i2c_msg, 1);
	// err = i3c_master_i2c_adapter_xfer(i2c_msg, master->base.i2C, 1);
	kfree(i2c_msg);
	return err;
}

static int nxp_i3c_chardev_release(struct inode *inode, struct file *file)
{
	printk(KERN_INFO "I3C NXP: Device closed\n");
	atomic_inc(&my_device_data.available);
	return 0;
}

static ssize_t nxp_i3c_chardev_read(struct file *file, char *buffer, size_t len,
				    loff_t *offset)
{
	/* TODO Write 'read' procedure for I3C */
	printk(KERN_INFO "I3C NXP: Device read");
	return 0;
}

static ssize_t nxp_i3c_chardev_write(struct file *file, const char *buffer,
				     size_t len, loff_t *offset)
{
	/* TODO Write 'write' procedure for I3C */
	printk(KERN_INFO "I3C NXP: Device write");
	return 0;
}

static long nxp_i3c_chardev_ioctl(struct file *file, unsigned int cmd,
				  unsigned long arg)
{
	int err = 0;
	struct nxp_i3c_master *master = my_device_data.master;
	struct siginfo info;
	i3c_cmd_param cmd_params;
	i3c_config config_params;
	i3c_cmd_ccc_param cmd_ccc_param;
	struct i3c_dev_desc *i3cdev, *i3ctmp;
	// struct i2c_dev_desc *i2cdev, *i2ctmp;
	struct nxp_i3c_cmd *ccmd;
	u8 buffer[255];
	i3c_devices_list i3c_dev_list;
	struct i3c_device_info devices_info[16];
	int idx;
	struct i3c_ccc_cmd i3c_ccc_cmd;
	struct ibi_event ibi_event;
	u32 evt_count;
	memset(buffer, 0, 255);

	printk(KERN_INFO "I3C NXP: Device ioctl");

	/*
	 * The first step is the conversion of userland to kernel-land data
	 * depending on command type
	 */
	switch (cmd) {
	case NXP_I3C_CMD_CONF:
		err = copy_from_user(&config_params, (i3c_config *)arg,
				     sizeof(config_params));
		break;
	case NXP_I3C_CMD_CCC:
		err = copy_from_user(&cmd_ccc_param, (i3c_cmd_ccc_param *)arg,
				     sizeof(i3c_cmd_ccc_param));
		break;
	case NXP_I3C_CMD_GET_EVENT:
		err = copy_from_user(&ibi_event, (struct ibi_event *)arg,
				     sizeof(struct ibi_event));
		break;
	default:
		err = copy_from_user(&cmd_params, (i3c_cmd_param *)arg,
				     sizeof(i3c_cmd_param));
		break;
	}

	if (err < 0) {
		printk(KERN_ERR
		       "I3X NXP : Fail to copy parameters from user.\n");
		return err;
	}

	/*
	 * Second step, execute the command
	 */
	switch (cmd) {
	case NXP_I3C_CMD_CONF:
		memcpy(&master->conf, &config_params, sizeof(master->conf));
		nxp_i3c_master_set_config(master);
		break;
	case NXP_I3C_CMD_DAA:
		err = i3c_master_do_daa(&master->base);
		break;
	case NXP_I3C_CMD_WI3C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode == NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_write_i3c_buffer(master, &cmd_params);
		break;
	case NXP_I3C_CMD_RI3C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode == NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_read_i3c_buffer(master, &cmd_params, buffer);
		break;
	case NXP_I3C_CMD_WR3C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode == NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_write_i3c_buffer(master, &cmd_params);
		if (!err)
			err = nxp_i3c_tool_read_i3c_buffer(master, &cmd_params,
							   buffer);
		break;
	case NXP_I3C_CMD_WI2C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode != NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_write_i2c_buffer(master, &cmd_params);
		break;
	case NXP_I3C_CMD_RI2C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode != NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_read_i2c_buffer(master, &cmd_params, buffer);
		break;
	case NXP_I3C_CMD_WR2C:
		if (cmd_params.buffer_len < 1)
			break;

		if (cmd_params.mode != NXP_I3C_MODE_I2C)
			break;

		err = nxp_i3c_tool_write_i2c_buffer(master, &cmd_params);
		if (!err)
			err = nxp_i3c_tool_read_i2c_buffer(master, &cmd_params,
							   buffer);
		break;
	case NXP_I3C_CMD_STOP:
		err = nxp_i3c_master_write_stop(master);
		break;
	case NXP_I3C_CMD_EXIT:
		err = nxp_i3c_master_write_exit(master);
		break;
	case NXP_I3C_CMD_GET_DEVICE_LIST:
		idx = 0;
		list_for_each_entry_safe (i3cdev, i3ctmp,
					  &master->base.bus.devs.i3c,
					  common.node) {
			devices_info[idx++] = i3cdev->info;
		}
		i3c_dev_list.nb_devices = idx;
		i3c_dev_list.device_list = devices_info;
		err = copy_to_user(cmd_params.i3c_devices_list, &i3c_dev_list,
				   sizeof(i3c_dev_list));
		break;
	case NXP_I3C_CMD_CCC:
		i3c_ccc_cmd.rnw = cmd_ccc_param.read;
		i3c_ccc_cmd.id = cmd_ccc_param.cmd_id;
		i3c_ccc_cmd.ndests = 1;
		i3c_ccc_cmd.err = 0;
		i3c_ccc_cmd.dests[0].addr = cmd_ccc_param.dest_addr;
		i3c_ccc_cmd.dests[0].payload.len = cmd_ccc_param.payload_len;
		if (cmd_ccc_param.read)
			i3c_ccc_cmd.dests[0].payload.data = buffer;
		else
			i3c_ccc_cmd.dests[0].payload.data =
				cmd_ccc_param.write_payload;
		err = nxp_i3c_master_send_ccc_cmd(&master->base, &i3c_ccc_cmd);

		if (!err && cmd_ccc_param.read) {
			err = copy_to_user(cmd_ccc_param.read_payload, buffer,
					   sizeof(buffer));
		}
		break;
	case NXP_I3C_CMD_SET_HJ_MODE:
		master->hj.mode = cmd_params.mode;
		break;
	case NXP_I3C_CMD_GET_EVENT:

		err = kfifo_to_user(&(my_device_data.evt_fifo), ibi_event,
				    sizeof(struct ibi_event), &(evt_count));
		if (err) {
			printk(KERN_INFO "I3C NXP: kfifo_to_user "
					 "returns: err = %d, copied = %d\n",
			       err, evt_count);
		}
		break;
	case NXP_I3C_CMD_SET_TASK:
		if (arg) {
			my_device_data.task = get_current();
			if (my_device_data.task != NULL) {
				printk(KERN_INFO "I3C NXP: "
						 "Registered user application"
						 "task, PID = %d\n",
				       my_device_data.task->pid);
				err = 0;
			} else {
				printk(KERN_ALERT
				       "I3C NXP: "
				       "Could not register application task\n");
				err = -1;
			}
		} else {
			my_device_data.task = NULL;
			err = 0;
			printk(KERN_INFO
			       "I3C NXP: "
			       "Unregistered user application task\n");
		}
		break;
	default:
		err = -EPERM;
	}

	if (!err && (cmd == NXP_I3C_CMD_RI3C || cmd == NXP_I3C_CMD_WR3C ||
		     cmd == NXP_I3C_CMD_RI2C || cmd == NXP_I3C_CMD_WR2C)) {
		err = copy_to_user(cmd_params.read_data, buffer,
				   cmd_params.buffer_len);
		if (err < 0)
			printk(KERN_ERR
			       "I3X NXP : Failed to copy read data to user err = %d\n",
			       err);
	}

	return err;
}

// Definition of actions linked to the char device exposed to the userspace
static struct file_operations fops = {
	.open = nxp_i3c_chardev_open,
	.read = nxp_i3c_chardev_read,
	.write = nxp_i3c_chardev_write,
	.unlocked_ioctl = nxp_i3c_chardev_ioctl,
	.release = nxp_i3c_chardev_release,
};
/*
 *   ------------------- BLOC END --------------------------
 */

// Link with device_tree definition of I3C entry
static const struct of_device_id nxp_i3c_master_of_match[] = {
	{
		.compatible = "nxp,nxp-i3c-nfcc",
	},
	{},
};
MODULE_DEVICE_TABLE(of, nxp_i3c_master_of_match);

// Properly stop & destruct i3c driver and its ressources
static int nxp_i3c_driver_remove(struct platform_device *op)
{
	struct nxp_i3c_master *master = platform_get_drvdata(op);
	int ret;

	ret = i3c_master_unregister(&master->base);
	if (ret)
		return ret;

	device_destroy(i3c_nxp_class, MKDEV(major_number, 0));
	unregister_chrdev(major_number, DEVICE_NAME);
	printk(KERN_INFO "I3C NXP: Destroying the character device\n");
	return 0;
}

static const struct i3c_master_controller_ops nxp_i3c_master_ops = {
	.bus_init = nxp_i3c_master_bus_init,
	// .bus_cleanup = cdns_i3c_master_bus_cleanup,
	.do_daa = nxp_i3c_master_do_daa,
	// .attach_i3c_dev = cdns_i3c_master_attach_i3c_dev,
	// .reattach_i3c_dev = cdns_i3c_master_reattach_i3c_dev,
	// .detach_i3c_dev = cdns_i3c_master_detach_i3c_dev,
	// .attach_i2c_dev = cdns_i3c_master_attach_i2c_dev,
	// .detach_i2c_dev = cdns_i3c_master_detach_i2c_dev,
	// .supports_ccc_cmd = cdns_i3c_master_supports_ccc_cmd,
	.send_ccc_cmd = nxp_i3c_master_send_ccc_cmd,
	.priv_xfers = nxp_i3c_master_priv_xfers,
	.i2c_xfers = nxp_i3c_master_i2c_xfers,
	// .enable_ibi = cdns_i3c_master_enable_ibi,
	// .disable_ibi = cdns_i3c_master_disable_ibi,
	// .request_ibi = cdns_i3c_master_request_ibi,
	// .free_ibi = cdns_i3c_master_free_ibi,
	// .recycle_ibi_slot = cdns_i3c_master_recycle_ibi_slot,
};

// initialisation of i3c driver ( hardware link and userspace link)
static int nxp_i3c_driver_probe(struct platform_device *pdev)
{
	struct nxp_i3c_master *master;
	struct resource *res;
	int ret, rc;

	printk(KERN_INFO "I3C NXP: Initializing the character device\n");
	/* Dynamically allocate the major number */
	major_number = register_chrdev(0, DEVICE_NAME, &fops);
	if (major_number < 0) {
		printk(KERN_ALERT "I3C NXP: Failed to register major number\n");
		return major_number;
	}

	/* Register the device driver */
	i3c_nxp_device = device_create(
		i3c_nxp_class, NULL, MKDEV(major_number, 0), NULL, DEVICE_NAME);
	if (IS_ERR(i3c_nxp_device)) {
		unregister_chrdev(major_number, DEVICE_NAME);
		printk(KERN_ALERT "I3C NXP: Failed to create the device\n");
		return PTR_ERR(i3c_nxp_device);
	}

	master = devm_kzalloc(&pdev->dev, sizeof(*master), GFP_KERNEL);
	if (!master)
		return -ENOMEM;

	/* Get register address from device tree */
	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	master->regs = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(master->regs)) {
		device_destroy(i3c_nxp_class, MKDEV(major_number, 0));
		unregister_chrdev(major_number, DEVICE_NAME);
		printk(KERN_ALERT "I3C NXP: Failed get HW registers\n");
		return -ENXIO;
	}
	printk(KERN_INFO "I3C NXP: Master register address: %p\n",
	       master->regs);

	spin_lock_init(&master->xferqueue.lock);
	INIT_LIST_HEAD(&master->xferqueue.list);

	master->hj.mode = NXP_I3C_HOTJOIN_ACK;
	INIT_WORK(&master->hj.hjwork, nxp_i3c_master_ibi_hj);
	INIT_WORK(&master->ibi.slvreq_work, nxp_i3c_master_slvreq);

	/* Get interrupt line from device tree */
	my_device_data.irq = irq_of_parse_and_map(pdev->dev.of_node, 0);
	rc = devm_request_irq(&pdev->dev, my_device_data.irq,
			      nxp_i3c_master_interrupt, 0, DEVICE_NAME,
			      &master);
	if (rc) {
		device_destroy(i3c_nxp_class, MKDEV(major_number, 0));
		unregister_chrdev(major_number, DEVICE_NAME);
		printk(KERN_INFO "I3C NXP: Failed to register IRQ  handler.\n");
		return -1;
	}
	printk(KERN_INFO "I3C NXP: Registered IRQ number: %d\n",
	       my_device_data.irq);

	my_device_data.task = NULL;

	platform_set_drvdata(pdev, master);
	dev_set_drvdata(i3c_nxp_device, master);
	ret = i3c_master_register(&master->base, &pdev->dev,
				  &nxp_i3c_master_ops, false);

	my_device_data.ibi_event =
		devm_kzalloc(&pdev->dev, sizeof(struct ibi_event), GFP_ATOMIC);

	if (!my_device_data.ibi_event) {
		printk(KERN_ALERT "I3C NXP: Couldn't allocate memory\n");
		return -ENOMEM;
	}

	/* enable the SLVSTART interruption catch to detect slave IBI request */
	nxp_i3c_master_ibi_upd_rules(IBIRULES_NOBYTE | IBIRULES_MSB0);
	master->ibi.enable = 1; // Enable IBI interruption & response

	my_device_data.master = master;
	// Set the Flag for availability control a driver open call
	atomic_set(&my_device_data.available, 1);

	if (ret)
		printk(KERN_ERR "I3C NXP : unable to tegister the master.\n");

	return ret;
}

static struct platform_driver nxp_i3c_driver = {
	.probe = nxp_i3c_driver_probe,
	.remove = nxp_i3c_driver_remove,
	.driver =
		{
			.name = DRIVER_NAME,
			.of_match_table = of_match_ptr(nxp_i3c_master_of_match),
			.owner = THIS_MODULE,
		},
};

static int __init nxp_i3c_module_init(void)
{
	int retval;

	printk(KERN_INFO "I3C NXP: Initializing the character driver\n");

	/* Register the device class */
	i3c_nxp_class = class_create(THIS_MODULE, CLASS_NAME);
	if (IS_ERR(i3c_nxp_class)) {
		class_destroy(i3c_nxp_class);
		printk(KERN_ALERT "I3C NXP: "
				  "Failed to create the device class\n");
		return PTR_ERR(i3c_nxp_class);
	}

	retval = platform_driver_register(&nxp_i3c_driver);
	if (retval) {
		class_destroy(i3c_nxp_class);
		printk(KERN_ALERT "I3C NXP: "
				  "Failed to register platform driver\n");
		return retval;
	}

	return 0;
}

static void __exit nxp_i3c_module_exit(void)
{
	class_unregister(i3c_nxp_class);
	class_destroy(i3c_nxp_class);
	platform_driver_unregister(&nxp_i3c_driver);
	printk(KERN_INFO "I3C NXP: Exiting the character driver.");
}

module_init(nxp_i3c_module_init);
module_exit(nxp_i3c_module_exit);
