/***************************************************************************************************
 *  I2C link layer for the NXP NCI driver
 ***************************************************************************************************/
#include <linux/...>

struct nxp_nci_i2c_phy {
	struct i2c_client *i2c_dev;
	struct nci_dev *ndev;
	unsigned int gpio_en;
};

int nxp_nci_i2c_enable(void *phy_id) {
   struct nxp_nci_i2c_phy *phy = phy_id;
   gpio_set_value(phy->gpio_en, 1);
	return 0;
}

int nxp_nci_i2c_disable(void *phy_id) {
   struct nxp_nci_i2c_phy *phy = phy_id;
   gpio_set_value(phy->gpio_en, 0);
	return 0;
}

static int nxp_nci_i2c_write(void *phy_id, struct sk_buff *skb) {
	struct i2c_client *client = phy_id->i2c_dev;

	return i2c_master_send(client, skb->data, skb->len);
}

static struct nxp_nci_phy_ops i2c_phy_ops = {
	.enable = nxp_nci_i2c_enable,
	.disable = nxp_nci_i2c_disable,
	.write = nxp_nci_i2c_write,
};

static int nxp_nci_i2c_nci_read(struct nxp_nci_i2c_phy *phy,
				struct sk_buff **skb) {
	struct nci_ctrl_hdr header; /* May actually be a data header */
	struct i2c_client *client = phy->i2c_dev; int r;

	r = i2c_master_recv(client, (u8 *) &header, NCI_CTRL_HDR_SIZE); if (r < 0) {...}
	*skb = alloc_skb(NCI_CTRL_HDR_SIZE + header.plen, GFP_KERNEL);
	if (*skb == NULL) {...}
	memcpy(skb_put(*skb, NCI_CTRL_HDR_SIZE), (void *) &header, NCI_CTRL_HDR_SIZE);
	r = i2c_master_recv(client, skb_put(*skb, header.plen), header.plen); if (r != header.plen) {...}
	return r;
}

static irqreturn_t nxp_nci_i2c_irq_thread_fn(int irq, void *phy_id) {
	struct nxp_nci_i2c_phy *phy = phy_id;
	struct i2c_client *client = phy->i2c_dev;
	struct nxp_nci_info *info;
	struct sk_buff *skb = NULL;

	info = nci_get_drvdata(phy->ndev);
	nxp_nci_i2c_nci_read(phy, &skb);
	nci_recv_frame(phy->ndev, skb);
	return IRQ_HANDLED;
}

static int nxp_nci_i2c_probe(struct i2c_client *client,
			    const struct i2c_device_id *id)
{
	struct nxp_nci_i2c_phy *phy;
	struct nxp_nci_nfc_platform_data *pdata;

	phy = devm_kzalloc(&client->dev, sizeof(struct nxp_nci_i2c_phy), GFP_KERNEL);
	phy->i2c_dev = client;
	i2c_set_clientdata(client, phy);
	pdata = client->dev.platform_data;
	phy->gpio_en = pdata->gpio_en;
	client->irq = pdata->irq;

	r = nxp_nci_probe(phy, &client->dev, &i2c_phy_ops, NXP_NCI_I2C_MAX_PAYLOAD, &phy->ndev);
	if (r < 0) {...}
	r = request_threaded_irq(client->irq, NULL,
				 nxp_nci_i2c_irq_thread_fn,
				 IRQF_TRIGGER_RISING | IRQF_ONESHOT,
				 NXP_NCI_I2C_DRIVER_NAME, phy);
	if (r < 0) {...}
	return r;
}

static int nxp_nci_i2c_remove(struct i2c_client *client) {
	struct nxp_nci_i2c_phy *phy = i2c_get_clientdata(client);
	nxp_nci_remove(phy->ndev);
	gpio_free(phy->gpio_en);
	free_irq(client->irq, phy);
	return 0;
}

static struct i2c_driver nxp_nci_i2c_driver = {
	.probe = nxp_nci_i2c_probe,
	.id_table = nxp_nci_i2c_id_table,
	.remove = nxp_nci_i2c_remove,
};

module_i2c_driver(nxp_nci_i2c_driver);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("I2C driver for NXP NCI NFC controllers");
MODULE_AUTHOR("Andre Lepine <andre.lepine@nxp.com>");

/***************************************************************************************************
 *  Generic driver for NXP NCI NFC chips
 ***************************************************************************************************/

#include <linux/....h>

static int nxp_nci_open(struct nci_dev *ndev) {
   struct nxp_nci_info *info = nci_get_drvdata(ndev); int r = 0;

   mutex_lock(&info->info_lock);
   r = info->phy_ops->enable(info->phy_id); if (r < 0) {...}
   mutex_unlock(&info->info_lock);
   return r;
}

static int nxp_nci_close(struct nci_dev *ndev) {
   struct nxp_nci_info *info = nci_get_drvdata(ndev); int r = 0;

   mutex_lock(&info->info_lock);
   r = info->phy_ops->disable(info->phy_id); if (r < 0) {...}
   mutex_unlock(&info->info_lock);
   return r;
}

static int nxp_nci_send(struct nci_dev *ndev, struct sk_buff *skb) {
   struct nxp_nci_info *info = nci_get_drvdata(ndev); int r;

   r = info->phy_ops->write(info->phy_id, skb); if (r < 0) {...}
   return r;
}

static struct nci_ops nxp_nci_ops = {
   .open = nxp_nci_open,
   .close = nxp_nci_close,
   .send = nxp_nci_send,
};

int nxp_nci_probe(void *phy_id, struct device *pdev,
        struct nxp_nci_phy_ops *phy_ops, unsigned int max_payload,
        struct nci_dev **ndev)
{
   struct nxp_nci_info *info; int r;

   info = devm_kzalloc(pdev, sizeof(struct nxp_nci_info), GFP_KERNEL);
   info->phy_id = phy_id;
   info->pdev = pdev;
   info->phy_ops = phy_ops;
   info->max_payload = max_payload;
   INIT_WORK(&info->fw_info.work, nxp_nci_fw_work);
   init_completion(&info->fw_info.cmd_completion);
   mutex_init(&info->info_lock);

   if (info->phy_ops->disable) {
      r = info->phy_ops->disable(info->phy_id); if (r < 0) {...}
   }

   info->ndev = nci_allocate_device(&nxp_nci_ops, NXP_NCI_NFC_PROTOCOLS,
                NXP_NCI_HDR_LEN, 0);
   if (!info->ndev) return -ENOMEM;

   nci_set_parent_dev(info->ndev, pdev);
   nci_set_drvdata(info->ndev, info);
   r = nci_register_device(info->ndev); if (r < 0) nci_free_device(info->ndev);
   *ndev = info->ndev;
   return r;
}

void nxp_nci_remove(struct nci_dev *ndev) {
   struct nxp_nci_info *info = nci_get_drvdata(ndev);

   cancel_work_sync(&info->fw_info.work);
   mutex_lock(&info->info_lock);
   info->phy_ops->disable(info->phy_id);
   nci_unregister_device(ndev);
   nci_free_device(ndev);
   mutex_unlock(&info->info_lock);
}

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("NXP NCI NFC driver");
MODULE_AUTHOR("Andre Lepine <andre.lepine@nxp.com>");

/***************************************************************************************************
 *  Generic driver for NXP NCI NFC chips
 ***************************************************************************************************/
#include <linux/...>

static int nxp_nci_fw_send(struct nxp_nci_info *info) {
   struct nxp_nci_fw_info *fw_info = &info->fw_info;
   long completion_rc; int r;

   reinit_completion(&fw_info->cmd_completion);
   if (fw_info->written == 0) {
      fw_info->frame_size = get_unaligned_be16(fw_info->data) & NXP_NCI_FW_FRAME_LEN_MASK;
      fw_info->data += NXP_NCI_FW_HDR_LEN;
      fw_info->size -= NXP_NCI_FW_HDR_LEN;
   }
   r = nxp_nci_fw_send_chunk(info); if (r < 0) {...}

   fw_info->written += r;
   if (*fw_info->data == NXP_NCI_FW_CMD_RESET) {
      fw_info->cmd_result = 0;
      if (fw_info->fw)
         schedule_work(&fw_info->work);
   } else {
      completion_rc = wait_for_completion_interruptible_timeout(
         &fw_info->cmd_completion, NXP_NCI_FW_ANSWER_TIMEOUT);
      if (completion_rc == 0) return -ETIMEDOUT;
   }
   return 0;
}

void nxp_nci_fw_work(struct work_struct *work) {
   struct nxp_nci_info *info;
   struct nxp_nci_fw_info *fw_info; int r;

   fw_info = container_of(work, struct nxp_nci_fw_info, work);
   info = container_of(fw_info, struct nxp_nci_info, fw_info);

   mutex_lock(&info->info_lock);
   r = fw_info->cmd_result; if (r < 0) {...}
   if (fw_info->written == fw_info->frame_size) {
      fw_info->data += fw_info->frame_size;
      fw_info->size -= fw_info->frame_size;
      fw_info->written = 0;
   }
   if (fw_info->size > 0) r = nxp_nci_fw_send(info);

exit_work:
   mutex_unlock(&info->info_lock);
}

void nxp_nci_fw_recv_frame(struct nci_dev *ndev, struct sk_buff *skb) {
   struct nxp_nci_info *info = nci_get_drvdata(ndev);
   struct nxp_nci_fw_info *fw_info = &info->fw_info;

   complete(&fw_info->cmd_completion);
   fw_info->cmd_result = nxp_nci_fw_read_status(*skb_pull(skb, NXP_NCI_FW_HDR_LEN));
   kfree_skb(skb);
   schedule_work(&fw_info->work);
}
